package com.xforceplus.elephant.image.controller.common.process;

import com.alibaba.fastjson.JSONObject;
import com.xforceplus.elephant.basecommon.exception.ElephantException;
import com.xforceplus.elephant.basecommon.help.HttpUtils;
import com.xforceplus.elephant.basecommon.help.RandomUtil;
import com.xforceplus.elephant.basecommon.process.AbstractProcess;
import com.xforceplus.elephant.basecommon.process.response.CommonResponse;
import com.xforceplus.elephant.basecommon.vaildate.ValidatorUtil;
import com.xforceplus.elephant.image.client.model.FileDTO;
import com.xforceplus.elephant.image.client.model.MobileUploadHandleRequest;
import com.xforceplus.elephant.image.core.business.enums.MobileUploadEnum;
import com.xforceplus.elephant.image.core.business.infrastructure.domain.WechatInfoResponse;
import com.xforceplus.elephant.image.core.business.infrastructure.port.FileService;
import com.xforceplus.elephant.image.core.business.infrastructure.port.WechatInfoService;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Base64;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.validation.ValidationException;
import net.coobird.thumbnailator.Thumbnails;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.util.StreamUtils;

/**
 * 移动端上传处理
 *
 * @author rongying
 * @date 2021/5/17 18:05
 */
@Service
public class MobileUploadHandleProcess extends AbstractProcess<MobileUploadHandleRequest, List<JSONObject>> {

    @Value("${third.wechatUrl:https://api.weixin.qq.com}")
    private String wechatUrl;

    @Autowired
    private WechatInfoService wechatInfoService;
    @Autowired
    private FileService fileService;

    @Override
    protected void check(MobileUploadHandleRequest request) throws ValidationException {
        checkEmpty(request.getUploadType(), "上传类型不能为空!");
        checkEmpty(request.getImageList(), "上传影像不能为空!");
    }

    @Override
    protected CommonResponse<List<JSONObject>> process(MobileUploadHandleRequest request) throws RuntimeException {
        final IAuthorizedUser user = UserInfoHolder.get();
        List<JSONObject> fileList = new ArrayList<>();
        final MobileUploadEnum uploadEnum = MobileUploadEnum.fromCode(request.getUploadType());
        if (null == uploadEnum) {
            return CommonResponse.failed("暂不支持此上传方式");
        }
        switch (uploadEnum) {
            case BASE64:
                fileList = base64ForUrl(user, request.getImageList());
                break;
            case WECHAT:
                fileList = wechatForUrl(user, request.getImageList());
                break;
            default:
                return CommonResponse.failed("暂不支持此上传方式");
        }
        return CommonResponse.ok("成功", fileList);
    }

    /**
     * base64上传转url
     *
     * @param base64List base64List
     * @return 结果
     * @author rongying
     * @date 2021/5/18 18:31
     */
    private List<JSONObject> base64ForUrl(IAuthorizedUser user, List<FileDTO> base64List) {
        final List<JSONObject> fileList = new ArrayList<>();
        final Pattern pattern = Pattern.compile("data:(image|application)/(\\w+);base64,");
        final String[] fileSuffix = {"jpg"};
        base64List.stream().forEach(file -> {
            final Matcher matcher = pattern.matcher(file.getFileData());
            if (matcher.find()) {
                fileSuffix[0] = matcher.group(2);
            }
            final String imageContent = file.getFileData().split(",")[1];
            final byte[] imgBytes = Base64.getDecoder().decode(imageContent);
            fileList.add(upload(user, imgBytes, file.getFileName(), fileSuffix[0]));
        });
        return fileList;
    }

    /**
     * 微信上传
     *
     * @param mediaIds mediaIds
     * @return 结果
     * @author rongying
     * @date 2021/5/18 18:25
     */
    private List<JSONObject> wechatForUrl(IAuthorizedUser user, List<FileDTO> mediaIds) {
        final List<JSONObject> fileList = new ArrayList<>();
        final WechatInfoResponse wechatInfoResponse = wechatInfoService.wechatInfo();
        if (!wechatInfoResponse.isSuccess()) {
            return new ArrayList<>();
        }
        final String accessToken = wechatInfoResponse.getResult().getAccessToken();
        mediaIds.forEach(mediaId -> {
            final String remoteUrl = StringUtils.join(wechatUrl, "/cgi-bin/media/get?access_token=", accessToken, "&media_id=", mediaId.getFileData());
            try {
                final InputStream input = HttpUtils.doGetInputStream(remoteUrl);
                final byte[] bytes = StreamUtils.copyToByteArray(input);
                fileList.add(upload(user, bytes, mediaId.getFileName(), "jpg"));
            } catch (Exception e) {
                throw new ElephantException("获取微信图片异常：" + e.getMessage());
            }
        });
        return fileList;
    }

    private JSONObject upload(IAuthorizedUser user, byte[] imgBytes, String fileName, String fileSuffix) {
        final String batchNo = System.currentTimeMillis() + String.valueOf(RandomUtil.randomInt(99999));
        final String uploadFileName = StringUtils.join(batchNo, ".", fileSuffix);
        //上传
        final ByteArrayOutputStream baos = new ByteArrayOutputStream();
        try {
            final Thumbnails.Builder builder = Thumbnails.of(new ByteArrayInputStream(imgBytes)).scale(1).outputQuality(0.5);
            builder.toOutputStream(baos);
        } catch (IOException e) {
            throw new ElephantException("图片处理异常：" + e.getMessage());
        }
        final Long fileKey = fileService.uploadFile(user.getTenantId(), user.getId(), uploadFileName, new ByteArrayInputStream(baos.toByteArray()));
        if (ValidatorUtil.isEmpty(fileKey)) {
            throw new ElephantException("上传至文件服务失败!");
        }
        final String url = fileService.getFileUrl(user.getTenantId(), user.getId(), fileKey);
        if (ValidatorUtil.isEmpty(url)) {
            throw new ElephantException("上传至文件服务失败!");
        }
        final JSONObject jsonObject = new JSONObject();
        jsonObject.put("fileUrl", url);
        jsonObject.put("fileType", StringUtils.join(".", fileSuffix));
        jsonObject.put("fileName", fileName);
        return jsonObject;
    }

}
