package com.xforceplus.elephant.image.controller.compare.ticket.process.deleteperson;

import com.google.common.collect.Maps;
import com.xforceplus.elephant.basecommon.annotation.Dispatch;
import com.xforceplus.elephant.basecommon.process.AbstractProcess;
import com.xforceplus.elephant.basecommon.process.response.CommonResponse;
import com.xforceplus.elephant.basecommon.vaildate.ValidatorUtil;
import com.xforceplus.elephant.image.client.model.DeleteTicketListRequest;
import com.xforceplus.elephant.image.core.domain.compare.image.CompareImageService;
import com.xforceplus.elephant.image.core.domain.compare.ticket.CompareTicketService;
import com.xforceplus.elephant.image.core.expand.compare.CompareBillImageTicketService;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.dict.BackStatus;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.dict.CommitStatus;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.dict.ExtStatus;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.entity.Image;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.entity.Ticket;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.meta.EntityMeta;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import javax.validation.ValidationException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

@Dispatch(isDefault = true)
@Service
public class DefaultCompareDeletePersonTicketListProcess extends AbstractProcess<DeleteTicketListRequest, Boolean> {

    @Autowired
    private CompareBillImageTicketService billImageTicketService;
    @Autowired
    private CompareTicketService ticketService;
    @Autowired
    private CompareImageService imageService;

    @Override
    protected void check(DeleteTicketListRequest request) throws ValidationException {
        super.check(request);
        if (ValidatorUtil.isEmpty(request.getTicketIds())) {
            throw new ValidationException("单证ID集合【ticketIds】不能为空");
        }
        if (ValidatorUtil.isEmpty(request.getTicketCode())) {
            throw new ValidationException("单证code【ticketCode】不能为空");
        }
    }

    @Override
    protected CommonResponse<Boolean> process(DeleteTicketListRequest request) throws RuntimeException {
        final IAuthorizedUser authorizedUser = UserInfoHolder.get();
        final List<Ticket> entities = ticketService.selectTicketList(request.getTicketIds(), request.getTicketCode());
        if (ValidatorUtil.isEmpty(entities)) {
            return CommonResponse.failed("未查询到数据");
        }
        //规则过滤出可删除的发票
        final Map<String, List<Long>> resultMap = billImageTicketService.deleteTicketData(
            authorizedUser.getTenantId(), request.getTicketIds(), request.getTicketCode());
        final List<Long> imageIds = resultMap.get("imageIds");
        final List<Long> ticketIds = resultMap.get("ticketIds");
        if (ValidatorUtil.isEmpty(ticketIds)) {
            return CommonResponse.failed("没有找到可操作的数据");
        }
        //查询发票下挂接的附件，然后一并删除
        if (ValidatorUtil.isNotEmpty(imageIds)) {
            final List<Long> attachmentAndSalesImageId = imageService.selectTicketAttachmentAndSalesList(authorizedUser.getTenantId(), imageIds).stream().map(Image::getId)
                .collect(Collectors.toList());
            imageIds.addAll(attachmentAndSalesImageId);
        }
        if (ValidatorUtil.isEmpty(imageIds)) {
            return CommonResponse.failed("没有找到可操作的数据");
        }

        //删除
        final boolean result = billImageTicketService.deleteImageAndTicketByImageIds(authorizedUser.getTenantId(), imageIds);
        if (result) {
            String message = "删除成功";
            if (request.getTicketIds().size() != ticketIds.size()) {
                message = String.format("%d条记录成功删除%d条", request.getTicketIds().size(), ticketIds.size());
            }
            return CommonResponse.ok(message, true);
        } else {
            return CommonResponse.failed("无记录删除");
        }
    }

    /**
     * 个人单证池删除，是否下发
     *
     * @param tenantId   租户ID
     * @param imageIds   影像IDs
     * @param deleteFlag true or false 下发
     * @return void
     * @author rongying
     * @date 2021/8/18 14:40
     */
    private void personDeleteSend(Long tenantId, List<Long> imageIds, boolean deleteFlag) {
        if (!deleteFlag) {
            return;
        }
        //查询单证信息
        final List<Ticket> deleteList = ticketService.selectBaseTicketByImageIds(tenantId, imageIds);
        if (ValidatorUtil.isEmpty(deleteList)) {
            return;
        }
        final Map<String, Object> update = Maps.newHashMap();
        update.put(EntityMeta.Ticket.TICKET_STATUS.code(), ExtStatus._99.getCode());
        deleteList.forEach(item -> {
            //已提交||退回 删除需要下发
            if ((ValidatorUtil.isNotEmpty(item.getCommitStatus()) && !CommitStatus._0.getCode().equals(item.getCommitStatus()))
                || (ValidatorUtil.isNotEmpty(item.getBackStatus()) && BackStatus._1.getCode().equals(item.getBackStatus()))) {
                //先更新删除标记
                final Integer count = ticketService.updateTicketByParam(item.getId(), item.getTicketCode(), update);
                if (count > 0) { //在下发
                    billImageTicketService.syncTicket(tenantId, item.getImageId());
                }
            }
        });
    }

}