package com.xforceplus.elephant.image.controller.devops.process;

import com.xforceplus.elephant.basecommon.enums.bill.OperateEnum;
import com.xforceplus.elephant.basecommon.process.AbstractProcess;
import com.xforceplus.elephant.basecommon.process.response.CommonResponse;
import com.xforceplus.elephant.image.client.model.TicketRelationHandleRequest;
import com.xforceplus.elephant.image.core.business.application.calculate.check.custom.ticket.CustomTicketAttachmentCheck;
import com.xforceplus.elephant.image.core.business.application.calculate.check.engine.registry.ticket.TicketConsecutiveCheck;
import com.xforceplus.elephant.image.core.business.application.calculate.check.engine.registry.ticket.TicketSalesListCheck;
import com.xforceplus.elephant.image.core.business.application.calculate.relation.RelationConfig;
import com.xforceplus.elephant.image.core.business.application.calculate.relation.RelationContext;
import com.xforceplus.elephant.image.core.business.application.calculate.relation.RelationContext.Operator;
import com.xforceplus.elephant.image.core.business.application.calculate.relation.ticket.TicketRelationConfig;
import com.xforceplus.elephant.image.core.business.application.calculate.relation.ticket.TicketRelationHandle;
import com.xforceplus.elephant.image.core.business.application.collect.ticket.service.TicketService;
import com.xforceplus.elephant.image.core.business.application.config.common.service.ConfigService;
import com.xforceplus.elephant.image.core.business.enums.DictEnum;
import com.xforceplus.elephant.image.core.business.util.MultiOQSUtil;
import com.xforceplus.elephant.image.core.expand.BillImageTicketService;
import com.xforceplus.elephant.image.core.expand.compare.CompareBillImageTicketService;
import com.xforceplus.elephant.image.core.util.RequestBuilder;
import com.xforceplus.tech.base.core.context.ContextKeys.StringKeys;
import com.xforceplus.tech.base.core.context.ContextService;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.dict.TicketRelType;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.dict.YesNo;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.meta.EntityMeta;
import com.xforceplus.ultraman.metadata.domain.vo.dto.ConditionOp;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import lombok.RequiredArgsConstructor;
import lombok.Setter;
import lombok.experimental.Accessors;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.ListUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

@Slf4j
@RequiredArgsConstructor
@Service
public class TicketRelationHandleProcess extends AbstractProcess<TicketRelationHandleRequest, Boolean> {

    private final BillImageTicketService billImageTicketService;
    private final CompareBillImageTicketService compareBillImageTicketService;
    private final TicketService ticketService;
    private final ConfigService configService;
    private final ExecutorService imageThreadPool;
    private final ContextService contextService;
    private final MultiOQSUtil multiOQSUtil;

    @Setter
    @Accessors(chain = true)
    private class HandleContext {

        private Long tenantId;
        private String tenantCode;
        private RelationConfig relationConfig;
        private boolean isNewRelationHandle;
        private boolean isCompare;
        private int pageSize;

        private AtomicLong handleCount = new AtomicLong(0);
        private AtomicInteger handleSuccessCount = new AtomicInteger(0);

    }

    @Override
    protected CommonResponse<Boolean> process(final TicketRelationHandleRequest request) throws RuntimeException {
        final RelationConfig relationConfig = Optional.ofNullable(configService.selectDictObj(request.getTenantId(), DictEnum.TICKET_RELATION_CONFIG.getCode(), TicketRelationConfig.class))
            .orElse(new TicketRelationConfig());
        final String tenantCode = multiOQSUtil.getTenantCodeByTenantId(request.getTenantId());
        final HandleContext context = new HandleContext()
            .setTenantId(request.getTenantId())
            .setTenantCode(tenantCode)
            .setRelationConfig(relationConfig)
            .setNewRelationHandle(request.isNewRelationHandle())
            .setCompare(request.isCompare())
            .setPageSize(request.getPageSize());
        if (CollectionUtils.isNotEmpty(request.getImageIds())) {
            context.handleCount.set(request.getImageIds().size());
            ListUtils.partition(request.getImageIds(), request.getPageSize()).forEach(imageIds -> handleTicketsByImageIds(context, imageIds));
            return CommonResponse.ok(String.format("处理成功，处理数量：%s", request.getImageIds().size()));
        }
        if (CollectionUtils.isEmpty(request.getTypes())) {
            return CommonResponse.failed("没有可处理数据");
        }
        try {
            contextService.set(StringKeys.TENANTCODE_KEY, tenantCode);
            request.getTypes().forEach(relType -> {
                final TicketRelType ticketRelType = TicketRelType.fromCode(relType);
                if (relType == null) {
                    return;
                }
                switch (ticketRelType) {
                    case REPEAT:
                        final RequestBuilder repeatBuilder = new RequestBuilder()
                            .field(EntityMeta.Ticket.IS_REPEAT.code(), ConditionOp.eq, YesNo._1.getCode());
                        selectAndHandleTickets(context, repeatBuilder);
                        break;
                    case SHEET:
                        final RequestBuilder sheetBuilder = new RequestBuilder()
                            .field(EntityMeta.Ticket.IS_EXIST_SHEET.code(), ConditionOp.eq, YesNo._1.getCode());
                        selectAndHandleTickets(context, sheetBuilder);
                        break;
                    case SALES:
                        final RequestBuilder salesBuilder = new RequestBuilder()
                            .field(EntityMeta.Ticket.EXCEPTION_KEY.code(), ConditionOp.in, TicketSalesListCheck.CHECK_CODE);
                        selectAndHandleTickets(context, salesBuilder);
                        salesBuilder.removeField(EntityMeta.Ticket.EXCEPTION_KEY.code());
                        salesBuilder.field(EntityMeta.Ticket.WARNING_KEY.code(), ConditionOp.in, TicketSalesListCheck.CHECK_CODE);
                        selectAndHandleTickets(context, salesBuilder);
                        break;
                    case SERIES:
                        final RequestBuilder seriesBuilder = new RequestBuilder()
                            .field(EntityMeta.Ticket.EXCEPTION_KEY.code(), ConditionOp.in, TicketConsecutiveCheck.CHECK_CODE);
                        selectAndHandleTickets(context, seriesBuilder);
                        seriesBuilder.removeField(EntityMeta.Ticket.EXCEPTION_KEY.code());
                        seriesBuilder.field(EntityMeta.Ticket.WARNING_KEY.code(), ConditionOp.in, TicketConsecutiveCheck.CHECK_CODE);
                        selectAndHandleTickets(context, seriesBuilder);
                        break;
                    case ATTACH:
                        final RequestBuilder attachBuilder = new RequestBuilder()
                            .field(EntityMeta.Ticket.EXCEPTION_KEY.code(), ConditionOp.in, CustomTicketAttachmentCheck.CHECK_CODE);
                        selectAndHandleTickets(context, attachBuilder);
                        attachBuilder.removeField(EntityMeta.Ticket.EXCEPTION_KEY.code());
                        attachBuilder.field(EntityMeta.Ticket.WARNING_KEY.code(), ConditionOp.in, CustomTicketAttachmentCheck.CHECK_CODE);
                        selectAndHandleTickets(context, attachBuilder);
                        break;
                    default:
                        log.warn("未知的处理类型：{}", relType);
                        break;
                }
            });
        } catch (Throwable throwable) {
            log.error("处理异常", throwable);
            return CommonResponse.failed("处理异常:" + throwable.getMessage());
        } finally {
            contextService.clear();
        }
        return CommonResponse.ok("处理成功");
    }

    private void handleTicketsByImageIds(final HandleContext context, final List<Long> imageIds) {
        if (context.isCompare) {
            if (context.isNewRelationHandle) {
                final List<Map<String, Object>> tickets = ticketService.selectTicketByImageIds(context.tenantId, imageIds, EntityMeta.CompareTicket.code());
                newRelationHandle(context, tickets);
            } else {
                compareBillImageTicketService.checkTicketRepeatAndReuse(context.tenantId, imageIds, OperateEnum.HOOK, null);
            }
        } else {
            if (context.isNewRelationHandle) {
                final List<Map<String, Object>> tickets = ticketService.selectTicketByImageIds(context.tenantId, imageIds, EntityMeta.Ticket.code());
                newRelationHandle(context, tickets);
            } else {
                billImageTicketService.checkTicketRepeatAndReuse(context.tenantId, imageIds, OperateEnum.HOOK, null);
            }
        }
    }

    private void newRelationHandle(final HandleContext context, List<Map<String, Object>> tickets) {

        tickets.forEach(ticket -> {
            final long start = System.currentTimeMillis();
            final RelationContext relationContext = new RelationContext()
                .setTenantId(context.tenantId)
                .setConfig(context.relationConfig)
                .setOperator(Operator.CREATE)
                .setNewValue(ticket);
            TicketRelationHandle.RELATIONS.stream()
                .parallel()
                .filter(relationHandle -> relationHandle.enable(relationContext))
                .forEach(relationHandle -> {
                    try {
                        contextService.set(StringKeys.TENANTID_KEY, String.valueOf(context.tenantId));
                        contextService.set(StringKeys.TENANTCODE_KEY, context.tenantCode);
                        relationHandle.handle(relationContext);
                    } catch (Exception e) {
                        log.error("处理异常", e);
                    } finally {
                        contextService.clear();
                    }
                });
            context.handleSuccessCount.incrementAndGet();
            log.info("{}/{}, 处理票据关联耗时：{}ms", context.handleSuccessCount.get(), context.handleCount.get(), System.currentTimeMillis() - start);
        });
    }

    private void selectAndHandleTickets(final HandleContext context, final RequestBuilder requestBuilder) {
        Stream.of(EntityMeta.Ticket.code(), EntityMeta.CompareTicket.code()).forEach(ticketBoCode -> {
            Long id = 0L;
            List<Map<String, Object>> tickets;
            requestBuilder.pageSize(context.pageSize);
            do {
                requestBuilder.removeField(EntityMeta.Ticket.ID.code());
                requestBuilder.field(EntityMeta.Ticket.ID.code(), ConditionOp.gt, id);
                tickets = ticketService.selectTicketMapByParams(context.tenantId, requestBuilder, ticketBoCode);
                if (CollectionUtils.isEmpty(tickets)) {
                    break;
                }
                context.handleCount.addAndGet(tickets.size());
                final List<Map<String, Object>> finalTickets = tickets;
                CompletableFuture.runAsync(() -> {
                    try {
                        contextService.set(StringKeys.TENANTID_KEY, String.valueOf(context.tenantId));
                        contextService.set(StringKeys.TENANTCODE_KEY, context.tenantCode);
                        this.handleTickets(context, finalTickets, StringUtils.startsWith(ticketBoCode, "compare"));
                    } catch (Exception e) {
                        log.error("处理异常", e);
                    } finally {
                        contextService.clear();
                    }
                }, imageThreadPool);
                id = MapUtils.getLong(tickets.get(tickets.size() - 1), EntityMeta.Ticket.ID.code());
            } while (CollectionUtils.size(tickets) == context.pageSize);
        });

    }

    private void handleTickets(final HandleContext context, final List<Map<String, Object>> tickets, final boolean isCompare) {
        final List<Long> imageIds = tickets.stream().map(ticket -> MapUtils.getLong(ticket, EntityMeta.Ticket.IMAGE_ID.code())).collect(Collectors.toList());
        if (isCompare) {
            if (context.isNewRelationHandle) {
                newRelationHandle(context, tickets);
            } else {
                compareBillImageTicketService.checkTicketRepeatAndReuse(context.tenantId, imageIds, OperateEnum.HOOK, null);
            }
        } else {
            if (context.isNewRelationHandle) {
                newRelationHandle(context, tickets);
            } else {
                billImageTicketService.checkTicketRepeatAndReuse(context.tenantId, imageIds, OperateEnum.HOOK, null);
            }
        }
    }

}
