package com.xforceplus.elephant.image.controller.image.process;

import com.alibaba.fastjson.JSON;
import com.xforceplus.domain.user.UserDto;
import com.xforceplus.elephant.basecommon.exception.ElephantException;
import com.xforceplus.elephant.basecommon.help.DateHelp;
import com.xforceplus.elephant.basecommon.help.StringHelp;
import com.xforceplus.elephant.basecommon.process.AbstractProcess;
import com.xforceplus.elephant.basecommon.process.response.CommonResponse;
import com.xforceplus.elephant.basecommon.system.paas.NotificationUtils;
import com.xforceplus.elephant.basecommon.vaildate.ValidatorUtil;
import com.xforceplus.elephant.image.client.model.DownloadImageRequest;
import com.xforceplus.elephant.image.core.business.infrastructure.port.FileService;
import com.xforceplus.elephant.image.core.domain.image.ImageService;
import com.xforceplus.elephant.image.core.facade.application.push.download.DownloadFacade;
import com.xforceplus.elephant.image.core.util.ValidateUtils;
import com.xforceplus.tech.base.core.context.ContextKeys;
import com.xforceplus.tech.base.core.context.ContextService;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import javax.validation.ValidationException;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

/**
 * 下载影像
 *
 * @author liuhongbin
 * @date 2021/12/7 16:41
 */
@Slf4j
@RequiredArgsConstructor
@Service
public class DownloadImagesProcess extends AbstractProcess<DownloadImageRequest, Object> {

    private final ImageService imageService;
    private final FileService fileService;
    private final DownloadFacade downloadFacade;
    private final NotificationUtils notificationUtils;
    private final ExecutorService imageThreadPool;
    private final ContextService contextService;

    @Value("${image.maxDownloadCount:1000}")
    private Integer maxDownloadCount;
    private static final String ZIP_SUFFIX = ".zip";

    @Override
    protected void check(DownloadImageRequest request) throws ValidationException {
        if (ValidatorUtil.isEmpty(request.getImageIds()) && ValidatorUtil.isEmpty(request.getBillCodes()) && ValidatorUtil.isEmpty(request.getBillCode())) {
            throw new ValidationException("请选择要下载的影像");
        }
    }

    @Override
    protected CommonResponse<Object> process(DownloadImageRequest request) throws RuntimeException {
        final IAuthorizedUser user = UserInfoHolder.get();
        ValidateUtils.checkNotNull(user, "获取用户信息为空，请重新登陆");

        Integer count = 0;
        if (ValidatorUtil.isNotEmpty(request.getImageIds())) {
            count = request.getImageIds().size();
        } else if (ValidatorUtil.isNotEmpty(request.getBillCodes())) {
            count = imageService.countImage(user.getTenantId(), request.getBillCodes());
        } else if (ValidatorUtil.isNotEmpty(request.getBillCode())) {
            count = imageService.countImage(user.getTenantId(), Arrays.asList(request.getBillCode()));
        }
        if (count > maxDownloadCount) {
            final String msg = String.format("当前勾选影像%s张已超下载上限，不可操作（注下载上限为%s张）", count, maxDownloadCount);
            throw new ElephantException(msg);
        }
        final UserDto userDto = JSON.parseObject(JSON.toJSONString(user), UserDto.class);
        CompletableFuture.runAsync(() -> {
            contextService.set(ContextKeys.StringKeys.TENANTID_KEY, StringHelp.safeToString(userDto.getTenantId()));
            contextService.set(ContextKeys.StringKeys.TENANTCODE_KEY, userDto.getTenantCode());
            contextService.set(ContextKeys.LongKeys.ID, userDto.getId());
            contextService.set(ContextKeys.StringKeys.USER_DISPLAYNAME, userDto.getUserName());
            try {
                final String fileName = "影像下载" + DateHelp.getTimeShortStr(new Date()) + ZIP_SUFFIX;
                final ByteArrayOutputStream baos = new ByteArrayOutputStream();
                if (ValidatorUtil.isNotEmpty(request.getImageIds())) {
                    downloadFacade.downloadZIPImagesByImageId(userDto, request.getImageIds(), baos);
                } else if (ValidatorUtil.isNotEmpty(request.getBillCodes())) {
                    downloadFacade.downloadZIPImagesByBillCode(userDto, request.getBillCodes(), baos, false);
                } else if (ValidatorUtil.isNotEmpty(request.getBillCode())) {
                    downloadFacade.downloadZIPImagesByBillCode(userDto, Arrays.asList(request.getBillCode()), baos, false);
                }
                final Long fileKey = fileService.uploadFile(userDto.getTenantId(), userDto.getId(), fileName, new ByteArrayInputStream(baos.toByteArray()), 7);
                final String fileUrl = fileService.getFileUrl(userDto.getTenantId(), userDto.getId(), fileKey);
                final String contentTemplate = "<a href=\"%s\">%s</a>";
                final String content = String.format(contentTemplate, fileUrl, fileName);
                notificationUtils.sendSingleMessage(userDto.getTenantId(), Collections.singletonList(userDto.getId()), "影像下载", content);
            } catch (Exception e) {
                log.error("下载异常", e);
            } finally {
                contextService.clear();
            }
        }, imageThreadPool);
        return CommonResponse.ok("下载中，请到消息中心查看最终的下载文件");
    }

}
