package com.xforceplus.elephant.image.controller.image.process;

import com.xforceplus.elephant.basecommon.process.AbstractProcess;
import com.xforceplus.elephant.basecommon.process.response.CommonResponse;
import com.xforceplus.elephant.basecommon.vaildate.ValidatorUtil;
import com.xforceplus.elephant.image.client.model.BillSelectAllRequest;
import com.xforceplus.elephant.image.core.business.application.collect.task.enums.TaskTypeEnum;
import com.xforceplus.elephant.image.core.business.application.collect.task.service.TaskQueueService;
import com.xforceplus.elephant.image.core.business.domain.ProcessContext;
import com.xforceplus.elephant.image.core.domain.image.ImageService;
import com.xforceplus.elephant.image.core.domain.image.bean.ImageSearchBean;
import com.xforceplus.elephant.image.core.domain.imagefile.ImageFileService;
import com.xforceplus.elephant.image.core.expand.BillImageTicketService;
import com.xforceplus.elephant.image.core.facade.application.collect.image.ImageFacade;
import com.xforceplus.elephant.image.mapper.ImageSearchMapper;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.dict.ImageType;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.entity.Image;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.entity.ImageFile;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.meta.EntityMeta;
import java.util.List;
import java.util.stream.Collectors;
import javax.validation.ValidationException;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

/**
 * @author yuhongxia
 * @description 重新转换
 * @date 2021/7/7 9:52
 */
@Service
public class ReTransfromImageProcess extends AbstractProcess<BillSelectAllRequest, Boolean> {

    @Autowired
    private ImageFacade imageFacade;
    @Autowired
    private ImageSearchMapper mapper;
    @Autowired
    private ImageService imageService;
    @Autowired
    private TaskQueueService taskQueueService;
    @Autowired
    private ImageFileService imageFileService;
    @Autowired
    private BillImageTicketService billImageTicketService;

    @Override
    protected void check(BillSelectAllRequest request) throws ValidationException {
        super.check(request);
        if (request.isAllSelected()) {
            checkEmpty(request.getBillCode(), "单据code不能为空");
        }
        if (!request.isAllSelected()) {
            checkEmpty(request.getIncluded(), "影像id不能为空");
        }
    }

    /**
     * 准备要操作的数据 格式化全选反选数据到ids 需要注意此处是要考虑全选操作是带着筛选条件的
     *
     * @param request 请求
     * @return void
     * @author liming
     * @date 2021/7/30 11:18
     */
    private void prepareOperateData(BillSelectAllRequest request) {
        //全选
        final IAuthorizedUser user = UserInfoHolder.get();
        if (request.isAllSelected()) {
            //条件筛选
            final ImageSearchBean searchBean = mapper.map(request);
            searchBean.setTenantId(user.getTenantId());
            final List<Image> imageList = imageFacade.selectBillImagesByCustomTicketParams(searchBean, EntityMeta.Image.code()).getRows();
            final List<Long> ids = imageList.stream()
                                            .filter(image -> !ImageType._1.getCode().equals(image.getImageType()))
                                            .map(image -> image.getId())
                                            .collect(Collectors.toList());
            if (ValidatorUtil.isNotEmpty(request.getExcluded())) {
                final List<Long> excludedList = request.getExcluded().stream().map(s -> Long.parseLong(s)).collect(Collectors.toList());
                ids.removeAll(excludedList);
            }
            request.setIds(ids);
        }
        //非全选
        if (!request.isAllSelected()) {
            final List<Long> ids = request.getIncluded().stream().map(s -> Long.parseLong(s)).collect(Collectors.toList());
            request.setIds(ids);
        }
    }

    @Override
    protected CommonResponse<Boolean> process(BillSelectAllRequest request) throws RuntimeException {
        final IAuthorizedUser authorizedUser = UserInfoHolder.get();
        if (null == authorizedUser) {
            throw new ValidationException("获取用户信息为空，请重新登陆");
        }

        prepareOperateData(request);

        if (CollectionUtils.isEmpty(request.getIds())) {
            throw new ValidationException("没有符合条件的数据，请重新选择");
        }

        final StringBuilder stringBuilder = new StringBuilder();
        List<Image> imageList = imageService.selectImages(authorizedUser.getTenantId(), request.getIds());
        //查询到转换失败的Image
        final List<Long> fileIds = imageList.stream().filter(image -> ValidatorUtil.isNotEmpty(image.getFileId())).map(Image::getFileId).collect(Collectors.toList());
        if (ValidatorUtil.isEmpty(fileIds)) {
            stringBuilder.append("当前共勾选").append(request.getIds().size()).append("条,其中").append(0).append("条发起转换");
            return CommonResponse.ok(stringBuilder.toString());
        }

        List<ImageFile> imageFileList = imageFileService.selectFailedFileByIds(fileIds);
        if (ValidatorUtil.isEmpty(imageFileList)) {
            stringBuilder.append("当前共勾选").append(request.getIds().size()).append("条,其中").append(0).append("条发起转换");
            return CommonResponse.ok(stringBuilder.toString());
        }
        imageFileList = imageFileList.stream().filter(imageFile -> ValidatorUtil.isNotEmpty(imageFile.getTaskType())).collect(Collectors.toList());
        if (ValidatorUtil.isEmpty(imageFileList)) {
            stringBuilder.append("当前共勾选").append(request.getIds().size()).append("条,其中").append(0).append("条发起转换");
            return CommonResponse.ok(stringBuilder.toString());
        }
        final List<Long> failedFileIds = imageFileList.stream().map(ImageFile::getId).collect(Collectors.toList());
        //重新发起
        imageFileList.stream().filter(imageFile -> ValidatorUtil.isNotEmpty(imageFile.getTaskType())).forEach(imageFile -> {
            imageFile.setId(null);
            //个人单证池的文件挂机后不会更新billCode
            if (ValidatorUtil.isEmpty(imageFile.getBillCode()) && ValidatorUtil.isNotEmpty(request.getBillCode())) {
                imageFile.setBillCode(request.getBillCode());
            }
            final ProcessContext context = ProcessContext.context().put(imageFile);
            taskQueueService.initiate(TaskTypeEnum.fromType(imageFile.getTaskType()), context);
        });
        //删除影像
        imageList = imageList.stream().filter(image -> failedFileIds.contains(image.getFileId())).collect(Collectors.toList());
        //删除结构数据
        final List<Long> failedImageIds = imageList.stream().map(Image::getId).collect(Collectors.toList());
        billImageTicketService.deleteImageAndTicketByImageIds(authorizedUser.getTenantId(), failedImageIds);
        imageFileService.deleteImageFileByIds(failedFileIds);

        //当前共勾选X条，其中X
        stringBuilder.append("当前共勾选").append(request.getIds().size()).append("条,其中").append(imageFileList.size()).append("条发起转换");
        return CommonResponse.ok(stringBuilder.toString());
    }

}
