package com.xforceplus.elephant.image.controller.image.process;

import com.alibaba.fastjson.JSONObject;
import com.xforceplus.elephant.basecommon.annotation.Dispatch;
import com.xforceplus.elephant.basecommon.help.FileUtils;
import com.xforceplus.elephant.basecommon.help.RandomUtil;
import com.xforceplus.elephant.basecommon.process.AbstractProcess;
import com.xforceplus.elephant.basecommon.process.response.CommonResponse;
import com.xforceplus.elephant.basecommon.vaildate.ValidatorUtil;
import com.xforceplus.elephant.image.client.model.UploadRequest;
import com.xforceplus.elephant.image.core.business.application.collect.batch.domain.EnableFlowContext;
import com.xforceplus.elephant.image.core.business.application.collect.batch.service.BatchService;
import com.xforceplus.elephant.image.core.business.application.collect.task.domain.FileTransformTypeConfig;
import com.xforceplus.elephant.image.core.business.config.queue.MQUtils;
import com.xforceplus.elephant.image.core.business.enums.FileTypeEnum;
import com.xforceplus.elephant.image.core.business.enums.MQEnum;
import com.xforceplus.elephant.image.core.business.util.LogUtil;
import com.xforceplus.elephant.image.core.domain.config.ConfigSettingsService;
import com.xforceplus.elephant.image.core.domain.imagefile.ImageFileService;
import com.xforceplus.elephant.image.core.expand.BillImageTicketService;
import com.xforceplus.elephant.image.core.facade.application.collect.upload.UploadFacade;
import com.xforceplus.elephant.image.core.facade.dto.upload.UploadImageDto;
import com.xforceplus.elephant.image.core.facade.dto.upload.UploadImageRequestDto;
import com.xforceplus.elephant.image.core.repository.model.BatchTaskEntity;
import com.xforceplus.elephant.image.core.util.ImageUtils;
import com.xforceplus.elephant.image.listener.MqQueueSizeListener;
import com.xforceplus.elephant.image.mapper.UploadMapper;
import com.xforceplus.purchaser.bizratelimiter.model.Application;
import com.xforceplus.purchaser.bizratelimiter.model.BizRequest;
import com.xforceplus.purchaser.bizratelimiter.model.Channel;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.dict.IsPublic;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.dict.TransformScene;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.dict.YesNo;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.entity.Image;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.entity.ImageFile;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.meta.EntityMeta;
import com.xforceplus.xlog.core.model.LogContext;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import javax.validation.ValidationException;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.groovy.util.Maps;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

/**
 * @author yuhongxia
 * @description 上传影像
 * @date 2021/4/7
 */
@Dispatch(isDefault = true)
@Service
public class UploadImageProcess extends AbstractProcess<UploadRequest, List<Long>> {

    @Autowired
    protected BillImageTicketService billImageTicketService;
    @Autowired
    private ImageFileService imageFileService;
    @Autowired
    private UploadFacade uploadFacade;
    @Autowired
    private UploadMapper uploadMapper;
    @Autowired
    private ConfigSettingsService configSettingsService;
    @Autowired
    private BatchService batchService;
    @Autowired
    private MQUtils mqUtils;
    @Autowired(required = false)
    private Application application;
    @Value("${image.pdfQueue.threshold:1000}")
    private int pdfQueueSize;
    @Value("${image.pdfQueue.tpm:90}")
    private int pdfQueueTpm;
    @Value("${image.pdfQueue.tips:当前pdf转换队列已积压%s条，预估等到时间%s分钟}")
    private String pdfQueueTips;

    @Override
    protected void check(UploadRequest request) throws ValidationException {
        checkEmpty(request.getSystemOrig(), "系统来源不能为空!");
    }

    @Override
    protected CommonResponse<List<Long>> process(UploadRequest request) throws RuntimeException {
        final IAuthorizedUser user = UserInfoHolder.get();
        //logger.debug("UploadImageProcess user:{}", JSONObject.toJSONString(user));
        if (ValidatorUtil.isEmpty(user)) {
            return CommonResponse.failed("用户信息获取失败!");
        }
        final String batchNo = System.currentTimeMillis() + String.valueOf(RandomUtil.randomInt(99999));
        final String[] isPublic = {IsPublic._0.getCode()};
        if (ValidatorUtil.isNotEmpty(request.getIsPublic())) {
            isPublic[0] = request.getIsPublic();
        }
        //1-获取文件转换字典配置
        final FileTransformTypeConfig fileTransformTypeConfig = imageFileService.getFileTransformTypeConfig(user.getTenantId());
        //2-上传
        final List<Long> imageIds = new ArrayList<>();
        final AtomicInteger index = new AtomicInteger();
        //2-0 存在zip，解析完在上传
        final UploadImageRequestDto requestDto = uploadMapper.map(request);
        uploadFacade.initSystemField(requestDto, user);
        final List<UploadImageDto> resultList = uploadFacade.checkZipFile(requestDto.getUploadImageDtoList(), user.getTenantId());
        final List<String> failedList = new ArrayList<>();//上传失败文件
        //是否开启限流
        final EnableFlowContext context = new EnableFlowContext().setTenantId(user.getTenantId()).setUserId(user.getId());
        final boolean flag = batchService.enable(context);
        final AtomicBoolean resetLevel = new AtomicBoolean(false);
        final List<JSONObject> requestDtos = new ArrayList<>();
        resultList.forEach(imageDto -> {
            final String newFileOrder = ImageUtils.sequence(index.incrementAndGet(), resultList.size());
            //2-1初始化影像文件
            final ImageFile imageFile = uploadFacade.initImageFile(requestDto);
            // 进口类发票 -- 针对不需要识别场景， 设置为只上传不识别
            final TransformScene transformScene = YesNo._0.getCode().equals(imageDto.getRequireOcrFlag()) ? TransformScene.TRANS_NOT_REC : TransformScene.TRANS_REC;
            imageFile.setTransformScene(transformScene.getCode());
            //2-2获取影像上传地址,存储 fileKey,fileName
            final String fileUrl = uploadFacade.getUploadImageUrl(user.getTenantId(), user.getId(), imageDto.getFileStream(), imageDto.getFileType(), imageFile);
            if (ValidatorUtil.isEmpty(fileUrl)) {
                failedList.add(imageDto.getFileName() + imageDto.getFileType());
                return;
            }
            final String fileSuffix = ValidatorUtil.isNotEmpty(FileUtils.getFileSuffix(fileUrl)) ? FileUtils.getFileSuffix(fileUrl) : imageDto.getFileType();
            imageFile.setFileSuffix(fileSuffix);
            imageFile.setFileUrl(fileUrl);
            imageFile.setAttachmentName(imageDto.getFileName());
            imageFile.setFileOrder(newFileOrder);
            imageFile.setIsPublic(isPublic[0]);
            imageFile.setBatchNo(batchNo);

            //2-3初始化影像
            final Image image = uploadFacade.initImage(requestDto);
            image.setFileUrl(fileUrl);
            image.setFileUrlHandle(fileUrl);
            image.setFileType(fileSuffix);
            image.setFileTypeHandle(fileSuffix);
            image.setFileOrder(newFileOrder);
            image.setIsPublic(isPublic[0]);
            image.setBatchNo(batchNo);
            image.setFileName(imageDto.getFileName());
            image.setOrgId(request.getOrgId());
            image.setOrgCode(request.getOrgCode());
            image.setOrgName(request.getOrgName());
            // 进口类发票需求 --如果不需要识别，这里需要知道是否指定类型，不指定默认是附件。
            image.setTicketCode(Optional.ofNullable(imageDto.getTicketCode()).orElseGet(EntityMeta.TicketAttachment::code));
            //2-4上传
            if (YesNo._0.getCode().equals(imageFile.getRequireOcrFlag())) {
                resetLevel.set(true);
            }
            if (!flag) {
                final Long imageId = uploadFacade.executeUpload(null, fileSuffix, image, imageFile, fileTransformTypeConfig);
                if (ValidatorUtil.isNotEmpty(imageId)) {
                    imageIds.add(imageId);
                }
                return;
            }
            final JSONObject dto = new JSONObject();
            dto.put("image", image);
            dto.put("imageFile", imageFile);
            requestDtos.add(dto);
        });
        LogUtil.attachIsEnableFlow(flag);
        if (flag && CollectionUtils.isNotEmpty(requestDtos)) {
            final String channelCode = batchService.getChannelCode(user.getTenantCode());
            final BatchTaskEntity batchTaskEntity = new BatchTaskEntity();
            batchTaskEntity.setTenantId(user.getTenantId());
            batchTaskEntity.setTenantCode(user.getTenantCode());
            batchTaskEntity.setCreateUserId(user.getId());
            batchTaskEntity.setCreateUserName(user.getUsername());
            batchTaskEntity.setCreateTime(new Date());
            batchTaskEntity.setTaskCount(resultList.size());
            batchTaskEntity.setChannelCode(channelCode);
            final Long batchId = batchService.insert(batchTaskEntity);
            requestDtos.forEach(dto -> {
                ((Image) dto.get("image")).setBatchNo(String.valueOf(batchId));
                ((ImageFile) dto.get("imageFile")).setBatchNo(String.valueOf(batchId));
            });
            final Map<String, Object> headers = new HashMap<>();
            headers.put(com.xforceplus.elephant.image.core.business.consts.Constants.TENANT_CODE, user.getTenantCode());
            headers.put("batchNo", batchId);
            headers.put("userInfo", JSONObject.toJSONString(user));
            try {
                if (batchService.sendEnableFlow(context)) {
                    LogUtil.attachEnableFlow(batchId.toString(), MQEnum.UPLOAD_IMAGE_SYNC_QUEUE);
                    final Channel channel = application.loadChannel(channelCode);
                    final BizRequest bizRequest = BizRequest.builder()
                                                            .bizQuantity(batchTaskEntity.getTaskCount())
                                                            .properties(headers)
                                                            .body(JSONObject.toJSONString(requestDtos))
                                                            .attrs(Maps.of("queueName", MQEnum.UPLOAD_IMAGE_SYNC_QUEUE))
                                                            .build();
                    channel.flowInRateLimit(bizRequest);
                } else {
                    mqUtils.sendByDirectExchange(MQEnum.UPLOAD_IMAGE_SYNC_QUEUE, JSONObject.toJSONString(requestDtos), headers);
                }
            } catch (Exception e) {
                LogContext.setLogPoint("enableFlowError", e.getMessage());
                logger.error("使用限流异常", e);
                LogContext.setLogPoint("error", String.format("使用限流异常:%s", e.getMessage()));
                mqUtils.sendByDirectExchange(MQEnum.UPLOAD_IMAGE_SYNC_QUEUE, JSONObject.toJSONString(requestDtos), headers);
            } finally {
                LogUtil.attachSaveCount(CollectionUtils.size(request.getUploadImageDTOList()));
            }
            return CommonResponse.ok("成功");
        }
        if (resetLevel.get()) {
            billImageTicketService.resetLevel(batchNo, user.getId());
        }
        if (ValidatorUtil.isNotEmpty(failedList)) {
            String message = String.format("共上传%d条，其中%d条上传成功，%d条上传失败", resultList.size(), resultList.size() - failedList.size(), failedList.size());
            logger.info("upload上传失败【{}】", failedList.stream().collect(Collectors.joining(",")));
            if (failedList.size() == resultList.size()) {
                message = "上传失败!";
            }
            return CommonResponse.ok(message);
        }
        final Integer queueSize = MqQueueSizeListener.QUEUE_SIZE.get(MQEnum.IMAGE_PDF_TRANSFORM_CALLBACK_QUEUE);
        if (resultList.stream().anyMatch(dto -> FileTypeEnum.PDF.getCode().equals(dto.getFileType()))
            && queueSize != null && queueSize > pdfQueueSize) {
            return CommonResponse.ok(String.format(pdfQueueTips, queueSize, queueSize / pdfQueueTpm + 1), imageIds);
        }
        LogUtil.attachSaveCount(CollectionUtils.size(request.getUploadImageDTOList()));
        return CommonResponse.ok("上传成功", imageIds);
    }

}
