package com.xforceplus.elephant.image.controller.image.process.imagelist;

import com.xforceplus.elephant.basecommon.process.AbstractProcess;
import com.xforceplus.elephant.basecommon.process.response.CommonResponse;
import com.xforceplus.elephant.basecommon.vaildate.ValidatorUtil;
import com.xforceplus.elephant.image.client.annotation.ApiV1Image;
import com.xforceplus.elephant.image.client.model.GetImageListRequest;
import com.xforceplus.elephant.image.client.model.ImageDTO;
import com.xforceplus.elephant.image.core.business.application.config.common.service.ConfigService;
import com.xforceplus.elephant.image.core.business.enums.DictEnum;
import com.xforceplus.elephant.image.core.domain.image.bean.ImageListResult;
import com.xforceplus.elephant.image.core.domain.image.bean.ImageSearchBean;
import com.xforceplus.elephant.image.core.domain.image.bean.ImageSearchResultBean;
import com.xforceplus.elephant.image.core.facade.application.collect.image.ImageFacade;
import com.xforceplus.elephant.image.core.facade.application.collect.ticket.TicketFacade;
import com.xforceplus.elephant.image.core.facade.dto.statistics.TicketStatisticsDto;
import com.xforceplus.elephant.image.core.util.RequestBuilder;
import com.xforceplus.elephant.image.mapper.ImageSearchMapper;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.dict.ImageType;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.dict.YesNo;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.entity.Image;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.meta.EntityMeta;
import com.xforceplus.ultraman.metadata.domain.vo.DataCollection;
import com.xforceplus.ultraman.metadata.domain.vo.dto.ConditionOp;
import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;
import javax.validation.ValidationException;
import lombok.RequiredArgsConstructor;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

@RequiredArgsConstructor
@Service
public class DefaultImageListProcess extends AbstractProcess<GetImageListRequest, ImageSearchResultBean> {

    private final ImageFacade imageFacade;
    private final TicketFacade ticketFacade;
    private final ImageSearchMapper mapper;
    private final ConfigService configService;

    @Override
    protected void check(final GetImageListRequest request) throws ValidationException {
        super.check(request);
        checkEmpty(request.getBillCode(), "单据CODE不能为空");
    }

    @Override
    protected CommonResponse<ImageSearchResultBean> process(final GetImageListRequest request) throws RuntimeException {
        final IAuthorizedUser user = UserInfoHolder.get();
        if (user == null) {
            throw new ValidationException("获取用户信息为空，请重新登陆");
        }
        //查询
        final ImageSearchBean searchBean = mapper.map(request);
        searchBean.setTenantId(user.getTenantId());
        final DataCollection<Image> imageList = imageFacade.selectBillImagesByCustomTicketParams(searchBean, EntityMeta.Image.code());
        if (ValidatorUtil.isEmpty(imageList.getRows())) {
            final ImageSearchResultBean resultBean = new ImageSearchResultBean();
            resultBean.setTotal(imageList.getRowNum().longValue());
            return CommonResponse.ok("成功", resultBean);
        }
        final List<Long> imageIds = imageList.getRows().stream().map(Image::getId).collect(Collectors.toList());
        final List<Map<String, Object>> ticketList = ticketFacade.selectTicketByImageIds(user.getTenantId(), imageIds, EntityMeta.Ticket.code());
        final Map<Long, List<Map<String, Object>>> ticketMap = ticketList.stream().collect(Collectors.groupingBy(map -> MapUtils.getLongValue(map, EntityMeta.Ticket.IMAGE_ID.code())));
        final List<String> tenantCodes = configService.selectDict(0L, DictEnum.NEED_WRAPPER_URL_TENANT_CONFIG.getCode(), String.class);
        final ImageListResult<ImageDTO> result = new ImageListResult<>(imageList.getRowNum().longValue(),
            imageList.getRows().stream().map(entity -> {
                final ImageDTO dto = mapper.map(entity);
                String fileUrl = ApiV1Image.PATH + "/image/" + entity.getId() + entity.getFileTypeHandle() + "?rid=" + System.currentTimeMillis();
                if (CollectionUtils.isNotEmpty(tenantCodes) && !CollectionUtils.containsAny(tenantCodes, user.getTenantCode()) && !ImageType._3.getCode().equals(entity.getImageType())) {
                    fileUrl = ValidatorUtil.isNotEmpty(entity.getFileUrlLocal()) ? entity.getFileUrlLocal() : entity.getFileUrlHandle();
                }
                dto.setFileUrl(fileUrl);
                dto.setFileUrlHandle(null);
                dto.setFileUrlOrigin(null);
                dto.setIsAdd(YesNo._0.getCode());
                dto.setIsChange(YesNo._0.getCode());
                dto.setIsReplace(YesNo._0.getCode());
                dto.setAuditBackStatus(YesNo._0.getCode());
                dto.setAmountWithTax("0");
                dto.setTaxAmount("0");

                final List<Map<String, Object>> tickets = ticketMap.get(entity.getId());
                if (ValidatorUtil.isNotEmpty(tickets)) {
                    final AtomicReference<BigDecimal> totalAmountWithTax = new AtomicReference<>(BigDecimal.ZERO);
                    final AtomicReference<BigDecimal> totalTaxAmount = new AtomicReference<>(BigDecimal.ZERO);
                    tickets.forEach(ticket -> ticketFacade.calcAmountByTicket(totalAmountWithTax, totalTaxAmount, ticket));

                    dto.setAmountWithTax(totalAmountWithTax.get().toString());
                    dto.setTaxAmount(totalTaxAmount.get().toString());
                    dto.setIsAdd(tickets.stream().anyMatch(ticket -> YesNo._1.getCode().equals(MapUtils.getString(ticket, EntityMeta.Ticket.IS_ADD.code()))) ? YesNo._1.getCode()
                        : YesNo._0.getCode());
                    dto.setIsChange(tickets.stream().anyMatch(ticket -> YesNo._1.getCode().equals(MapUtils.getString(ticket, EntityMeta.Ticket.IS_CHANGE.code()))) ? YesNo._1.getCode()
                        : YesNo._0.getCode());
                    dto.setIsReplace(
                        tickets.stream().anyMatch(ticket -> YesNo._1.getCode().equals(MapUtils.getString(ticket, EntityMeta.Ticket.IS_REPLACE.code()))) ? YesNo._1.getCode()
                            : YesNo._0.getCode());
                    dto.setAuditBackStatus(
                        tickets.stream().anyMatch(ticket -> YesNo._1.getCode().equals(MapUtils.getString(ticket, EntityMeta.Ticket.AUDIT_BACK_STATUS.code()))) ? YesNo._1.getCode()
                            : YesNo._0.getCode());
                    dto.setIsDelete(tickets.stream().anyMatch(ticket -> MapUtils.getBooleanValue(ticket, EntityMeta.Ticket.IS_DELETE.code())));
                    dto.setIsReplaceOrder(tickets.stream().anyMatch(ticket -> MapUtils.getBooleanValue(ticket, EntityMeta.Ticket.IS_REPLACE_ORDER.code())));
                    final Optional<Map<String, Object>> warningDetail = tickets.stream().filter(ticket -> StringUtils.isNotBlank(MapUtils.getString(ticket, EntityMeta.Ticket.WARNING_DETAIL.code())))
                        .findAny();
                    dto.setWarningDetail(warningDetail.isPresent() ? MapUtils.getString(warningDetail.get(), EntityMeta.Ticket.WARNING_DETAIL.code()) : StringUtils.EMPTY);
                    final Optional<Map<String, Object>> warningkey = tickets.stream().filter(ticket -> StringUtils.isNotBlank(MapUtils.getString(ticket, EntityMeta.Ticket.WARNING_KEY.code())))
                        .findAny();
                    dto.setWarningKey(warningkey.isPresent() ? MapUtils.getString(warningkey.get(), EntityMeta.Ticket.WARNING_KEY.code()) : StringUtils.EMPTY);
                    final Optional<Map<String, Object>> exceptionDetail = tickets.stream()
                        .filter(ticket -> StringUtils.isNotBlank(MapUtils.getString(ticket, EntityMeta.Ticket.EXCEPTION_DETAIL.code())))
                        .findAny();
                    dto.setExceptionDetail(exceptionDetail.isPresent() ? MapUtils.getString(exceptionDetail.get(), EntityMeta.Ticket.EXCEPTION_DETAIL.code()) : StringUtils.EMPTY);
                    final Optional<Map<String, Object>> exceptionKey = tickets.stream().filter(ticket -> StringUtils.isNotBlank(MapUtils.getString(ticket, EntityMeta.Ticket.EXCEPTION_KEY.code())))
                        .findAny();
                    dto.setExceptionKey(exceptionKey.isPresent() ? MapUtils.getString(exceptionKey.get(), EntityMeta.Ticket.EXCEPTION_KEY.code()) : StringUtils.EMPTY);

                }
                return dto;
            }).collect(Collectors.toList()));
        //统计
        final RequestBuilder ticketBuilder = new RequestBuilder()
            .field(EntityMeta.Ticket.TENANT_ID.code(), ConditionOp.eq, searchBean.getTenantId())
            .field(EntityMeta.Ticket.BILL_CODE.code(), ConditionOp.eq, searchBean.getBillCode());
        searchBean.getTicketParams().entrySet().stream()
            .filter(entry -> ValidatorUtil.isNotEmpty(entry.getValue()))
            .forEach(entry -> {
                final String[] values = entry.getValue().split(",");
                if (values.length > 1) {
                    ticketBuilder.field(entry.getKey(), ConditionOp.in, values);
                    return;
                }
                ticketBuilder.field(entry.getKey(), ConditionOp.eq, entry.getValue());
            });
        final TicketStatisticsDto ticketStatisticsDto = ticketFacade.statisticsTicketByParams(user.getTenantId(), ticketBuilder, EntityMeta.Ticket.code());
        final ImageSearchResultBean resultBean = mapper.map(ticketStatisticsDto);
        searchBean.setCountCoverFlag(true);
        searchBean.setPageSize(1);
        //单据封面个数
        final long billCoverCount = imageFacade.selectBillImagesByCustomTicketParams(searchBean, EntityMeta.Image.code()).getRowNum().longValue();
        resultBean.setBillCoverTotal(billCoverCount);
        resultBean.setTotal(result.getTotal());
        resultBean.setList(result.getList());
        if ((result.getTotal() == null || result.getTotal() == 0) && CollectionUtils.isNotEmpty(result.getList())) {
            logger.error("单据查询结果异常：total:{}, listSize:{}", result.getTotal(), CollectionUtils.size(result.getList()));
        }
        return CommonResponse.ok("成功", resultBean);
    }

}