package com.xforceplus.elephant.image.controller.image.process.personsave;

import com.alibaba.fastjson.JSONObject;
import com.xforceplus.elephant.basecommon.annotation.Dispatch;
import com.xforceplus.elephant.basecommon.dispatch.BeanDispatcher;
import com.xforceplus.elephant.basecommon.enums.image.ImageCategoryEnum;
import com.xforceplus.elephant.basecommon.enums.image.RecStatusEnum;
import com.xforceplus.elephant.basecommon.enums.image.RequireOcrFlagEnum;
import com.xforceplus.elephant.basecommon.help.BeanUtils;
import com.xforceplus.elephant.basecommon.process.AbstractProcess;
import com.xforceplus.elephant.basecommon.process.response.CommonResponse;
import com.xforceplus.elephant.basecommon.system.paas.NotificationUtils;
import com.xforceplus.elephant.basecommon.vaildate.ValidatorUtil;
import com.xforceplus.elephant.image.client.model.ImageDTO;
import com.xforceplus.elephant.image.client.model.SaveImageRequest;
import com.xforceplus.elephant.image.common.ScanCommonService;
import com.xforceplus.elephant.image.core.business.application.collect.batch.domain.EnableFlowContext;
import com.xforceplus.elephant.image.core.business.application.collect.batch.service.BatchService;
import com.xforceplus.elephant.image.core.business.application.config.common.enums.ReceiveFilterEnums;
import com.xforceplus.elephant.image.core.business.config.queue.MQUtils;
import com.xforceplus.elephant.image.core.business.consts.Constants;
import com.xforceplus.elephant.image.core.business.enums.MQEnum;
import com.xforceplus.elephant.image.core.business.util.LogUtil;
import com.xforceplus.elephant.image.core.domain.bill.BaseBillService;
import com.xforceplus.elephant.image.core.domain.config.ConfigDictionaryService;
import com.xforceplus.elephant.image.core.domain.config.ConfigSettingsService;
import com.xforceplus.elephant.image.core.domain.image.ImageService;
import com.xforceplus.elephant.image.core.expand.BillImageTicketService;
import com.xforceplus.elephant.image.core.expand.bean.TicketRecogBean;
import com.xforceplus.elephant.image.core.expand.impl.billimageticket.DefaultBillImageTicketServiceImpl;
import com.xforceplus.elephant.image.core.facade.application.config.dictionary.extfields.ExtFieldsReceiveFacade;
import com.xforceplus.elephant.image.core.repository.model.BatchTaskEntity;
import com.xforceplus.purchaser.bizratelimiter.model.Application;
import com.xforceplus.purchaser.bizratelimiter.model.BizRequest;
import com.xforceplus.purchaser.bizratelimiter.model.Channel;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.dict.ImageType;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.dict.YesNo;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.entity.Image;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.meta.EntityMeta;
import com.xforceplus.xlog.core.model.LogContext;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import javax.validation.ValidationException;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.groovy.util.Maps;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

@Slf4j
@Dispatch(isDefault = true)
@Service
public class DefaultPersonSaveProcess extends AbstractProcess<SaveImageRequest, List<Long>> {

    @Autowired
    protected ImageService imageService;
    @Autowired
    protected BaseBillService baseBillService;
    @Autowired
    protected NotificationUtils notificationUtils;
    @Autowired
    protected ConfigSettingsService configSettingsService;
    @Autowired
    protected ConfigDictionaryService configDictionaryService;
    @Autowired
    protected BillImageTicketService billImageTicketService;
    @Autowired
    protected MQUtils rabbitmqUtils;
    @Autowired
    protected BeanDispatcher beanDispatcher;
    @Autowired
    protected ExtFieldsReceiveFacade extFieldsReceiveFacade;
    @Autowired
    protected BatchService batchService;
    @Autowired(required = false)
    private Application application;
    @Autowired
    private ScanCommonService scanCommonService;

    @Override
    protected void check(SaveImageRequest request) throws ValidationException {
        super.check(request);
        checkEmpty(request.getEntities(), "操作对象不能为空");
        if (!YesNo._0.getCode().equals(request.getIsPublic())) {
            throw new RuntimeException("只能上传个人影像");
        }
        for (ImageDTO entity : request.getEntities()) {
            checkEmpty(entity.getImageCategory(), "影像目录不能为空");
            checkEmpty(entity.getImageSource(), "来源不能为空");
            checkEmpty(entity.getFileUrl(), "原始文件路径不能为空");
            checkEmpty(entity.getFileUrlHandle(), "处理文件路径不能为空");
            checkEmpty(entity.getFileType(), "原始文件类型不能为空");
            checkEmpty(entity.getFileTypeHandle(), "处理文件类型不能为空");
        }
    }

    @Override
    protected CommonResponse<List<Long>> process(SaveImageRequest request) throws RuntimeException {
        final IAuthorizedUser user = UserInfoHolder.get();
        final String batchNo = Long.toString(System.currentTimeMillis());
        //
        final List<JSONObject> images = requestToJSON(request, batchNo, user);
        LogUtil.attachSaveCount(CollectionUtils.size(request.getEntities()));
        final EnableFlowContext context = new EnableFlowContext().setTenantId(user.getTenantId()).setUserId(user.getId());
        final boolean flag = batchService.enable(context);
        LogContext.setLogPoint("enableFlow", flag);
        if (flag) {
            final String channelCode = batchService.getChannelCode(user.getTenantCode());
            final BatchTaskEntity batchTaskEntity = new BatchTaskEntity();
            batchTaskEntity.setTenantId(user.getTenantId());
            batchTaskEntity.setTenantCode(user.getTenantCode());
            batchTaskEntity.setCreateUserId(user.getId());
            batchTaskEntity.setCreateUserName(user.getUsername());
            batchTaskEntity.setCreateTime(new Date());
            batchTaskEntity.setTaskCount(request.getEntities().size());
            batchTaskEntity.setChannelCode(channelCode);
            final Long batchId = batchService.insert(batchTaskEntity);
            images.forEach(image -> image.put(EntityMeta.Image.BATCH_NO.code(), batchId));
            final Map<String, Object> headers = new HashMap<>();
            headers.put(com.xforceplus.elephant.image.core.business.consts.Constants.TENANT_CODE, user.getTenantCode());
            headers.put("batchNo", batchId);
            headers.put("userInfo", JSONObject.toJSONString(user));
            headers.put("channelCode", channelCode);
            headers.put(Constants.START_TIME, System.currentTimeMillis());
            try {
                if (batchService.sendEnableFlow(context)) {
                    final Channel channel = application.loadChannel(channelCode);
                    final BizRequest bizRequest = BizRequest.builder()
                                                            .bizQuantity(batchTaskEntity.getTaskCount())
                                                            .properties(headers)
                                                            .body(JSONObject.toJSONString(images))
                                                            .attrs(Maps.of("queueName", MQEnum.PERSON_TICKET_SYNC_QUEUE))
                                                            .build();
                    channel.flowInRateLimit(bizRequest);
                } else {
                    rabbitmqUtils.sendByDirectExchange(MQEnum.PERSON_TICKET_SYNC_QUEUE, JSONObject.toJSONString(images), headers);
                }
            } catch (Exception e) {
                LogContext.setLogPoint("enableFlowError", e.getMessage());
                logger.error("使用限流异常", e);
                LogContext.setLogPoint("error", String.format("使用限流异常:%s", e.getMessage()));
                rabbitmqUtils.sendByDirectExchange(MQEnum.PERSON_TICKET_SYNC_QUEUE, JSONObject.toJSONString(images), headers);
            }
            return CommonResponse.ok("成功");
        }
        final List<Long> imageIds = scanCommonService.personSave(images, batchNo, user.getId());

        return CommonResponse.ok("成功", imageIds);
    }

    private List<JSONObject> requestToJSON(final SaveImageRequest request, final String batchNo, final IAuthorizedUser user) {
        return request.getEntities().stream().map(dto -> {
            final Image entity = Image.fromOQSMap(BeanUtils.convertJSON(dto));
            //上传封面强转附件
            if (ValidatorUtil.isNotEmpty(entity.getBillCode())) {
                entity.setImageType(ImageType._99.getCode());
                entity.setImageCategory(ImageCategoryEnum.ATTACHMENT.getCode().toString());
            }
            entity.setBatchNo(batchNo);
            entity.setFileOrder("");
            entity.setBillCode(StringUtils.EMPTY);
            entity.setCreateUserCode(user.getUserCode());
            entity.setTenantId(user.getTenantId());
            entity.setTenantCode(user.getTenantCode());
            entity.setCreateUserId(user.getId());
            entity.setCreateUserCode(user.getUserCode());
            entity.setCreateUserName(user.getUsername());
            // 进口类发票场景改为前端传值, 未传值场景下默认需要识别
            final String requireOCRFlag = Optional.ofNullable(dto.getRequireOcrFlag()).orElseGet(RequireOcrFlagEnum.NEED.getCode()::toString);
            entity.setRequireOcrFlag(requireOCRFlag);
            if (YesNo._0.getCode().equals(requireOCRFlag)) {
                entity.setRecStatus(RecStatusEnum.FINISH_REC.getCode().toString());
            } else {
                entity.setRecStatus(RecStatusEnum.NOT_REC.getCode().toString());
            }

            entity.setIsPublic(YesNo._0.getCode());
            entity.setSerialNumber(request.getSerialNumber());
            entity.setBillEntityCode(request.getBillEntityCode());
            //处理扩展字段
            final JSONObject json = new JSONObject(entity.toOQSMap());
            if (ValidatorUtil.isNotEmpty(request.getExt())) {
                final JSONObject ticketPreData = new JSONObject(request.getExt());
                ticketPreData.remove(EntityMeta.Image.ID.code());
                ticketPreData.remove("token");
                final JSONObject area = ObjectUtils.firstNonNull(request.getExt().getJSONObject("area"), request.getExt().getJSONObject("areas"));
                if (ValidatorUtil.isNotEmpty(area)) {
                    ticketPreData.put(EntityMeta.Ticket.AREAS_ID.code(), area.get("orgId"));
                    ticketPreData.put(EntityMeta.Ticket.AREA_ID.code(), area.get("orgId"));
                    ticketPreData.put(EntityMeta.Ticket.AREA_CODE.code(), area.get("orgCode"));
                    ticketPreData.put(EntityMeta.Ticket.AREA_NAME.code(), area.get("orgName"));
                }
                ticketPreData.remove("area");
                ticketPreData.remove("areas");
                json.put(EntityMeta.Image.TICKET_PRE_DATA.code(), ticketPreData.toJSONString());
                json.putAll(extFieldsReceiveFacade.getExtFields(user.getTenantId(), ReceiveFilterEnums.IMAGE, request.getExt()));
            }
            return json;
        }).collect(Collectors.toList());
    }

    private void toAnnex(final List<JSONObject> images) {
        if (ValidatorUtil.isEmpty(images)) {
            return;
        }
        images.forEach(image -> {
            //附件保存
            final Long tenantId = image.getLong(EntityMeta.Image.TENANT_ID.code());
            final TicketRecogBean bean = new TicketRecogBean();
            bean.setImageId(image.getLong(EntityMeta.Image.ID.code()));
            bean.setTenantId(tenantId);
            bean.setBatchNo(image.getString(EntityMeta.Image.BATCH_NO.code()));
            bean.setIsPublic(image.getString(EntityMeta.Image.IS_PUBLIC.code()));
            bean.setScanUserId(image.getLong(EntityMeta.Image.CREATE_USER_ID.code()));
            bean.setScanUserName(image.getString(EntityMeta.Image.CREATE_USER_NAME.code()));
            final JSONObject discern = this.imageToTicket(image);
            final Map<String, Object> data = new HashMap<>();
            data.put("entityCode", image.getString(EntityMeta.Image.TICKET_CODE.code()));
            data.put("recogJson", discern);
            bean.setRecogList(Collections.singletonList(data));
            log.info("发送无需识别数据，imageId:【{}】,单据号：【{}】，所属单据：【{}】", image.getLong(EntityMeta.Image.ID.code()), image.getString(EntityMeta.Image.BILL_CODE.code()),
                image.getString(EntityMeta.Image.BILL_ENTITY_CODE.code()));
            beanDispatcher.dispatch(tenantId, DefaultBillImageTicketServiceImpl.class).insertOrUpdateRecogInfo(bean);
        });
    }

    /**
     * image字段字段转为ticket的字段
     *
     * @param image 扫描image对象
     * @return 识别结果对象
     */
    private JSONObject imageToTicket(JSONObject image) {
        final JSONObject discern = new JSONObject();
        discern.put(EntityMeta.Ticket.IMAGE_ID.code(), image.getString(EntityMeta.Image.ID.code()));
        discern.put(EntityMeta.Ticket.IMAGE_O_T_M_TICKET_ID.code(), image.getString(EntityMeta.Image.ID.code()));
        discern.put(EntityMeta.Ticket.SERIAL_NUMBER.code(), image.getString(EntityMeta.Image.SERIAL_NUMBER.code()));
        discern.put(EntityMeta.Ticket.SYSTEM_ORIG.code(), image.getString(EntityMeta.Image.IMAGE_SOURCE.code()));
        discern.put(EntityMeta.Ticket.SYSTEM_SOURCE.code(), image.getString(EntityMeta.Image.IMAGE_SOURCE.code()));
        discern.put(EntityMeta.Ticket.CREATE_USER_CODE.code(), image.getString(EntityMeta.Image.CREATE_USER_CODE.code()));
        //  增加ext1到ext10扩展字段更新
        discern.put(EntityMeta.Ticket.EXT1.code(), image.getString(EntityMeta.Image.EXT1.code()));
        discern.put(EntityMeta.Ticket.EXT2.code(), image.getString(EntityMeta.Image.EXT2.code()));
        discern.put(EntityMeta.Ticket.EXT3.code(), image.getString(EntityMeta.Image.EXT3.code()));
        discern.put(EntityMeta.Ticket.EXT4.code(), image.getString(EntityMeta.Image.EXT4.code()));
        discern.put(EntityMeta.Ticket.EXT5.code(), image.getString(EntityMeta.Image.EXT5.code()));
        discern.put(EntityMeta.Ticket.EXT6.code(), image.getString(EntityMeta.Image.EXT6.code()));
        discern.put(EntityMeta.Ticket.EXT7.code(), image.getString(EntityMeta.Image.EXT7.code()));
        discern.put(EntityMeta.Ticket.EXT8.code(), image.getString(EntityMeta.Image.EXT8.code()));
        discern.put(EntityMeta.Ticket.EXT9.code(), image.getString(EntityMeta.Image.EXT9.code()));
        discern.put(EntityMeta.Ticket.EXT10.code(), image.getString(EntityMeta.Image.EXT10.code()));
        return discern;
    }

}