package com.xforceplus.elephant.image.controller.ticket.process;

import static com.xforceplus.elephant.image.core.business.consts.Constants.TENANT_CODE;

import com.alibaba.fastjson.JSONObject;
import com.xforceplus.elephant.basecommon.enums.common.WebsocketNoticeTypeEnum;
import com.xforceplus.elephant.basecommon.help.BeanUtils;
import com.xforceplus.elephant.basecommon.help.RedisUtils;
import com.xforceplus.elephant.basecommon.process.AbstractProcess;
import com.xforceplus.elephant.basecommon.process.response.CommonResponse;
import com.xforceplus.elephant.basecommon.vaildate.ValidatorUtil;
import com.xforceplus.elephant.image.client.model.SendVerifyRequest;
import com.xforceplus.elephant.image.core.business.application.collect.task.domain.TaskVerifyContext;
import com.xforceplus.elephant.image.core.business.application.collect.task.enums.TaskTypeEnum;
import com.xforceplus.elephant.image.core.business.application.collect.ticket.service.TicketService;
import com.xforceplus.elephant.image.core.business.config.queue.MQUtils;
import com.xforceplus.elephant.image.core.business.domain.ProcessContext;
import com.xforceplus.elephant.image.core.business.enums.MQEnum;
import com.xforceplus.elephant.image.core.domain.image.bean.ImageSearchBean;
import com.xforceplus.elephant.image.core.expand.BillImageTicketService;
import com.xforceplus.elephant.image.core.expand.compare.CompareBillImageTicketService;
import com.xforceplus.elephant.image.core.facade.application.collect.image.ImageFacade;
import com.xforceplus.elephant.image.core.facade.application.collect.task.TaskFacade;
import com.xforceplus.elephant.image.core.util.RequestBuilder;
import com.xforceplus.elephant.image.mapper.ImageSearchMapper;
import com.xforceplus.tech.base.core.context.ContextKeys;
import com.xforceplus.tech.base.core.context.ContextService;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.dict.CheckStatus;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.dict.ImageType;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.entity.Image;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.entity.Ticket;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.meta.EntityMeta;
import com.xforceplus.ultraman.metadata.domain.vo.dto.ConditionOp;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.stream.Collectors;
import javax.validation.ValidationException;
import lombok.RequiredArgsConstructor;
import lombok.var;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

@Service
@RequiredArgsConstructor
public class SendVerifyFacadeProcess extends AbstractProcess<SendVerifyRequest, Boolean> {

    private final TicketService ticketService;
    private final BillImageTicketService billImageTicketService;
    private final CompareBillImageTicketService compareBillImageTicketService;
    private final MQUtils rabbitmqUtils;
    private final TaskFacade taskFacade;
    private final ExecutorService batchVerifyThreadPool;
    private final RedisUtils redisUtils;
    private final ContextService contextService;
    private final ImageSearchMapper imageSearchMapper;
    private final ImageFacade imageFacade;
    private final com.xforceplus.elephant.image.core.domain.ticket.TicketService oldTicketService;

    @Override
    protected void check(SendVerifyRequest request) throws ValidationException {
        super.check(request);
        if (request.getAllSelected() == null) {
            if (ValidatorUtil.isEmpty(request.getTicketId()) && ValidatorUtil.isEmpty(request.getTicketIds())) {
                throw new ValidationException("单证ID【ticketId】不能为空");
            }
        }
    }

    @Override
    protected CommonResponse<Boolean> process(SendVerifyRequest request) throws RuntimeException {
        final IAuthorizedUser authorizedUser = UserInfoHolder.get();
        if (null == authorizedUser) {
            logger.warn("个人单证批量验真 获取用户信息为空");
            throw new ValidationException("获取用户信息为空，请重新登陆.");
        }
        if (null != request.getTicketId()) {
            return sendSingleVerify(request.getTicketId());
        }
        if (null != request.getAllSelected()) {
            if (StringUtils.isBlank(request.getBillCode())) {
                throw new ValidationException("根据影像发送批量验真单据号不能为空");
            }
            swapTicketIdsByImageIds(request, authorizedUser);
            return sendBatchVerify(request, authorizedUser);
        }
        if (CollectionUtils.isNotEmpty(request.getTicketIds())) {
            return sendBatchVerify(request, authorizedUser);
        }
        return CommonResponse.ok("发送验真成功.");
    }

    private void swapTicketIdsByImageIds(SendVerifyRequest request, IAuthorizedUser authorizedUser) {
        prepareOperateData(request, authorizedUser);
        if (ValidatorUtil.isEmpty(request.getIds())) {
            throw new ValidationException("未查询到发票数据");
        }
        final String ticketCode = (null != request.getIsCompare() && request.getIsCompare()) ? EntityMeta.CompareTicket.code() : EntityMeta.Ticket.code();
        final var tickets = ticketService.selectReCheckTicketByImageIds(authorizedUser.getTenantId(), request.getIds(), ticketCode);
        if (CollectionUtils.isEmpty(tickets)) {
            throw new ValidationException("所勾选的数据未包含待验真和验真失败的单证，请核实后重新选择！");
        }
        final List<Long> ticketIds = tickets.stream().map(t -> MapUtils.getLong(t, EntityMeta.Ticket.ID.code())).collect(Collectors.toList());
        request.setTicketIds(ticketIds);
    }

    private CommonResponse<Boolean> sendBatchVerify(SendVerifyRequest request, IAuthorizedUser authorizedUser) {
        final String prefix = "verify:" + authorizedUser.getTenantId();
        request.getTicketIds().removeIf(id -> {
            final String cacheKey = prefix + ":" + id;
            return null != get(cacheKey);
        });

        if (CollectionUtils.isEmpty(request.getTicketIds())) {
            return CommonResponse.ok("选中的票10秒内已发过验真，请稍后重试!");
        }

        final String tenantCode = authorizedUser.getTenantCode();
        CompletableFuture.runAsync(() -> {
            try {
                contextService.set(ContextKeys.StringKeys.TENANTCODE_KEY, tenantCode);
                sendBatch(Boolean.TRUE.equals(request.getIsCompare()), prefix, request.getTicketIds(), authorizedUser);
            } finally {
                contextService.clear();
            }
        }, batchVerifyThreadPool);
        return CommonResponse.ok("发送批量验真成功!");
    }

    private void sendBatch(boolean isCompare, String prefix, List<Long> ticketIds, IAuthorizedUser user) {
        final String ticketBoCode = isCompare ? EntityMeta.CompareTicket.code() : EntityMeta.Ticket.code();
        final RequestBuilder ticketRequestBuilder = new RequestBuilder()
            .field(EntityMeta.Ticket.ID.code(), ConditionOp.in, ticketIds);
        final List<Ticket> ticketEntities;
        try {
            Optional.ofNullable(contextService.getAll()).ifPresent(map -> map.put("polymorphic", true));
            ticketEntities = ticketService.selectTicketByParams(user.getTenantId(), ticketRequestBuilder, ticketBoCode);
        } finally {
            Optional.ofNullable(contextService.getAll()).ifPresent(map -> map.remove("polymorphic"));
        }
        if (CollectionUtils.isEmpty(ticketEntities)) {
            logger.info("个人单证批量验真 未查到有效单证[{}]", ticketIds);
            return;
        }

        for (Ticket entity : ticketEntities) {
            final Map<String, Object> entityMap = entity.toOQSMap();
            if (null == entityMap) {
                logger.warn("个人单证批量验真 没有找到可操作的数据 单证[{}]", entity.getId());
                continue;
            }
            final String cacheKey = prefix + ":" + entity.getId();
            try {
                // 10秒内同样的单证不允许重复验真
                set(cacheKey);
                //发送验真
                final ProcessContext context = ProcessContext.context().put(TaskVerifyContext.builder()
                    .tenantId(user.getTenantId()).ticketMap(entityMap).userId(user.getId())
                    .userName(user.getUsername()).build());
                taskFacade.initiate(TaskTypeEnum.VERIFY, context);

                final JSONObject updateJson = isCompare
                    ? compareBillImageTicketService.checkTicketExceAndWarn(user.getTenantId(), BeanUtils.convertJSON(entity.toOQSMap())) :
                    billImageTicketService.checkTicketExceAndWarn(user.getTenantId(), BeanUtils.convertJSON(entity.toOQSMap()));
                if (null != updateJson && updateJson.size() > 0) {
                    final Map<String, Object> updateMap = new HashMap<>();
                    updateJson.forEach(updateMap::put);
                    final Integer count = ticketService.updateTicketByParam(entity.getId(), entity.getTicketCode(), updateMap);
                    if (count > 0) {
                        if (isCompare) {
                            compareBillImageTicketService.updateImageExceWarnData(user.getTenantId(), entity.getImageId());
                        } else {
                            billImageTicketService.updateImageExceWarnData(user.getTenantId(), entity.getImageId());
                        }
                    }
                }

                logger.info("个人单证批量验真 单证[{}]发送成功", entity.getId());
            } catch (Exception e) {
                logger.info("个人单证批量验真 单证[{}]验真异常[{}]", entity.getId(), e.getMessage(), e);
            }
        }
    }

    private CommonResponse<Boolean> sendSingleVerify(Long requestTicketId) {
        final Ticket entity = ticketService.selectTicketById(requestTicketId);
        if (null == entity) {
            return CommonResponse.failed("没有找到可操作的数据.");
        }
        if (CheckStatus._1.getCode().equals(entity.getCheckStatus())) {
            return CommonResponse.failed("发票已处于验真中，请稍后重试.");
        }
        final IAuthorizedUser authorizedUser = UserInfoHolder.get();// 获取登录用户上下文
        if (null == authorizedUser) {
            throw new ValidationException("获取用户信息为空，请重新登陆.");
        }
        final Map<String, Object> entityMap = entity.toOQSMap();
        if (null == entityMap) {
            return CommonResponse.failed("没有找到可操作的数据.");
        }
        final Long ticketId = Long.valueOf(entityMap.get("id").toString());
        //发送验真
        final ProcessContext context = ProcessContext.context().put(TaskVerifyContext.builder()
            .tenantId(authorizedUser.getTenantId()).ticketMap(entityMap).userId(authorizedUser.getId())
            .userName(authorizedUser.getUsername()).build());
        taskFacade.initiate(TaskTypeEnum.VERIFY, context);
        //重算异常
        final Map<String, Object> ticket = oldTicketService.selectByTicketCode(entity.getTicketCode(), entity.getId());
        if (null != ticket) {
            final JSONObject updateJson = billImageTicketService.checkTicketExceAndWarn(authorizedUser.getTenantId(), new JSONObject(ticket));
            if (null != updateJson && updateJson.size() > 0) {
                final Map<String, Object> updateMap = new HashMap<>();
                updateJson.forEach(updateMap::put);
                final Integer count = ticketService.updateTicketByParam(ticketId, entity.getTicketCode(), updateMap);
                if (count > 0) {
                    billImageTicketService.updateImageExceWarnData(authorizedUser.getTenantId(), entity.getImageId());
                }
            }
        }
        return CommonResponse.ok("发送验真成功.");
    }

    private void prepareOperateData(final SendVerifyRequest request, IAuthorizedUser user) {
        //全选
        if (request.getAllSelected()) {
            //条件筛选
            final ImageSearchBean searchBean = imageSearchMapper.map(request);
            searchBean.setTenantId(user.getTenantId());
            final String imageBillCode = Boolean.TRUE.equals(request.getIsCompare()) ? EntityMeta.CompareImage.code() : EntityMeta.Image.code();
            final List<Image> imageList = imageFacade.selectBillImagesByCustomTicketParams(searchBean, imageBillCode).getRows();
            final List<Long> ids = imageList.stream()
                .filter(image -> !ImageType._1.getCode().equals(image.getImageType()))
                .map(Image::getId)
                .collect(Collectors.toList());
            if (ValidatorUtil.isNotEmpty(request.getExcluded())) {
                final List<Long> excludedList = request.getExcluded().stream().map(Long::parseLong).collect(Collectors.toList());
                ids.removeAll(excludedList);
            }
            request.setIds(ids);
        }
        //非全选
        if (!request.getAllSelected()) {
            final List<Long> ids = request.getIncluded().stream().map(Long::parseLong).collect(Collectors.toList());
            request.setIds(ids);
        }
    }

    /**
     * 发送通知
     *
     * @param authorizedUser 用户
     * @param billCode       业务单号
     */
    private void sendWebScoket(IAuthorizedUser authorizedUser, String billCode) {
        final JSONObject mqMessage = new JSONObject();
        mqMessage.put("noticeType", WebsocketNoticeTypeEnum.CHECK_EXCEPTION_AND_WARNING.getCode());
        mqMessage.put("tenantId", authorizedUser.getTenantId());
        mqMessage.put("source", "发送验真完成");
        mqMessage.put("createUserId", authorizedUser.getId());
        mqMessage.put("billCode", billCode);
        final Map<String, Object> headers = new HashMap<>();
        headers.put(TENANT_CODE, authorizedUser.getTenantCode());
        rabbitmqUtils.sendByTopicExchange(MQEnum.WEBSOCKET_NOTICE_QUEUE, mqMessage, headers);
    }

    private Object get(String cacheKey) {
        try {
            return redisUtils.get(cacheKey);
        } catch (Exception e) {
            logger.error("个人单证批量验真 查询缓存异常", e);
            return null;
        }
    }

    private void set(String cacheKey) {
        try {
            redisUtils.set(cacheKey, 1, 10);
        } catch (Exception e) {
            logger.error("个人单证批量验真 设置缓存异常", e);
        }
    }

}