package com.xforceplus.elephant.image.controller.ticket.process.submit;

import com.xforceplus.elephant.basecommon.process.AbstractProcess;
import com.xforceplus.elephant.basecommon.process.response.CommonResponse;
import com.xforceplus.elephant.basecommon.vaildate.ValidatorUtil;
import com.xforceplus.elephant.image.client.model.SubmitTicketRequest;
import com.xforceplus.elephant.image.core.business.application.collect.ticket.service.TicketService;
import com.xforceplus.elephant.image.core.business.infrastructure.domain.purchase.InvoiceMainDTO;
import com.xforceplus.elephant.image.core.business.infrastructure.domain.purchase.InvoiceResponse;
import com.xforceplus.elephant.image.core.business.infrastructure.port.PurchaseService;
import com.xforceplus.elephant.image.core.facade.dto.commit.TicketCommitFailDto;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.dict.InvoiceType;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.entity.Ticket;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.meta.EntityMeta;
import java.util.Arrays;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.stream.Collectors;
import javax.validation.ValidationException;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.stereotype.Service;

/**
 * @Description: 单证提交前置校验（因现有租户定制没能力定制，交由产线定制开发；凯德租户定制）
 * @Author liuhongbin
 * @Date 2024/3/25 15:23
 */
@Slf4j
@RequiredArgsConstructor
@Service
public class SubmitCheckProcess extends AbstractProcess<SubmitTicketRequest, Set<TicketCommitFailDto>> {

    private final TicketService ticketService;
    private final PurchaseService purchaseService;
    private final ExecutorService imageThreadPool;

    @Override
    protected void check(SubmitTicketRequest request) throws ValidationException {
        super.check(request);
        checkEmpty(request.getImageIds(), "请先勾选发票");
        if (CollectionUtils.size(request.getImageIds()) > 200) {
            throw new ValidationException("最多支持200张发票提交");
        }
    }

    @Override
    protected CommonResponse<Set<TicketCommitFailDto>> process(SubmitTicketRequest request) throws RuntimeException {
        final IAuthorizedUser user = UserInfoHolder.get();
        if (null == user) {
            throw new ValidationException("获取用户信息为空，请重新登陆");
        }
        logger.info("提交校验请求,tenantId:{},tenantCode:{},ids:{}", user.getTenantId(), user.getTenantCode(), request.getImageIds());
        //查询单证
        final List<Ticket> ticketList = ticketService.selectBaseTicketByImageIds(user.getTenantId(), request.getImageIds(), EntityMeta.Ticket.code())
            .stream().filter(ticket -> Arrays.asList(EntityMeta.TicketInvoice.code(), EntityMeta.TicketVehicle.code()).contains(ticket.getTicketCode()))
            .collect(Collectors.toList());
        if (CollectionUtils.isEmpty(ticketList)) {
            return CommonResponse.ok("提交发票成功", new HashSet<>());
        }
        //根据发票代码号码查询发票池
        final List<Future<InvoiceMainDTO>> futures = ticketList.stream().map(ticket -> imageThreadPool.submit(() -> {
            final InvoiceResponse response = purchaseService.findInvoice(ticket.getTicketCode(), String.valueOf(ticket.getTenantId()), ticket.getInvoiceCode(), ticket.getInvoiceNo());
            final InvoiceMainDTO invoiceMain = response != null && ValidatorUtil.isNotEmpty(response.getResult()) ? response.getResult() : null;
            if (invoiceMain == null) {
                return null;
            }
            if ("1".equals(invoiceMain.getSystemSource())) {
                return invoiceMain;
            }
            return null;
        })).collect(Collectors.toList());
        final Set<TicketCommitFailDto> failDtoList = new LinkedHashSet<>();
        futures.forEach(future -> {
            try {
                final InvoiceMainDTO invoiceMain = future.get();
                if (invoiceMain == null) {
                    return;
                }
                final String invoiceTypeName = Optional.ofNullable(InvoiceType.fromCode(invoiceMain.getInvoiceKind())).map(InvoiceType::getDesc).orElse("");
                final String message = String.format("此发票已被【%s】通过【易快报】在【%s】上传占用，请勿重复采集", invoiceMain.getRecogUserName(), invoiceMain.getRecogTime());
                failDtoList.add(new TicketCommitFailDto(invoiceTypeName, invoiceMain.getInvoiceNo(), message));
            } catch (Exception e) {
                logger.error("校验发票异常", e);
            }
        });
        //返回校验结果
        return CommonResponse.ok(CollectionUtils.isEmpty(failDtoList) ? "提交发票成功" : "提交发票失败", failDtoList);
    }

}
