package com.xforceplus.elephant.image.controller.ticketsourcefile.process;

import com.xforceplus.elephant.basecommon.help.DateHelp;
import com.xforceplus.elephant.basecommon.help.HttpUtils;
import com.xforceplus.elephant.basecommon.vaildate.Validator;
import com.xforceplus.elephant.basecommon.vaildate.ValidatorUtil;
import com.xforceplus.elephant.image.client.model.UploadFileRequest;
import com.xforceplus.elephant.image.core.business.infrastructure.port.FileService;
import com.xforceplus.elephant.image.core.domain.ticket.TicketSourceFileService;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.entity.TicketSourceFile;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Date;
import java.util.List;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;
import javax.validation.ValidationException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StreamUtils;

@Service
public class UploadFileProcess implements Validator {

    private Logger logger = LoggerFactory.getLogger(getClass());

    @Autowired
    private TicketSourceFileService ticketSourceFileService;
    @Autowired
    private FileService fileService;

    protected void check(UploadFileRequest request) throws ValidationException {
        //ticketSourceFileIds为空的时候invoiceCode、invoiceNo不能为空
        if (ValidatorUtil.isEmpty(request.getTicketSourceFileIds())) {
            if (ValidatorUtil.isEmpty(request.getInvoiceCode())) {
                throw new ValidationException("发票代码不能为空");
            }
            if (ValidatorUtil.isEmpty(request.getInvoiceNo())) {
                throw new ValidationException("发票号码不能为空");
            }
        }
        //invoiceCode、invoiceNo为空的时候ticketSourceFileIds不能为空
        if (ValidatorUtil.isEmpty(request.getInvoiceCode()) || ValidatorUtil.isEmpty(request.getInvoiceNo())) {
            if (ValidatorUtil.isEmpty(request.getTicketSourceFileIds())) {
                throw new ValidationException("ticketIds不能为空");
            }
        }
    }

    /**
     * 电票源文件下载 支持批量下载和单个下载两种场景 批量下载（电票源文件列表）传ticketSourceFileIds参数 我的单证列表单个下载（此列表前端拿不到ticketSourceFileId参数)传 invoiceCode、invoiceNo参数
     *
     * @param request 请求
     * @return
     */
    public ResponseEntity<byte[]> execute(UploadFileRequest request) {
        check(request);
        return process(request);
    }

    protected ResponseEntity<byte[]> process(UploadFileRequest request) throws RuntimeException {
        final IAuthorizedUser user = UserInfoHolder.get();
        if (null == user) {
            throw new ValidationException("获取用户信息为空，请重新登陆");
        }
        if (request.getTicketSourceFileIds().size() > 0) {
            final List<TicketSourceFile> ticketSourceFileEntities = ticketSourceFileService.selectByIds(request.getTicketSourceFileIds());

            //打包下载
            ZipOutputStream zip = null;
            FileOutputStream fos = null;
            File file = null;
            FileInputStream fis = null;
            byte[] bytes = new byte[0];
            try {
                file = new File((new File("")).getAbsolutePath() + File.separator, "abc.zip");
                fos = new FileOutputStream(file);
                zip = new ZipOutputStream(fos);
                InputStream input = null;
                for (TicketSourceFile ticketSourceFile : ticketSourceFileEntities) {
                    try {
                        final String fileType = "." + ticketSourceFile.getFileType().toLowerCase();
                        final ZipEntry zipEntry = new ZipEntry(ticketSourceFile.getId() + fileType);
                        input = HttpUtils.doGetInputStream(ticketSourceFile.getFileUrl());
                        if (input != null) {
                            zip.putNextEntry(zipEntry);
                            int temp = 0;
                            while ((temp = input.read()) != -1) {
                                zip.write(temp);
                            }
                        }
                        zip.closeEntry();
                    } catch (Exception e) {
                        logger.error("读取图片并写入zip异常", e);
                    } finally {
                        if (input != null) {
                            input.close();
                        }
                    }
                }
                zip.close();
                fos.close();
                fis = new FileInputStream(file);
                bytes = StreamUtils.copyToByteArray(fis);
            } catch (IOException e) {
                logger.error("读取图片并写入zip异常", e);
            } finally {
                try {
                    if (fis != null) {
                        fis.close();
                    }
                    if (fos != null) {
                        fos.close();
                    }
                    if (file.exists()) {
                        file.delete();
                    }
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
            return ResponseEntity.ok()
                .header("Content-Disposition", "attachment;fileName=" + DateHelp.getTimeNormalStr(new Date()) + ".zip")
                .contentType(MediaType.APPLICATION_OCTET_STREAM)
                .contentLength(bytes.length).body(bytes);
        } else {
            final List<TicketSourceFile> ticketSourceFileList = ticketSourceFileService
                .selectSourceFileByCodeNo(user.getTenantId(), request.getInvoiceNo(), request.getInvoiceCode(), null);
            if (CollectionUtils.isEmpty(ticketSourceFileList)) {
                throw new ValidationException("电票源文件为空");
            }
            final TicketSourceFile ticketSourceFile = ticketSourceFileList.get(0);
            final String url = ticketSourceFile.getFileUrl();
            final String fileType = "." + ticketSourceFile.getFileType().toLowerCase();
            if (url.startsWith("http")) {
                try (InputStream is = HttpUtils.doGetInputStream(url)) {
                    final byte[] bytes = StreamUtils.copyToByteArray(is);
                    return ResponseEntity.ok()
                        .header("Content-Disposition", "attachment;fileName=" + ticketSourceFile.getId() + fileType)
                        .contentType(fileService.mediaType(fileType))
                        .contentLength(bytes.length).body(bytes);
                } catch (Exception e) {
                    logger.error("下载失败", e);
                }
            } else {
                try (InputStream is = fileService.getFileInputStream(user.getTenantId(), user.getId(), url)) {
                    final byte[] bytes = StreamUtils.copyToByteArray(is);
                    return ResponseEntity.ok()
                        .header("Content-Disposition", "attachment;fileName=" + ticketSourceFile.getId() + fileType)
                        .contentType(fileService.mediaType(fileType))
                        .contentLength(bytes.length).body(bytes);
                } catch (Exception e) {
                    logger.error("下载失败", e);
                }
            }

        }

        return ResponseEntity.notFound().build();
    }

}
