package com.xforceplus.elephant.image.common;

import static com.xforceplus.elephant.image.core.business.consts.Constants.TENANT_CODE;

import com.alibaba.fastjson.JSONObject;
import com.google.api.client.util.Lists;
import com.xforceplus.elephant.basecommon.baseconst.ReleationConstants;
import com.xforceplus.elephant.basecommon.dispatch.BeanDispatcher;
import com.xforceplus.elephant.basecommon.enums.common.WebsocketNoticeTypeEnum;
import com.xforceplus.elephant.basecommon.help.BeanUtils;
import com.xforceplus.elephant.basecommon.help.RedisUtils;
import com.xforceplus.elephant.basecommon.vaildate.ValidatorUtil;
import com.xforceplus.elephant.image.core.business.application.collect.bill.service.BillService;
import com.xforceplus.elephant.image.core.business.application.collect.image.service.ImageService;
import com.xforceplus.elephant.image.core.business.config.queue.MQUtils;
import com.xforceplus.elephant.image.core.business.enums.MQEnum;
import com.xforceplus.elephant.image.core.domain.common.TicketSupport;
import com.xforceplus.elephant.image.core.domain.compare.bill.CompareBaseBillService;
import com.xforceplus.elephant.image.core.domain.compare.image.CompareImageService;
import com.xforceplus.elephant.image.core.expand.bean.BillRecogBean;
import com.xforceplus.elephant.image.core.expand.bean.TicketRecogBean;
import com.xforceplus.elephant.image.core.expand.compare.CompareBillImageTicketService;
import com.xforceplus.elephant.image.core.facade.application.collect.image.ImageFacade;
import com.xforceplus.elephant.image.core.facade.application.compare.ticket.CompareTicketFacade;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.dict.BackType;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.dict.BillDataStatus;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.dict.CalculateStatus;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.dict.HangType;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.dict.ImageType;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.dict.ReceiveStatus;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.entity.BaseBill;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.entity.CompareImage;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.entity.Image;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.meta.EntityMeta;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

@Slf4j
@Component
@RequiredArgsConstructor
public class CompareScanCommonService {

    private final RedisUtils redisUtils;
    private final TicketSupport ticketSupport;
    private final CompareTicketFacade ticketFacade;
    private final CompareImageService imageService;
    private final CompareBillImageTicketService billImageTicketService;
    private final CompareBaseBillService baseBillService;
    private final BillService billService;
    private final ImageService newImageService;
    private final BeanDispatcher beanDispatcher;
    private final MQUtils mqUtils;
    @Value("${image.lock.leaseTimeMillis:180000}")
    private Integer leaseTimeMillis;
    @Value("${image.lock.maxRetryTimes:10}")
    private Integer maxRetryTimes;
    @Value("${image.lock.retryIntervalTimeMillis:2000}")
    private Integer retryIntervalTimeMillis;

    /**
     * 混扫通用逻辑
     *
     * @param images 影像列表
     * @return 影像id列表
     */
    public List<Long> saveImage(final List<Image> images, Map<String, BaseBill> mainBillMap, final long userId) {
        final String key = String.format("scan:saveImage:%s", userId);
        final List<Long> imageIds = Lists.newArrayList();

        boolean lock = false;
        try {
            lock = redisUtils.tryGetDistributedLock(key, key, leaseTimeMillis, maxRetryTimes, retryIntervalTimeMillis);
            if (!lock) {
                log.info("混扫通用逻辑{} 获取锁:{}失败!", userId, key);
                imageIds.addAll(saveImage(images, mainBillMap, userId));
                return imageIds;
            }
            imageIds.addAll(imageService.saveImages(images));
            //创建单据
            toBill(images, mainBillMap);
        } catch (Exception e) {
            log.error("混扫通用逻辑{} 异常:{}", userId, e);
        } finally {
            if (lock) {
                redisUtils.releaseDistributedLock(key, key);
            }
        }
        return imageIds;
    }

    /**
     * 创建单据
     *
     * @param images billImages
     */
    protected void toBill(List<Image> images, Map<String, BaseBill> mainBillMap) {
        if (ValidatorUtil.isEmpty(images)) {
            return;
        }
        final List<Image> billImages = images.stream()
            .filter(image -> ImageType._1.getCode().equals(image.getImageType()))
            .collect(Collectors.toList());
        final Map<String, List<Image>> billImageMap = images.stream().collect(Collectors.groupingBy(Image::getBillCode));
        final Map<String, BaseBill> billCodeBillMap = mainBillMap.values().stream().collect(Collectors.toMap(BaseBill::getBillCode, Function.identity(), (o1, o2) -> o1));
        billImages.forEach(image -> {
            //单据保存
            final BillRecogBean bean = new BillRecogBean();
            bean.setImageId(image.getId());
            bean.setTenantId(image.getTenantId());
            bean.setBatchNo(image.getBatchNo());
            bean.setIsPublic(image.getIsPublic());
            bean.setScanUserId(image.getCreateUserId());
            bean.setScanUserName(image.getCreateUserName());
            bean.setBillTypeCode(image.getBillEntityCode());
            final JSONObject discern = wrapperRecogInfo(image, billCodeBillMap.get(image.getBillCode()));
            if (billImageMap.containsKey(image.getBillCode())) {
                final int imageCount = billImageMap.get(image.getBillCode()).size();
                discern.put(EntityMeta.BaseBill.IMAGE_COUNT.code(), imageCount);
                final String receiveStatus = imageCount > 1 ? ReceiveStatus._2.toString() : ReceiveStatus._1.toString();
                discern.put(EntityMeta.BaseBill.RECEIVE_STATUS.code(), receiveStatus);
                final String calculateStatus = imageCount > 1 ? CalculateStatus._99.toString() : CalculateStatus._1.toString();
                discern.put(EntityMeta.BaseBill.CALCULATE_STATUS.code(), calculateStatus);
            }
            discern.put(ReleationConstants.BILL_OTO_IMAGE_ID, image.getId());
            bean.setRecogInfo(discern);
            log.info("发送单据，imageId:【{}】,bill_code：【{}】", image.getId(), image.getBillCode());
            billImageTicketService.insertOrUpdateBillRecogInfo(bean);
            //重算层级（不要注释此行，目的是解决单据后创建的问题）
            billImageTicketService.resetLevel(image.getTenantId(), image.getBillCode());
            //发送通知
            final JSONObject mqMessage = new JSONObject();
            mqMessage.put("noticeType", WebsocketNoticeTypeEnum.IMAGE_DISCERN.getCode());
            mqMessage.put("tenantId", image.getTenantId());
            mqMessage.put("source", "imageSaveBill");
            mqMessage.put("createUserId", image.getCreateUserId());
            mqMessage.put("billCode", image.getBillCode());
            mqMessage.put("isBill", true);
            final Map<String, Object> headers = new HashMap<>();
            headers.put(TENANT_CODE, image.getTenantCode());
            mqUtils.sendByTopicExchange(MQEnum.COMPARE_WEBSOCKET_NOTICE_QUEUE, mqMessage, headers);
        });
    }

    protected JSONObject wrapperRecogInfo(Image image, BaseBill baseBill) {
        final JSONObject discern = new JSONObject();
        //复制主数据字段
        if (baseBill != null) {
            discern.putAll(new JSONObject(baseBill.toOQSMap()));
            discern.remove(EntityMeta.BaseBill.ID.code());
            discern.put(EntityMeta.BaseBill.MAIN_BILL_DATA_STATUS.code(), baseBill.getBillDataStatus());
        }
        discern.put("bill_code", image.getBillCode());
        discern.put("create_user_code", image.getCreateUserCode());
        discern.put("system_source", image.getImageSource());
        discern.put("serial_number", image.getSerialNumber());
        return discern;
    }

    /**
     * 影像补扫通用逻辑
     *
     * @param user        用户
     * @param bill        单据
     * @param parentImage 插入位置影像
     * @param images      补扫列表
     * @return 影像id列表
     */
    public List<Long> insertBillScan(IAuthorizedUser user, BaseBill bill, Image parentImage, List<JSONObject> images) {
        final List<JSONObject> billImages = new ArrayList<>();
        final List<Long> imageIds = new ArrayList<>();
        if (ValidatorUtil.isEmpty(parentImage)) {
            images.forEach(item -> {
                final List<Long> itemIds = imageService.saveForMap(Arrays.asList(item));
                if (ValidatorUtil.isEmpty(itemIds)) {
                    return;
                }
                if (ticketSupport.isAttachSalesImage(item.getString("image_type"))) {
                    billImages.add(item);
                }
                ticketFacade.saveTicketVirtual(CompareImage.fromOQSMap(item), itemIds.get(0));
                imageIds.addAll(itemIds);
            });
        } else {
            images.forEach(item -> {
                final List<Long> itemIds = imageService.insertImagesForMap(parentImage, Arrays.asList(item));
                if (ValidatorUtil.isEmpty(itemIds)) {
                    return;
                }
                if (ticketSupport.isAttachSalesImage(item.getString("image_type"))) {
                    billImages.add(item);
                }
                ticketFacade.saveTicketVirtual(CompareImage.fromOQSMap(item), itemIds.get(0));
                imageIds.addAll(itemIds);
            });
        }
        if (ValidatorUtil.isNotEmpty(billImages)) {
            //创建附件
            toAnnex(billImages);
            //重算层级
            billImageTicketService.resetLevel(user.getTenantId(), bill.getBillCode());
            //发送通知
            final JSONObject mqMessage = new JSONObject();
            mqMessage.put("noticeType", WebsocketNoticeTypeEnum.IMAGE_DISCERN.getCode());
            mqMessage.put("tenantId", user.getTenantId());
            mqMessage.put("source", "imageInsertAttach");
            mqMessage.put("createUserId", user.getId());
            mqMessage.put("billCode", bill.getBillCode());
            final Map<String, Object> headers = new HashMap<>();
            headers.put(TENANT_CODE, user.getTenantCode());
            mqUtils.sendByTopicExchange(MQEnum.COMPARE_WEBSOCKET_NOTICE_QUEUE, mqMessage, headers);
        }
        //上传成功，则修改单据状态为待处理
        if (BillDataStatus.fromCode(bill.getBillDataStatus()) == BillDataStatus._6) {
            final JSONObject updater = new JSONObject();
            updater.put(EntityMeta.BaseBill.BILL_DATA_STATUS.code(), BillDataStatus._0.getCode());
            baseBillService.updateByBillIdSelective(bill.getId(), updater);
        }
        return imageIds;
    }

    private void toAnnex(final List<JSONObject> attachImages) {
        if (ValidatorUtil.isEmpty(attachImages)) {
            return;
        }
        attachImages.forEach(image -> {
            //附件保存
            final Long tenantId = image.getLong(EntityMeta.Image.TENANT_ID.code());
            final TicketRecogBean bean = new TicketRecogBean();
            bean.setImageId(image.getLong(EntityMeta.Image.ID.code()));
            bean.setTenantId(tenantId);
            bean.setBatchNo(image.getString(EntityMeta.Image.BATCH_NO.code()));
            bean.setIsPublic(image.getString(EntityMeta.Image.IS_PUBLIC.code()));
            bean.setScanUserId(image.getLong(EntityMeta.Image.CREATE_USER_ID.code()));
            bean.setScanUserName(image.getString(EntityMeta.Image.CREATE_USER_NAME.code()));
            final JSONObject discern = this.imageToTicket(image);
            final Map<String, Object> data = new HashMap<>();
            data.put("entityCode", EntityMeta.CompareTicketAttachment.code());
            data.put("recogJson", discern);
            bean.setRecogList(Arrays.asList(data));
            log.info("发送附件，imageId:【{}】,单据号：【{}】，所属单据：【{}】", image.getLong(EntityMeta.Image.ID.code()), image.getString(EntityMeta.Image.BILL_CODE.code()),
                image.getString(EntityMeta.Image.BILL_ENTITY_CODE.code()));
            beanDispatcher.dispatch(tenantId, CompareBillImageTicketService.class).insertOrUpdateRecogInfo(bean);
        });
    }

    /**
     * image字段字段转为ticket的字段
     *
     * @param image 扫描image对象
     * @return 识别结果对象
     */
    private JSONObject imageToTicket(JSONObject image) {
        final JSONObject discern = new JSONObject();
        discern.put("bill_code", image.getString(EntityMeta.Image.BILL_CODE.code()));
        discern.put("bill_entity_code", image.getString(EntityMeta.Image.BILL_ENTITY_CODE.code()));
        discern.put("bill_type_code", image.getString(EntityMeta.Image.BILL_ENTITY_CODE.code()));
        discern.put("serial_number", image.getString(EntityMeta.Image.SERIAL_NUMBER.code()));
        discern.put("system_orig", image.getString(EntityMeta.Image.IMAGE_SOURCE.code()));
        discern.put("system_source", image.getString(EntityMeta.Image.IMAGE_SOURCE.code()));
        discern.put("create_user_code", image.getString(EntityMeta.Image.CREATE_USER_CODE.code()));
        // 增加ext1到ext10扩展字段更新
        discern.put("ext1", image.getString(EntityMeta.Image.EXT1.code()));
        discern.put("ext2", image.getString(EntityMeta.Image.EXT2.code()));
        discern.put("ext3", image.getString(EntityMeta.Image.EXT3.code()));
        discern.put("ext4", image.getString(EntityMeta.Image.EXT4.code()));
        discern.put("ext5", image.getString(EntityMeta.Image.EXT5.code()));
        discern.put("ext6", image.getString(EntityMeta.Image.EXT6.code()));
        discern.put("ext7", image.getString(EntityMeta.Image.EXT7.code()));
        discern.put("ext8", image.getString(EntityMeta.Image.EXT8.code()));
        discern.put("ext9", image.getString(EntityMeta.Image.EXT9.code()));
        discern.put("ext10", image.getString(EntityMeta.Image.EXT10.code()));
        return discern;
    }

    /**
     * 混扫通用逻辑
     *
     * @param images 影像列表
     * @return 影像id列表
     */
    public void newSaveImage(final List<Image> images, Map<String, BaseBill> mainBillMap, Map<String, BaseBill> billMap, final long userId) {
        final String key = String.format("scan:saveImage:%s", userId);

        boolean lock = false;
        try {
            lock = redisUtils.tryGetDistributedLock(key, key, leaseTimeMillis, maxRetryTimes, retryIntervalTimeMillis);
            if (!lock) {
                log.info("混扫通用逻辑{} 获取锁:{}失败!", userId, key);
                newSaveImage(images, mainBillMap, billMap, userId);
                return;
            }
            //优先创建单据
            newToBill(images, mainBillMap, billMap);
            newImageService.createMultiWithId(images, EntityMeta.CompareImage.code());
            newImageService.ocrInitiate(images, true);
        } catch (Exception e) {
            log.error(String.format("混扫通用逻辑%s 异常:%s", userId, e.getMessage()), e);
        } finally {
            if (lock) {
                redisUtils.releaseDistributedLock(key, key);
            }
        }
    }

    /**
     * 创建单据
     *
     * @param images billImages
     */
    protected void newToBill(List<Image> images, Map<String, BaseBill> mainBillMap, Map<String, BaseBill> billMap) {
        if (ValidatorUtil.isEmpty(images)) {
            return;
        }
        final List<Image> billImages = images.stream()
            .filter(image -> ImageType._1.getCode().equals(image.getImageType()))
            .collect(Collectors.toList());
        final Map<String, List<Image>> billImageMap = images.stream().collect(Collectors.groupingBy(Image::getBillCode));

        final Map<String, List<Map<String, Object>>> createBills = new HashMap<>();
        final List<Map<String, Object>> updateBills = new ArrayList<>();
        billImages.forEach(image -> {
            if (billMap.containsKey(image.getBillCode())) {
                final BaseBill bill = new BaseBill();
                bill.setId(billMap.get(image.getBillCode()).getId());
                bill.setImageId(image.getId());
                bill.setBillOTOImageId(image.getId());
                bill.setBatchNo(image.getBatchNo());
                bill.setSerialNumber(image.getSerialNumber());
                bill.setScanUserId(image.getCreateUserId());
                bill.setScanUserName(image.getCreateUserName());
                bill.setScanCreateTime(LocalDateTime.now());
                bill.setUpdateTime(LocalDateTime.now());
                updateBills.add((Map<String, Object>) bill.toOQSMap());
                return;
            }
            final BaseBill bill = mainBillMap.containsKey(image.getBillCode()) ? mainBillMap.get(image.getBillCode()) : new BaseBill();
            bill.setId(null);
            bill.setCreateUserId(image.getCreateUserId());
            bill.setCreateUserName(image.getCreateUserName());
            bill.setCreateTime(LocalDateTime.now());
            bill.setMainBillDataStatus(bill.getBillDataStatus());
            bill.setBillDataStatus(BillDataStatus._0.getCode());
            bill.setBillCode(image.getBillCode());
            bill.setBillTypeCode(image.getBillEntityCode());
            bill.setCreateUserCode(image.getCreateUserCode());
            bill.setSystemSource(image.getImageSource());
            bill.setSerialNumber(image.getSerialNumber());
            bill.setImageId(image.getId());
            bill.setBillOTOImageId(image.getId());
            bill.setTenantId(image.getTenantId());
            bill.setTenantCode(image.getTenantCode());
            bill.setBatchNo(image.getBatchNo());
            bill.setIsPublic(image.getIsPublic());
            bill.setScanUserId(image.getCreateUserId());
            bill.setScanUserName(image.getCreateUserName());
            bill.setScanCreateTime(LocalDateTime.now());
            bill.setExceptionCount(0L);
            bill.setTicketCount(0L);
            bill.setWarningCount(0L);
            bill.setBackType(BackType._0.code());

            if (billImageMap.containsKey(image.getBillCode())) {
                final int scanCount = billImageMap.get(image.getBillCode()).size();
                bill.setImageCount((long) scanCount);
                bill.setReceiveStatus(scanCount > 1 ? ReceiveStatus._2.toString() : ReceiveStatus._1.toString());
                bill.setCalculateStatus(scanCount > 1 ? CalculateStatus._99.toString() : CalculateStatus._1.toString());
            }

            if (!createBills.containsKey(bill.getBillTypeCode())) {
                createBills.put(bill.getBillTypeCode(), new ArrayList<>());
            }
            createBills.get(bill.getBillTypeCode()).add((Map<String, Object>) bill.toOQSMap());
        });

        if (CollectionUtils.isNotEmpty(updateBills)) {
            billService.updateMulti(updateBills, EntityMeta.CompareBaseBill.code());
        }
        if (MapUtils.isNotEmpty(createBills)) {
            createBills.forEach((billTypeCode, bills) -> billService.createMulti(bills, billTypeCode));
        }
    }

}
