package com.xforceplus.elephant.image.controller.image.process.saveimage;

import static com.xforceplus.elephant.image.core.business.consts.Constants.TENANT_CODE;

import com.alibaba.fastjson.JSONObject;
import com.xforceplus.elephant.basecommon.annotation.Dispatch;
import com.xforceplus.elephant.basecommon.baseconst.ReleationConstants;
import com.xforceplus.elephant.basecommon.baseconst.SettingsConstants;
import com.xforceplus.elephant.basecommon.enums.EntityConstant;
import com.xforceplus.elephant.basecommon.enums.common.WebsocketNoticeTypeEnum;
import com.xforceplus.elephant.basecommon.enums.cqp.bill.BackTypeEnum;
import com.xforceplus.elephant.basecommon.help.BeanUtils;
import com.xforceplus.elephant.basecommon.process.response.CommonResponse;
import com.xforceplus.elephant.basecommon.system.paas.NotificationUtils;
import com.xforceplus.elephant.basecommon.vaildate.ValidatorUtil;
import com.xforceplus.elephant.image.client.model.ImageDTO;
import com.xforceplus.elephant.image.client.model.SaveImageRequest;
import com.xforceplus.elephant.image.core.business.application.collect.batch.domain.EnableFlowContext;
import com.xforceplus.elephant.image.core.business.application.collect.batch.service.BatchService;
import com.xforceplus.elephant.image.core.business.application.config.common.enums.ReceiveFilterEnums;
import com.xforceplus.elephant.image.core.business.config.queue.MQUtils;
import com.xforceplus.elephant.image.core.business.enums.MQEnum;
import com.xforceplus.elephant.image.core.domain.bill.BaseBillService;
import com.xforceplus.elephant.image.core.domain.config.ConfigSettingsService;
import com.xforceplus.elephant.image.core.domain.image.ImageService;
import com.xforceplus.elephant.image.core.repository.model.BatchTaskEntity;
import com.xforceplus.purchaser.bizratelimiter.model.Application;
import com.xforceplus.purchaser.bizratelimiter.model.BizRequest;
import com.xforceplus.purchaser.bizratelimiter.model.Channel;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.dict.BackType;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.dict.BillDataStatus;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.dict.CalculateStatus;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.dict.ImageType;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.dict.RecStatus;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.dict.YesNo;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.entity.BaseBill;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.entity.Image;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.meta.EntityMeta;
import com.xforceplus.xlog.core.model.LogContext;
import io.vavr.Tuple2;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javax.validation.ValidationException;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.groovy.util.Maps;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

@Dispatch(tenantCode = "CQP")
@Service
public class CQPSaveImageProcess extends DefaultSaveImageProcess {

    private static final String REPEAT_BILL = "[REPEAT_BILL]";

    @Autowired
    private ImageService imageService;
    @Autowired
    private BaseBillService baseBillService;
    @Autowired
    private NotificationUtils notificationUtils;
    @Autowired
    private ConfigSettingsService configSettingsService;
    @Autowired
    private MQUtils rabbitmqUtils;
    @Autowired
    private BatchService batchService;
    @Autowired(required = false)
    private Application application;

    @Override
    protected void check(SaveImageRequest request) throws ValidationException {
        super.check(request);
        checkEmpty(request.getEntities(), "操作对象不能为空");
        if (!YesNo._1.getCode().equals(request.getIsPublic())) {
            throw new ValidationException("只能上传公共影像");
        }
        for (ImageDTO entity : request.getEntities()) {
            checkEmpty(entity.getImageCategory(), "影像目录不能为空");
            checkEmpty(entity.getImageSource(), "来源不能为空");
            checkEmpty(entity.getFileUrl(), "原始文件路径不能为空");
            checkEmpty(entity.getFileUrlHandle(), "处理文件路径不能为空");
            checkEmpty(entity.getFileType(), "原始文件类型不能为空");
            checkEmpty(entity.getFileTypeHandle(), "处理文件类型不能为空");
            checkEmpty(entity.getIsPublic(), "是否公共上传不能为空");
        }
    }

    @Override
    protected CommonResponse<List<Long>> process(SaveImageRequest request) throws RuntimeException {
        final long billCount = request.getEntities().stream()
            .filter(dto -> ValidatorUtil.isNotEmpty(dto.getBillCode())).map(r -> r.getBillCode()).distinct()
            .count();
        //按单据封装请求
        final Map<String, List<ImageDTO>> dtoMap = dtoMap(request.getEntities());
        //查询已有单据
        final Map<String, BaseBill> billMap = billMap(dtoMap);
        //过滤
        final List<Tuple2<Integer, String>> billFilter = billFilter(dtoMap, billMap);
        //保存
        final List<Long> imageIds = saveImages(request, dtoMap, billMap);
        //消息
        final String message = message(billFilter, billCount);

        return CommonResponse.ok(message, imageIds);
    }

    /**
     * 根据单据组装请求dtos
     *
     * @param entities entities
     * @return
     */
    private Map<String, List<ImageDTO>> dtoMap(List<ImageDTO> entities) {
        final Map<String, List<ImageDTO>> dtoMap = new LinkedHashMap<>();
        String billCode = StringUtils.EMPTY;
        final String regexSplit = configSettingsService.select(UserInfoHolder.get().getTenantId(), SettingsConstants.BILL_CODE_REGEX_SPLIT);
        for (ImageDTO entity : entities) {
            if (ValidatorUtil.isNotEmpty(entity.getBillCode())) {
                billCode = entity.getBillCode();
                entity.setImageType(ImageType._1.getCode());
                entity.setRequireOcrFlag(YesNo._0.getCode());
                entity.setRecStatus(RecStatus._2.getCode());
                entity.setCalculateStatus(CalculateStatus._1.getCode());
                if (ValidatorUtil.isNotEmpty(regexSplit)) {
                    for (String reg : regexSplit.split(",")) {
                        final Pattern pattern = Pattern.compile(reg);
                        final Matcher matcher = pattern.matcher(billCode);
                        if (matcher.find()) {
                            billCode = matcher.group(1);
                            break;
                        }
                    }
                }
            } else {
                entity.setRequireOcrFlag(YesNo._1.getCode());
                entity.setRecStatus(RecStatus._0.getCode());
            }
            if (!dtoMap.containsKey(billCode)) {
                dtoMap.put(billCode, new ArrayList<>());
            }
            dtoMap.get(billCode).add(entity);
            entity.setBillCode(billCode);
        }
        return dtoMap;
    }

    /**
     * 根据扫描的单据号查询系统中对应的单据
     *
     * @param dtoMap 单据dtoMap
     * @return
     */
    private Map<String, BaseBill> billMap(Map<String, List<ImageDTO>> dtoMap) {
        final List<String> billCodes = dtoMap.keySet().stream()
            .filter(billCode -> ValidatorUtil.isNotEmpty(billCode))
            .collect(Collectors.toList());
        if (ValidatorUtil.isEmpty(billCodes)) {
            return new HashMap<>();
        }
        final Map<String, BaseBill> billMap = new HashMap<>();
        final long tenantId = UserInfoHolder.get().getTenantId();
        baseBillService.selectBillDataByBillCodes(tenantId, billCodes)
            .forEach(bill -> {
                billMap.put(bill.getBillCode(), bill);
                dtoMap.get(bill.getBillCode()).forEach(dto -> {
                    dto.setOrgId(bill.getOrgId());
                    dto.setOrgCode(bill.getOrgCode());
                    dto.setOrgName(bill.getOrgName());
                    dto.setBillEntityCode(bill.getBillTypeCode());
                });
            });
        return billMap;
    }

    /**
     * 单据过滤
     *
     * @param dtoMap  单据dtoMap
     * @param billMap 单据Map
     * @return
     */
    private List<Tuple2<Integer, String>> billFilter(Map<String, List<ImageDTO>> dtoMap,
        Map<String, BaseBill> billMap) {
        final List<Tuple2<Integer, String>> billFilter = new ArrayList<>();
        //扫描过滤
        billFilter.add(scanFilter(dtoMap, billMap));
        //封面过滤
        billFilter.add(coverFilter(dtoMap, billMap));
        //单据不存在过滤
        billFilter.add(noExistFilter(dtoMap, billMap));
        //退回未重新推送待办(单据)过滤
        billFilter.add(returnFilter(dtoMap, billMap));
        //作废过滤
        billFilter.add(invalidFilter(dtoMap, billMap));
        //重复过滤
        billFilter.add(repeatFilter(dtoMap, billMap));

        return billFilter;
    }

    /**
     * 扫描过滤项
     *
     * @param dtoMap  单据dtoMap
     * @param billMap 单据Map
     */
    private Tuple2<Integer, String> scanFilter(Map<String, List<ImageDTO>> dtoMap, Map<String, BaseBill> billMap) {
        //重复扫描过滤
        final StringBuilder sb = new StringBuilder();
        final List<String> billCodes = dtoMap.keySet().stream()
            .filter(code -> ValidatorUtil.isNotEmpty(code))
            .filter(code -> code.contains(REPEAT_BILL))
            .collect(Collectors.toList());
        billCodes.forEach(code -> {
            final String repeatBillCode = code.replace(REPEAT_BILL, "");
            sb.append("业务单号").append(repeatBillCode)
                .append("挂接了").append(dtoMap.remove(code).size())
                .append("张影像，与同批次扫描的单据重复;\n");
        });
        if (billCodes.size() > 0) {
            return new Tuple2<>(billCodes.size(), sb.toString());
        }
        return new Tuple2<>(0, "");
    }

    /**
     * 封面过滤
     *
     * @param dtoMap  单据dtoMap
     * @param billMap 单据Map
     * @return
     */
    private Tuple2<Integer, String> coverFilter(Map<String, List<ImageDTO>> dtoMap, Map<String, BaseBill> billMap) {
        //没有缺封面的，直接跳过
        if (!dtoMap.containsKey(StringUtils.EMPTY)) {
            return new Tuple2<>(0, "");
        }
        final int count = dtoMap.remove(StringUtils.EMPTY).size();
        if (count > 0) {
            return new Tuple2<>(0, String.format("共有%s张无封面的影像，不可上传，请检查是否有封面;\n", count));
        }
        return new Tuple2<>(0, "");
    }

    /**
     * 单据不存在过滤
     *
     * @param dtoMap  单据dtoMap
     * @param billMap 单据Map
     * @return
     */
    private Tuple2<Integer, String> noExistFilter(Map<String, List<ImageDTO>> dtoMap, Map<String, BaseBill> billMap) {
        if (ValidatorUtil.isEmpty(dtoMap)) {
            return new Tuple2<>(0, "");
        }
        final List<String> billCodes = dtoMap.keySet().stream()
            .filter(billCode -> ValidatorUtil.isNotEmpty(billCode))
            .collect(Collectors.toList());
        if (ValidatorUtil.isEmpty(billCodes)) {
            return new Tuple2<>(0, "");
        }

        //清理已存在业务单
        billMap.keySet().forEach(billCode -> billCodes.remove(billCode));
        final StringBuilder sb = new StringBuilder();
        billCodes.forEach(billCode -> {
            dtoMap.remove(billCode);
            sb.append("单据号：").append(billCode)
                .append("系统中不存在该单据的待办任务，请核实单据推送后再次尝试扫描/上传;\n");
            billMap.remove(billCode);
        });
        return new Tuple2<>(billCodes.size(), sb.toString());

    }

    /**
     * 作废过滤
     *
     * @param dtoMap  单据dtoMap
     * @param billMap 单据Map
     * @return
     */
    private Tuple2<Integer, String> invalidFilter(Map<String, List<ImageDTO>> dtoMap, Map<String, BaseBill> billMap) {
        if (ValidatorUtil.isEmpty(billMap)) {
            return new Tuple2<>(0, "");
        }
        final List<String> billCodes = billMap.entrySet().stream()
            .filter(entry -> BillDataStatus._5.getCode().equals(entry.getValue().getBillDataStatus()))
            .map(entry -> entry.getKey())
            .collect(Collectors.toList());
        if (ValidatorUtil.isEmpty(billCodes)) {
            return new Tuple2<>(0, "");
        }
        final StringBuilder sb = new StringBuilder();
        //单据号JW131903292，挂接单证10张影像,已作废不可重新扫描;
        billCodes.forEach(billCode -> {
            sb.append(billCodeName()).append(billCode)
                .append("挂接了").append(dtoMap.remove(billCode).size())
                .append("张影像，已作废不可重新扫描;\n");
            billMap.remove(billCode);
        });
        return new Tuple2<>(billCodes.size(), sb.toString());
    }

    /**
     * 重复过滤
     *
     * @param dtoMap  单据dtoMap
     * @param billMap 单据Map
     * @return
     */
    private Tuple2<Integer, String> repeatFilter(Map<String, List<ImageDTO>> dtoMap, Map<String, BaseBill> billMap) {
        if (ValidatorUtil.isEmpty(billMap)) {
            return new Tuple2<>(0, "");
        }
        final List<String> billCodes = billMap.entrySet().stream()
            .filter(entry -> entry.getValue().getImageId() != null && entry.getValue().getImageId() > 0L)
            .map(entry -> entry.getKey())
            .collect(Collectors.toList());
        if (ValidatorUtil.isEmpty(billCodes)) {
            return new Tuple2<>(0, "");
        }
        final StringBuilder sb = new StringBuilder();
        //单据号JW131903292，挂接单证10张,
        //与[扫描批次: 19-1203-0987, 扫描人:张三]重复
        billCodes.forEach(billCode -> {
            final BaseBill billEntity = billMap.get(billCode);
            sb.append("业务单号").append(billCode)
                .append("挂接了").append(dtoMap.remove(billCode).size())
                .append("张影像，与[扫描批次: ").append(billEntity.getBatchNo()).append(", 扫描人:")
                .append(billEntity.getScanUserName()).append("]重复;\n");
            billMap.remove(billCode);
        });
        return new Tuple2<>(billCodes.size(), sb.toString());
    }

    /**
     * 退回过滤
     *
     * @param dtoMap  单据dtoMap
     * @param billMap 单据Map
     * @return
     */
    private Tuple2<Integer, String> returnFilter(Map<String, List<ImageDTO>> dtoMap, Map<String, BaseBill> billMap) {
        if (ValidatorUtil.isEmpty(billMap)) {
            return new Tuple2<>(0, "");
        }
        //isPushBill =1 是需要处理待办不可直接混扫 扫描后重置为0
        final List<String> reUploadBillCodes = billMap.entrySet().stream()
            .filter(entry -> BillDataStatus._4.getCode().equals(entry.getValue().getBillDataStatus()) && YesNo._0.getCode().equals(entry.getValue().getIsPushBill()))
            .filter(entry -> BackType._1.getCode().equals(entry.getValue().getBackType()))
            .map(entry -> entry.getKey())
            .collect(Collectors.toList());

        if (ValidatorUtil.isEmpty(reUploadBillCodes)) {
            return new Tuple2<>(0, "");
        }

        final StringBuilder sb = new StringBuilder();
        //退回重走过滤 提示文案
        reUploadBillCodes.forEach(billCode -> {
            dtoMap.remove(billCode);
            sb.append("单据号：").append(billCode)
                .append("该单据为退回单据，请等待经办人重新提交单据后再次进行扫描。\n");
            billMap.remove(billCode);
        });

        return new Tuple2<>(reUploadBillCodes.size(), sb.toString());
    }

    /**
     * 存储影像
     *
     * @param request 请求
     * @param dtoMap  单据dtoMap
     * @param billMap 单据Map
     * @return
     */
    private List<Long> saveImages(SaveImageRequest request, Map<String, List<ImageDTO>> dtoMap, Map<String, BaseBill> billMap) {
        final IAuthorizedUser user = UserInfoHolder.get();
        final String batchNo = Long.toString(System.currentTimeMillis());
        final List<JSONObject> images = new ArrayList<>();
        final List<Long> ids = new ArrayList<>();
        final Map<String, JSONObject> billImagesMap = new HashMap<>();
        dtoMap.entrySet().stream().forEach(entry -> {
            final BaseBill billEntity = billMap.get(entry.getKey());
            billEntity.setBatchNo(batchNo);
            billEntity.setImageCount((long) entry.getValue().size());
            billEntity.setScanUserId(user.getId());
            billEntity.setScanUserName(user.getUsername());
            billEntity.setCreateUserCode(user.getUserCode());
            final boolean isAllBill = entry.getValue().stream().allMatch(dto -> ImageType._1.getCode().equals(dto.getImageType()));
            updateBill(billEntity, isAllBill);
            entry.getValue().forEach(dto -> {
                final JSONObject image = BeanUtils.convertJSON(dto);
                image.put("batch_no", batchNo);
                image.put("create_user_code", user.getUserCode());
                image.put("is_public", YesNo._1.getCode());
                //单据影像一对多关系
                image.put(ReleationConstants.BILL_OTM_IMAGE_ID, billEntity.getId());
                if (ImageType._1.getCode().equals(dto.getImageType())) {
                    image.put("file_order", "1");
                    if (!billImagesMap.containsKey(image.getString("bill_code"))) {
                        billImagesMap.put(image.getString("bill_code"), image);
                    }
                } else {
                    image.put("file_order", "");
                }
                if (ValidatorUtil.isNotEmpty(request.getExt())) {
                    final JSONObject ticketPreData = new JSONObject(request.getExt());
                    ticketPreData.remove(EntityMeta.Image.ID.code());
                    ticketPreData.remove("token");
                    final JSONObject area = ObjectUtils.firstNonNull(request.getExt().getJSONObject("area"), request.getExt().getJSONObject("areas"));
                    if (ValidatorUtil.isNotEmpty(area)) {
                        ticketPreData.put(EntityMeta.Ticket.AREAS_ID.code(), area.get("orgId"));
                        ticketPreData.put(EntityMeta.Ticket.AREA_ID.code(), area.get("orgId"));
                        ticketPreData.put(EntityMeta.Ticket.AREA_CODE.code(), area.get("orgCode"));
                        ticketPreData.put(EntityMeta.Ticket.AREA_NAME.code(), area.get("orgName"));
                    }
                    ticketPreData.remove("area");
                    ticketPreData.remove("areas");
                    image.put(EntityMeta.Image.TICKET_PRE_DATA.code(), ticketPreData.toJSONString());
                    image.putAll(extFieldsReceiveFacade.getExtFields(user.getTenantId(), ReceiveFilterEnums.IMAGE, request.getExt()));
                }
                images.add(image);
            });
        });
        final EnableFlowContext context = new EnableFlowContext().setTenantId(user.getTenantId()).setUserId(user.getId());
        final boolean flag = CollectionUtils.isNotEmpty(images) && batchService.enable(context);
        LogContext.setLogPoint("enableFlow", flag);
        if (flag) {
            final String channelCode = batchService.getChannelCode(user.getTenantCode());
            final BatchTaskEntity batchTaskEntity = new BatchTaskEntity();
            batchTaskEntity.setTenantId(user.getTenantId());
            batchTaskEntity.setTenantCode(user.getTenantCode());
            batchTaskEntity.setCreateUserId(user.getId());
            batchTaskEntity.setCreateUserName(user.getUsername());
            batchTaskEntity.setCreateTime(new Date());
            batchTaskEntity.setTaskCount(images.size());
            batchTaskEntity.setChannelCode(channelCode);
            final Long batchId = batchService.insert(batchTaskEntity);
            images.forEach(image -> image.put(EntityMeta.Image.BATCH_NO.code(), batchId));
            final List<Image> imageEntities = images.stream().map(image -> BeanUtils.convertObject(image, Image.class)).collect(Collectors.toList());
            final Map<String, Object> headers = new HashMap<>();
            headers.put(com.xforceplus.elephant.image.core.business.consts.Constants.TENANT_CODE, user.getTenantCode());
            headers.put("batchNo", batchId);
            headers.put("userInfo",JSONObject.toJSONString(user));
            headers.put("channelCode",channelCode);
            headers.put("startTime",System.currentTimeMillis());
            try {
                if (batchService.sendEnableFlow(context)) {
                    final Channel channel = application.loadChannel(channelCode);
                    final BizRequest bizRequest = BizRequest.builder()
                        .bizQuantity(batchTaskEntity.getTaskCount())
                        .properties(headers)
                        .body(JSONObject.toJSONString(imageEntities))
                        .attrs(Maps.of("queueName", MQEnum.SAVE_IMAGE_SYNC_QUEUE))
                        .build();
                    channel.flowInRateLimit(bizRequest);
                } else {
                    rabbitmqUtils.sendByDirectExchange(MQEnum.SAVE_IMAGE_SYNC_QUEUE, JSONObject.toJSONString(imageEntities), headers);
                }
            } catch (Exception e) {
                LogContext.setLogPoint("enableFlowError", e.getMessage());
                logger.error("使用限流异常", e);
                LogContext.setLogPoint("error", String.format("使用限流异常:%s", e.getMessage()));
                rabbitmqUtils.sendByDirectExchange(MQEnum.SAVE_IMAGE_SYNC_QUEUE, JSONObject.toJSONString(imageEntities), headers);
            }
            return ids;
        }
        if (images.size() > 0) {
            final Map<String, List<JSONObject>> stringListMap = images.stream().collect(Collectors.groupingBy(r -> r.getString("bill_code")));
            //为了使单证的file_order从下标2开始，故意使第一张单据的file_order重置为空
            stringListMap.entrySet().stream().forEach(r -> {
                r.getValue().stream().filter(q -> q.get("file_order").equals("1")).findFirst().get().put("file_order", "");
            });
            final Map<String, List<JSONObject>> billCodeMap = images.stream().collect(Collectors.groupingBy(r -> r.getString("bill_code")));
            billCodeMap.entrySet().stream().forEach(r -> {
                final List<Long> longs = imageService.saveForMap(r.getValue());
                ids.addAll(longs);
            });
            billCodeMap.entrySet().stream().forEach(r -> {
                r.getValue().stream().filter(q -> q.get("file_order").equals("1")).forEach(q -> {
                    final JSONObject updatejson = new JSONObject();
                    updatejson.put("file_order", r.getValue().get(0).get("file_order"));
                    imageService.updateByParam(q.getLong("id"), updatejson);
                });
            });

        }
        //更新单据影像一对一关系
        final List<JSONObject> billImages = billImagesMap.values().stream().collect(Collectors.toList());
        billImages.forEach(image -> {
            final Long billId = image.getLong(ReleationConstants.BILL_OTM_IMAGE_ID);
            final Long imageId = image.getLong("id");
            final JSONObject bill = new JSONObject();
            bill.put("image_id", imageId);
            bill.put(ReleationConstants.BILL_OTO_IMAGE_ID, imageId);
            baseBillService.updateByBillIdSelective(billId, bill);
            //发送通知
            final JSONObject mqMessage = new JSONObject();
            mqMessage.put("noticeType", WebsocketNoticeTypeEnum.IMAGE_DISCERN.getCode());
            mqMessage.put("tenantId", user.getTenantId());
            mqMessage.put("source", "重药混扫");
            mqMessage.put("createUserId", user.getId());
            mqMessage.put("billCode", image.getString("bill_code"));
            mqMessage.put("isBill", true);
            final Map<String, Object> headers = new HashMap<>();
            headers.put(TENANT_CODE, user.getTenantCode());
            rabbitmqUtils.sendByTopicExchange(MQEnum.WEBSOCKET_NOTICE_QUEUE, mqMessage, headers);
        });
        return ids;
    }

    /**
     * 更新单据
     *
     * @param billEntity billEntity
     */
    private void updateBill(BaseBill billEntity, boolean isAllBill) {
        //单据保存
        final JSONObject update = new JSONObject();
        update.put("is_push_bill", YesNo._0.getCode());
        update.put("image_count", billEntity.getImageCount());
        if (isAllBill) {
            update.put("calculate_status", CalculateStatus._1.getCode());
        }
        update.put("scan_user_id", billEntity.getScanUserId());
        update.put("scan_user_name", billEntity.getScanUserName());
        update.put("scan_create_time", System.currentTimeMillis());
        update.put("create_user_code", billEntity.getCreateUserCode());
        update.put("batch_no", billEntity.getBatchNo());
        update.put("bill_data_status", BillDataStatus._0.getCode());
        logger.info("更新单据，bill_code：【{}】，update：【{}】", billEntity.getBillCode(), update);
        baseBillService.updateBillData(EntityConstant.BASE_BILL, billEntity.getId(), update);
    }

    /**
     * 过滤消息整理
     *
     * @param billCount billCount
     * @return
     */
    private String message(List<Tuple2<Integer, String>> billFilter, long billCount) {
        int count = 0;
        final StringBuilder sb = new StringBuilder();
        for (Tuple2<Integer, String> filter : billFilter) {
            count += filter._1;
            sb.append(filter._2);
        }
        StringBuilder message = null;
        if (count > 0 || sb.length() > 0) {
            final String title = "影像上传提醒";
            message = new StringBuilder("本批次共扫描单据").append(billCount).append("份");
            if (count > 0) {
                message.append("，其中").append(count).append("份").append("被拦截");
            }
            message.append("：\n").append(sb);
            notificationUtils.sendSingleMessage(UserInfoHolder.get().getTenantId(),
                Arrays.asList(UserInfoHolder.get().getId()), title, message.toString());
        }
        return message != null ? message.toString() : "成功";
    }

}