package com.xforceplus.elephant.image.controller.batchnumberrule.process;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.xforceplus.elephant.basecommon.process.AbstractProcess;
import com.xforceplus.elephant.basecommon.process.request.BaseRequest;
import com.xforceplus.elephant.basecommon.vaildate.ValidatorUtil;
import com.xforceplus.elephant.image.client.model.BatchNumberRuleDefineDTO;
import com.xforceplus.elephant.image.client.model.GetSerialNumberRequest;
import com.xforceplus.elephant.image.core.business.enums.BatchNumberJsonEnum;
import com.xforceplus.elephant.image.core.facade.application.calculate.batchnumberrule.BatchNumberRuleFacade;
import com.xforceplus.elephant.image.core.repository.model.BatchNumberRuleEntity;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.dict.UseStatus;
import java.util.concurrent.atomic.AtomicInteger;
import javax.validation.ValidationException;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;

/**
 * 批次号规则抽象类 Datetime:    2021/6/11   14:30 Author:   cmq
 */
@Slf4j
public abstract class AbstractBatchNumberRuleProcess<ReqT extends BaseRequest, RepT> extends AbstractProcess<ReqT, RepT> {

    @Autowired
    protected BatchNumberRuleFacade batchNumberRuleFacade;

    protected static final String RULE_REDIS_KEY_PRE = "NUMBER_RULE";

    /**
     * 校验规则定义json 规则定义json没有做完全校验，留一点灵活性
     *
     * @param ruleDefine 规则定义json
     * @return void
     * @author cmq
     * @Datetime 2021/6/11 15:06
     */
    protected void checkRuleDefine(String ruleDefine) {
        try {
            final JSONArray jSONArray = JSON.parseArray(ruleDefine);
            if (jSONArray.size() == 0) {
                throw new ValidationException("规则定义不能为空");
            }
            jSONArray.stream().forEach(jsonObject -> {
                final JSONObject jsonObjectRule = ((JSONObject) jsonObject);
                if (!jsonObjectRule.containsKey(BatchNumberJsonEnum.COMMON_TYPE.getCode())) {
                    throw new ValidationException("规则定义格式不正确");
                }
                final String type = jsonObjectRule.getString(BatchNumberJsonEnum.COMMON_TYPE.getCode());

                if (BatchNumberJsonEnum.TYPE_DATE.getCode().equals(type)) {
                    if (!jsonObjectRule.containsKey(BatchNumberJsonEnum.DATE_FORMAT.getCode()) || !jsonObjectRule.containsKey(BatchNumberJsonEnum.COMMON_MODIFIABLE.getCode())) {
                        throw new ValidationException("规则定义日期格式不正确");
                    }
                } else if (BatchNumberJsonEnum.TYPE_ENUMS.getCode().equals(type)) {
                    if (!jsonObjectRule.containsKey(BatchNumberJsonEnum.ENUMS_VALUE.getCode()) || !jsonObjectRule.containsKey(BatchNumberJsonEnum.COMMON_MODIFIABLE.getCode())) {
                        throw new ValidationException("规则定义枚举格式不正确");
                    }

                } else if (BatchNumberJsonEnum.TYPE_SERIALNUMBER.getCode().equals(type)) {
                    if (!jsonObjectRule.containsKey(BatchNumberJsonEnum.SERIALNUMBER_LENGTH.getCode()) || !jsonObjectRule.containsKey(BatchNumberJsonEnum.COMMON_MODIFIABLE.getCode())) {
                        throw new ValidationException("规则定义流水号格式不正确");
                    }
                }
            });
        } catch (Exception e) {
            log.error("规则定义格式异常", e);
            throw new ValidationException("规则定义格式不正确");
        }

    }

    /**
     * 获取批次号规则
     *
     * @param ruleCode 规则代码
     * @param tenant   租户
     * @return com.xforceplus.elephant.image.core.repository.model.BatchNumberRuleEntity
     * @author cmq
     * @Datetime 2021/6/15 10:40
     */
    protected BatchNumberRuleEntity getBatchNumberRule(String ruleCode, Long tenant) {
        //校验放在此处，主要是序列号规则对象是一定要查询的，减少重复查询
        final BatchNumberRuleEntity batchNumberRuleEntity = batchNumberRuleFacade.selectByCode(ruleCode, tenant);
        if (batchNumberRuleEntity == null) {
            throw new ValidationException("批次号规则不存在");
        }
        return batchNumberRuleEntity;
    }

    /**
     * 格式化序列号（在序列号前面拼接0)
     *
     * @param serialNumber serialNumber
     * @param length       length
     * @return java.lang.String
     * @author cmq
     * @Datetime 2021/6/15 10:47
     */
    protected String appendSerialNumberPre(String serialNumber, int length) {
        final StringBuffer stringBuffer = new StringBuffer();
        if (ValidatorUtil.isNotEmpty(serialNumber) && length > serialNumber.length()) {
            for (int i = 0; i < length - serialNumber.length(); i++) {
                stringBuffer.append("0");
            }
        }
        stringBuffer.append(serialNumber);
        return stringBuffer.toString();
    }

    /**
     * 解析规则定义
     *
     * @param request    获取流水号数据Request
     * @param ruleDefine 规则定义json字符串
     * @param rulePre    序列号规则定义前缀
     * @return void
     * @author cmq
     * @Datetime 2021/6/15 11:04
     */
    protected BatchNumberRuleDefineDTO analysisRuleDefine(GetSerialNumberRequest request, String ruleDefine, StringBuffer rulePre) {
        final BatchNumberRuleDefineDTO batchNumberRuleDefineDTO = new BatchNumberRuleDefineDTO();
        try {
            final JSONArray jSONArray = JSON.parseArray(ruleDefine);
            final String[] enums = ValidatorUtil.isNotEmpty(request.getEnumeration()) ? request.getEnumeration().split(";") : null;
            final AtomicInteger atomicInteger = new AtomicInteger(0);
            final Integer[] length = {0};

            for (int i = 0; i < jSONArray.size(); i++) {
                final JSONObject jsonObject = jSONArray.getJSONObject(i);
                final String type = ((JSONObject) jsonObject).getString(BatchNumberJsonEnum.COMMON_TYPE.getCode());
                if (BatchNumberJsonEnum.TYPE_DATE.getCode().equals(type)) {
                    if (ValidatorUtil.isEmpty(request.getDate())) {
                        throw new ValidationException("日期值不能为空");
                    }
                    rulePre.append(request.getDate());
                } else if (BatchNumberJsonEnum.TYPE_ENUMS.getCode().equals(type)) {
                    if (ValidatorUtil.isEmpty(request.getEnumeration())) {
                        throw new ValidationException("枚举值不能为空");
                    }
                    rulePre.append(enums[atomicInteger.get()]);
                    atomicInteger.addAndGet(1);

                } else if (BatchNumberJsonEnum.TYPE_SERIALNUMBER.getCode().equals(type)) {
                    final String tmpAutoIncrement = jsonObject.getString(BatchNumberJsonEnum.COMMON_AUTOINCREMENTABLE.getCode());
                    batchNumberRuleDefineDTO.setAutoIncrement(tmpAutoIncrement);
                } else {
                    length[0] = ((JSONObject) jsonObject).getInteger(BatchNumberJsonEnum.SERIALNUMBER_LENGTH.getCode());
                }
            }
            batchNumberRuleDefineDTO.setLength(length);
        } catch (Exception e) {
            log.error("规则定义解析异常", e);
            if (e instanceof ValidationException) {
                throw e;
            }
            throw new ValidationException("规则定义格式不正确");
        }
        return batchNumberRuleDefineDTO;
    }

    /**
     * 校验批次号规则是否可用
     *
     * @param batchNumberRuleEntity batchNumberRuleEntity
     * @return void
     * @author cmq
     * @Datetime 2021/6/15 11:31
     */
    protected void checkAvailable(BatchNumberRuleEntity batchNumberRuleEntity) {
        if (batchNumberRuleEntity == null) {
            throw new ValidationException("未找到批次号规则");
        }
        if (batchNumberRuleEntity.getStatus() == null || !UseStatus._1.getCode().equals(String.valueOf(batchNumberRuleEntity.getStatus()))) {
            throw new ValidationException("批次号规则未启用");
        }

    }

    /**
     * 枚举值老值存取redis键
     *
     * @param ruleCode 规则代码
     * @param tenant   租户
     * @return
     */
    protected String getRuleRedisKey(String ruleCode, Long tenant) {
        return RULE_REDIS_KEY_PRE + "_" + ruleCode + "_" + tenant;
    }

}
