package com.xforceplus.elephant.image.controller.ticket.process;

import com.xforceplus.elephant.basecommon.process.AbstractProcess;
import com.xforceplus.elephant.basecommon.process.response.CommonResponse;
import com.xforceplus.elephant.basecommon.vaildate.ValidatorUtil;
import com.xforceplus.elephant.image.client.model.GetBaseTicketListRequest;
import com.xforceplus.elephant.image.core.business.application.collect.image.service.ImageService;
import com.xforceplus.elephant.image.core.business.application.collect.ticket.service.TicketService;
import com.xforceplus.elephant.image.core.util.RequestBuilder;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.entity.Image;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.meta.EntityMeta;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.meta.EntityMeta.Ticket;
import com.xforceplus.ultraman.metadata.domain.vo.dto.ConditionOp;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;
import java.util.stream.Collectors;
import javax.validation.ValidationException;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

/**
 * 获取基础单证列表
 *
 * @author rongying
 * @date 2021/6/29 11:14
 */
@Slf4j
@Service
public class GetBaseTicketListProcess extends AbstractProcess<GetBaseTicketListRequest, List<Map<String, Object>>> {

    @Autowired
    private TicketService ticketService;
    @Autowired
    private ImageService imageService;

    @Override
    protected void check(GetBaseTicketListRequest request) throws ValidationException {
        super.check(request);
        if (ValidatorUtil.isEmpty(request.getBillCode())) {
            throw new ValidationException("billCode不能为空");
        }
    }

    @Override
    protected CommonResponse<List<Map<String, Object>>> process(GetBaseTicketListRequest request) throws RuntimeException {
        final IAuthorizedUser authorizedUser = UserInfoHolder.get();// 获取登录用户上下文
        if (null == authorizedUser) {
            throw new ValidationException("获取用户信息为空，请重新登陆");
        }
        final Long tenantId = authorizedUser.getTenantId();
        final RequestBuilder requestBuilder = new RequestBuilder()
            .field(EntityMeta.Ticket.BILL_CODE.code(), ConditionOp.eq, request.getBillCode())
            .sort(EntityMeta.Ticket.SCAN_TIME.code(), "desc");
        if (StringUtils.isNotBlank(request.getInvoiceNo())) {
            requestBuilder.field(EntityMeta.Ticket.INVOICE_NO.code(), ConditionOp.eq, request.getInvoiceNo());
        }
        if (StringUtils.isNotBlank(request.getInvoiceCode())) {
            requestBuilder.field(EntityMeta.Ticket.INVOICE_CODE.code(), ConditionOp.eq, request.getInvoiceCode());
        }
        if (CollectionUtils.isNotEmpty(request.getInvoiceTypes())) {
            requestBuilder.field(EntityMeta.Ticket.INVOICE_TYPE.code(), ConditionOp.in, request.getInvoiceTypes());
        }
        if (StringUtils.isNotBlank(request.getAmountWithTaxStart())) {
            requestBuilder.field(Ticket.AMOUNT_WITH_TAX.code(), ConditionOp.ge, request.getAmountWithTaxStart());
        }
        if (StringUtils.isNotBlank(request.getAmountWithTaxEnd())) {
            requestBuilder.field(Ticket.AMOUNT_WITH_TAX.code(), ConditionOp.le, request.getAmountWithTaxEnd());
        }
        if (Objects.nonNull(request.getInvoiceDateStart())) {
            requestBuilder.field(Ticket.INVOICE_DATE.code(), ConditionOp.ge, request.getInvoiceDateStart());
        }
        if (Objects.nonNull(request.getInvoiceDateEnd())) {
            requestBuilder.field(Ticket.INVOICE_DATE.code(), ConditionOp.le, request.getInvoiceDateEnd());
        }
        //查询单证
        final List<Map<String, Object>> ticketList = ticketService.selectTicketMapByParams(tenantId, requestBuilder, EntityMeta.Ticket.code());
        if (CollectionUtils.isNotEmpty(ticketList)) {
            //查询影像
            final List<Long> imageIds = ticketList.stream().filter(r -> ValidatorUtil.isNotEmpty(r.get(EntityMeta.Ticket.IMAGE_ID.code())))
                .map(r -> Long.valueOf(r.get(EntityMeta.Ticket.IMAGE_ID.code()).toString())).collect(Collectors.toList());
            final List<Image> imageList = imageService.selectImages(tenantId, imageIds, EntityMeta.Image.code());
            if (CollectionUtils.isNotEmpty(imageList)) {
                //填充影像地址
                final Map<Long, Image> imageMap = imageList.stream().collect(Collectors.toMap(Image::getId, Function.identity()));
                ticketList.forEach(r -> {
                    final Long imageId = Long.valueOf(r.get(EntityMeta.Ticket.IMAGE_ID.code()).toString());
                    if (ValidatorUtil.isNotEmpty(imageMap) && imageMap.get(imageId) != null) {
                        r.put("image_url", imageMap.get(imageId).getFileUrlHandle());
                        r.put("file_name", imageMap.get(imageId).getFileName());
                    }
                });
            }
        }
        return CommonResponse.ok("成功", ticketList);
    }

}