package com.xforceplus.elephant.image.controller;

import com.google.common.net.MediaType;
import com.xforceplus.elephant.basecommon.help.RedisUtils;
import com.xforceplus.elephant.basecommon.process.response.CommonResponse;
import com.xforceplus.elephant.basecommon.vaildate.ValidatorUtil;
import com.xforceplus.elephant.image.core.business.consts.RedisKey;
import com.xforceplus.tenant.security.autoscan.annotation.AuthorizedDefinition;
import java.security.MessageDigest;
import javax.xml.bind.DatatypeConverter;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.compress.utils.IOUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.ClassPathResource;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.CrossOrigin;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.servlet.ModelAndView;

@Slf4j
@RestController
@RequiredArgsConstructor
public class StaticController extends BaseController {

    private final RedisUtils redisUtils;
    /**
     * 短链接有效时间，秒
     */
    @Value("${image.shotUrlEffectiveTime:1800}")
    private long shotUrlEffectiveTime;
    /**
     * 影像服务地址
     */
    @Value("${image.serviceUrl}")
    private String serviceUrl;

    @AuthorizedDefinition(authorization = false)
    @RequestMapping("/{svg}.svg")
    @CrossOrigin
    public ResponseEntity<byte[]> svg(@PathVariable("svg") String svg) throws Exception {
        final ClassPathResource resource = new ClassPathResource("static/" + svg + ".svg");
        final byte[] bytes = IOUtils.toByteArray(resource.getInputStream());
        return ResponseEntity.ok(bytes);
    }

    @AuthorizedDefinition(authorization = false)
    @PostMapping("/shortUrl")
    @CrossOrigin
    public CommonResponse<String> shortUrl(@RequestBody String url) throws Exception {
        final String md5 = jdkMD5(url);
        final String key = String.format(RedisKey.SHORT_URL_KEY, md5);
        final boolean flag = redisUtils.set(key, url, shotUrlEffectiveTime);
        if (flag) {
            final String shortUrl = String.format("%s/url/%s", serviceUrl, md5);
            return CommonResponse.ok("成功", shortUrl);
        }
        return CommonResponse.failed("申请失败，请重新申请！");
    }

    @AuthorizedDefinition(authorization = false)
    @GetMapping("/url/{key}")
    @CrossOrigin
    public ModelAndView redirect(@PathVariable("key") String key) throws Exception {
        final String shortKey = String.format(RedisKey.SHORT_URL_KEY, key);
        final String url = (String) redisUtils.get(shortKey);
        if (ValidatorUtil.isNotEmpty(url)) {
            return new ModelAndView("redirect:" + url);
        }
        final ModelAndView modelAndView = new ModelAndView();
        modelAndView.setStatus(HttpStatus.BAD_REQUEST);
        modelAndView.setView((map, request, response) -> {
            response.setContentType(MediaType.HTML_UTF_8.toString());
            final ClassPathResource resource = new ClassPathResource("static/short_error.html");
            final String shortError = new String(IOUtils.toByteArray(resource.getInputStream()), "utf-8");
            response.getWriter().println(shortError);
        });
        return modelAndView;
    }

    private static String jdkMD5(String src) {
        try {
            final MessageDigest messageDigest = MessageDigest.getInstance("MD5");
            final byte[] mdBytes = messageDigest.digest(src.getBytes());
            return DatatypeConverter.printHexBinary(mdBytes);
        } catch (Exception e) {
            log.error("md5加密失败", e);
        }
        return null;
    }

}
