package com.xforceplus.elephant.image.controller.image.process;

import com.xforceplus.elephant.basecommon.exception.ElephantException;
import com.xforceplus.elephant.basecommon.process.AbstractProcess;
import com.xforceplus.elephant.basecommon.process.response.CommonResponse;
import com.xforceplus.elephant.image.client.model.CutImageRequest;
import com.xforceplus.elephant.image.core.business.enums.FileTypeEnum;
import com.xforceplus.elephant.image.core.business.infrastructure.port.FileService;
import com.xforceplus.elephant.image.core.facade.base.BaseFacade;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.dict.ExceptionStatus;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.dict.RecStatus;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.dict.YesNo;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.meta.EntityMeta;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.Base64;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.validation.ValidationException;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import net.coobird.thumbnailator.Thumbnails;
import org.springframework.stereotype.Service;

/**
 * 切割并创建可识别影像
 *
 * @author liuhongbin
 * @date 2021/6/22 15:48
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class CutImageProcess extends AbstractProcess<CutImageRequest, Long> {

    public static final Pattern IMAGE_TYPE = Pattern.compile("data:image/(\\w+);base64,");
    private final BaseFacade baseFacade;
    private final FileService fileService;

    @Override
    protected void check(CutImageRequest request) throws ValidationException {
        super.check(request);
        checkEmpty(request.getImageStream(), "影像流不能为空");
        checkEmpty(request.getCutImageId(), "被切割影像不能为空");
        if (!IMAGE_TYPE.matcher(request.getImageStream()).find()) {
            throw new ValidationException("影像流不是base64，无法解析");
        }
    }

    @Override
    protected CommonResponse<Long> process(CutImageRequest request) throws RuntimeException {
        final IAuthorizedUser user = UserInfoHolder.get();
        final Map<String, Object> image = baseFacade.findOne(EntityMeta.Image.code(), request.getCutImageId());
        final Matcher matcher = IMAGE_TYPE.matcher(request.getImageStream());
        String fileExt = FileTypeEnum.JPG.getCode();
        if (matcher.find()) {
            fileExt = "." + matcher.group(1);
        }
        final ByteArrayOutputStream baos = new ByteArrayOutputStream();
        try {
            final String imageContent = request.getImageStream().split(",")[1];
            final byte[] imgBytes = Base64.getDecoder().decode(imageContent);
            Thumbnails.of(new ByteArrayInputStream(imgBytes)).scale(1).outputQuality(0.5).toOutputStream(baos);
        } catch (IOException e) {
            throw new ElephantException("图片处理异常");
        }
        final String fileName = new StringBuilder("image_").append(user.getId())
            .append("_").append(System.currentTimeMillis())
            .append(fileExt).toString();
        final Long fileKey = fileService.uploadFile(user.getTenantId(), user.getId(), fileName, new ByteArrayInputStream(baos.toByteArray()));
        final String url = fileService.getFileUrl(user.getTenantId(), user.getId(), fileKey);
        final String fileOrder = (String) image.get(EntityMeta.Image.FILE_ORDER.code());
        final Map<String, Object> update = new HashMap<>(1);
        update.put(EntityMeta.Image.FILE_ORDER.code(), fileOrder + "0");
        baseFacade.updateById(EntityMeta.Image.code(), request.getCutImageId(), update);
        image.remove(EntityMeta.Image.ID.code());
        image.remove(EntityMeta.Image.IMAGE_TYPE.code());
        image.put(EntityMeta.Image.FILE_ORDER.code(), fileOrder + "1");
        image.put(EntityMeta.Image.EXCEPTION_STATUS.code(), ExceptionStatus._0.getCode());
        image.put(EntityMeta.Image.WARNING_INFO.code(), "");
        image.put(EntityMeta.Image.EXCEPTION_INFO.code(), "");
        image.put(EntityMeta.Image.FILE_URL_HANDLE.code(), url);
        image.put(EntityMeta.Image.FILE_TYPE_HANDLE.code(), fileExt);
        image.put(EntityMeta.Image.REQUIRE_OCR_FLAG.code(), YesNo._1.getCode());
        image.put(EntityMeta.Image.REC_STATUS.code(), RecStatus._0.getCode());
        image.put(EntityMeta.Image.BATCH_NO.code(), System.currentTimeMillis());
        final Long imageId = baseFacade.create(EntityMeta.Image.code(), image);
        return CommonResponse.ok("成功", imageId);
    }

}
