package com.xforceplus.elephant.image.controller.devops.process;

import com.alibaba.fastjson.JSON;
import com.xforceplus.elephant.basecommon.process.AbstractProcess;
import com.xforceplus.elephant.basecommon.process.response.CommonResponse;
import com.xforceplus.elephant.basecommon.vaildate.ValidatorUtil;
import com.xforceplus.elephant.image.client.model.RestoreHookTicketRequest;
import com.xforceplus.elephant.image.core.business.application.collect.bill.service.BillService;
import com.xforceplus.elephant.image.core.business.application.collect.image.service.ImageService;
import com.xforceplus.elephant.image.core.business.application.collect.ticket.service.TicketService;
import com.xforceplus.elephant.image.core.util.RequestBuilder;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.dict.YesNo;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.entity.BaseBill;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.entity.Ticket;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.meta.EntityMeta;
import com.xforceplus.ultraman.metadata.domain.record.EmptyValue;
import com.xforceplus.ultraman.metadata.domain.vo.dto.ConditionOp;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import javax.validation.ValidationException;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;

@Slf4j
@Service
@RequiredArgsConstructor
public class RestoreHookTicketProcess extends AbstractProcess<RestoreHookTicketRequest, Object> {

    private final ImageService imageService;
    private final TicketService ticketService;
    private final RedisTemplate redisTemplate;
    private final BillService billService;

    @Override
    protected void check(RestoreHookTicketRequest request) throws ValidationException {
        super.check(request);
        checkEmpty(request.getTenantId(), "租户Id不能为空");
        checkAllEmpty("单据号和影像id不能都为空", request.getBillCodes(), request.getImageIds());
    }

    @Override
    protected CommonResponse<Object> process(RestoreHookTicketRequest request) throws RuntimeException {
        log.info("接收到还原请求：{}", JSON.toJSONString(request));
        final StringBuilder message = new StringBuilder();
        message.append("接收到请求，tenantId:").append(request.getTenantId()).append(", billCodes:")
            .append(request.getBillCodes()).append(", imageIds:").append(request.getImageIds()).append(";");
        final RequestBuilder ticketRequestBuilder = new RequestBuilder();
        if (StringUtils.isNotBlank(request.getBillCodes())) {
            ticketRequestBuilder.field(EntityMeta.Ticket.BILL_CODE.code(), ConditionOp.in, request.getBillCodes().split(","));
        }
        if (StringUtils.isNotBlank(request.getImageIds())) {
            ticketRequestBuilder.field(EntityMeta.Ticket.IMAGE_ID.code(), ConditionOp.in, request.getImageIds().split(","));
        }
        final Set<String> billCodes = new HashSet<>();
        final List<Ticket> tickets = ticketService.selectTicketByParams(request.getTenantId(), ticketRequestBuilder, EntityMeta.Ticket.code());
        if (CollectionUtils.isEmpty(tickets)) {
            message.append("无法查询到可操作数据。");
            return CommonResponse.failed(message.toString());
        }
        tickets.forEach(ticket -> billCodes.add(ticket.getBillCode()));
        //提取主票imageId
        final List<Long> mainImageIds = tickets.stream().map(ticket -> ticket.getMainImageId() != null && ticket.getMainImageId() > 0 ? ticket.getMainImageId() : ticket.getImageId())
            .distinct().collect(Collectors.toList());
        message.append("已查询到主ImageId：").append(mainImageIds).append(";");
        //搜索复制票并删除
        final RequestBuilder copyRequestBuilder = new RequestBuilder()
            .field(EntityMeta.Ticket.MAIN_IMAGE_ID.code(), ConditionOp.in, mainImageIds);
        final List<Ticket> copyTickets = ticketService.selectTicketByParams(request.getTenantId(), copyRequestBuilder, EntityMeta.Ticket.code());
        if (CollectionUtils.isNotEmpty(copyTickets)) {
            final List<Long> copyTicketIds = copyTickets.stream().map(Ticket::getId).collect(Collectors.toList());
            final List<Long> copyImageIds = copyTickets.stream().map(Ticket::getImageId).collect(Collectors.toList());
            imageService.deleteMulti(copyImageIds, EntityMeta.Image.code());
            ticketService.deleteMulti(copyTicketIds, EntityMeta.Ticket.code());
            message.append("清理复制发票：【");
            copyTickets.forEach(
                ticket -> message.append("单据号：").append(ticket.getBillCode()).append("发票代码：").append(ticket.getInvoiceCode()).append(",发票号码：").append(ticket.getInvoiceNo()).append(";"));
            message.append("】;");
            copyTickets.forEach(ticket -> billCodes.add(ticket.getBillCode()));
        }
        //将主票的单据信息和影像的单据信息清空
        final Map<String, Object> imageUpdate = new HashMap<>();
        imageUpdate.put(EntityMeta.Image.BILL_CODE.code(), StringUtils.EMPTY);
        final int imageCount = imageService.updateMulti(mainImageIds, imageUpdate, EntityMeta.Image.code());
        final List<Ticket> mainTickets = ticketService.selectBaseTicketByImageIds(request.getTenantId(), mainImageIds, EntityMeta.Ticket.code());
        if (CollectionUtils.isEmpty(mainTickets)) {
            return CommonResponse.failed("无法查询到可操作数据");
        }
        final List<Map<String, Object>> ticketUpdates = mainTickets.stream().map(ticket -> {
            final Map<String, Object> ticketUpdate = new HashMap<>();
            ticketUpdate.put(EntityMeta.Ticket.ID.code(), ticket.getId());
            ticketUpdate.put(EntityMeta.Ticket.BILL_CODE.code(), StringUtils.EMPTY);
            ticketUpdate.put(EntityMeta.Ticket.IS_HOOKED.code(), YesNo._0.getCode());
            ticketUpdate.put(EntityMeta.Ticket.HOOK_TIME.code(), EmptyValue.emptyValue);
            ticketUpdate.put(EntityMeta.Ticket.BILL_SYSTEM_SOURCE.code(), EmptyValue.emptyValue);
            ticketUpdate.put(EntityMeta.Ticket.BILL_O_T_M_TICKET_ID.code(), 0L);
            if (EntityMeta.TicketInvoice.code().equals(ticket.getTicketCode())) {
                ticketUpdate.put(EntityMeta.Ticket.USED_AMOUNT.code(), 0);
                ticketUpdate.put(EntityMeta.Ticket.BALANCE_AMOUNT.code(), ticket.getAmountWithTax());
            }
            return ticketUpdate;
        }).collect(Collectors.toList());
        final int ticketCount = ticketService.updateMulti(ticketUpdates, EntityMeta.Ticket.code());
        message.append("还原发票：【");
        mainTickets.forEach(
            ticket -> message.append("单据号：").append(ticket.getBillCode()).append("发票代码：").append(ticket.getInvoiceCode()).append(",发票号码：").append(ticket.getInvoiceNo()).append(";"));
        message.append("】;");
        //清理缓存
        final List<String> balanceAmountKeys = mainImageIds.stream().map(id -> String.format("imageIdForBalanceAmount:%s", id)).collect(Collectors.toList());
        redisTemplate.delete(balanceAmountKeys);
        mainImageIds.forEach(id -> {
            final String usedAmountKey = String.format("imageIdForUsedAmount:%s_*", id);
            final Set<String> keys = redisTemplate.keys(usedAmountKey);
            if (!keys.isEmpty()) {
                redisTemplate.delete(keys);
            }
        });
        mainTickets.forEach(ticket -> billCodes.add(ticket.getBillCode()));
        billCodes.remove(StringUtils.EMPTY);
        if (ValidatorUtil.isNotEmpty(billCodes)) {
            final List<BaseBill> bills = billService.selectBillDataByBillCodes(request.getTenantId(), new ArrayList<>(billCodes), EntityMeta.BaseBill.code());
            if (CollectionUtils.isNotEmpty(bills)) {
                final Map<String, Object> billUpdate = new HashMap<>();
                billUpdate.put(EntityMeta.BaseBill.IMAGE_COUNT.code(), 0);
                billUpdate.put(EntityMeta.BaseBill.TICKET_COUNT.code(), 0);
                billUpdate.put(EntityMeta.BaseBill.ELECTRIC_COUNT.code(), 0);
                billUpdate.put(EntityMeta.BaseBill.PAPER_COUNT.code(), 0);
                bills.forEach(bill -> billService.updateBillByParam(bill.getId(), billUpdate, EntityMeta.BaseBill.code()));
                message.append("重置单据：").append(billCodes).append("影像发票数量为0；");
            }
        }

        message.append(String.format("还原影像%s个，还原单证%s个。", imageCount, ticketCount));
        return CommonResponse.ok(message.toString());
    }

}
