package com.xforceplus.elephant.image.controller.image.process.delete;

import com.alibaba.fastjson.JSONObject;
import com.xforceplus.elephant.basecommon.annotation.Dispatch;
import com.xforceplus.elephant.basecommon.log.MyThreadLocal;
import com.xforceplus.elephant.basecommon.process.AbstractProcess;
import com.xforceplus.elephant.basecommon.process.response.CommonResponse;
import com.xforceplus.elephant.basecommon.vaildate.ValidatorUtil;
import com.xforceplus.elephant.image.client.model.BillSelectAllRequest;
import com.xforceplus.elephant.image.core.business.enums.ResourceEnum;
import com.xforceplus.elephant.image.core.domain.bill.BaseBillService;
import com.xforceplus.elephant.image.core.domain.image.ImageService;
import com.xforceplus.elephant.image.core.domain.image.bean.ImageSearchBean;
import com.xforceplus.elephant.image.core.expand.BillImageTicketService;
import com.xforceplus.elephant.image.core.facade.application.auth.DataResourceFacade;
import com.xforceplus.elephant.image.core.facade.application.collect.image.ImageFacade;
import com.xforceplus.elephant.image.mapper.ImageSearchMapper;
import com.xforceplus.tech.base.core.context.ContextKeys;
import com.xforceplus.tech.base.core.context.ContextService;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.dict.BillDataStatus;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.dict.ImageType;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.entity.BaseBill;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.entity.Image;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.meta.EntityMeta;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.stream.Collectors;
import javax.validation.ValidationException;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

@Dispatch(isDefault = true)
@Service
public class DefaultDeleteImageProcess extends AbstractProcess<BillSelectAllRequest, Boolean> {

    @Autowired
    private BillImageTicketService billImageTicketService;
    @Autowired
    private ImageFacade imageFacade;
    @Autowired
    private ImageSearchMapper mapper;
    @Autowired
    private ImageService imageService;
    @Autowired
    private BaseBillService baseBillService;
    @Autowired
    private DataResourceFacade dataResourceFacade;
    @Autowired
    private ExecutorService imageThreadPool;
    @Autowired
    private ContextService contextService;

    @Override
    protected void check(final BillSelectAllRequest request) throws ValidationException {
        super.check(request);
        if (request.isAllSelected()) {
            checkEmpty(request.getBillCode(), "billCode参数不能为空");
        }
        if (!request.isAllSelected()) {
            checkEmpty(request.getIncluded(), "影像id不能为空");
        }
    }

    /**
     * 准备要操作的数据 格式化全选反选数据到ids 需要注意此处是要考虑全选操作是带着筛选条件的
     *
     * @param request 请求
     * @return void
     * @author liming
     * @date 2021/7/30 11:18
     */
    protected void prepareOperateData(final BillSelectAllRequest request) {
        //全选
        final IAuthorizedUser user = UserInfoHolder.get();
        if (request.isAllSelected()) {
            //条件筛选
            final ImageSearchBean searchBean = mapper.map(request);
            searchBean.setTenantId(user.getTenantId());
            final List<Image> imageList = imageFacade.selectBillImagesByCustomTicketParams(searchBean, EntityMeta.Image.code()).getRows();
            final List<Long> ids = imageList.stream()
                                            .filter(image -> !ImageType._1.getCode().equals(image.getImageType()))
                                            .map(image -> image.getId())
                                            .collect(Collectors.toList());
            if (ValidatorUtil.isNotEmpty(request.getExcluded())) {
                final List<Long> excludedList = request.getExcluded().stream().map(s -> Long.parseLong(s)).collect(Collectors.toList());
                ids.removeAll(excludedList);
            }
            request.setIds(ids);
        }
        //非全选
        if (!request.isAllSelected()) {
            final List<Long> ids = request.getIncluded().stream().map(s -> Long.parseLong(s)).collect(Collectors.toList());
            request.setIds(ids);
        }
    }

    @Override
    protected CommonResponse<Boolean> process(final BillSelectAllRequest request) throws RuntimeException {
        final IAuthorizedUser authorizedUser = UserInfoHolder.get();
        if (null == authorizedUser) {
            throw new ValidationException("获取用户信息为空，请重新登陆");
        }
        prepareOperateData(request);

        final String number = StringUtils.firstNonBlank(request.getBillCode(), String.valueOf(authorizedUser.getId()));
        MyThreadLocal.setObject(authorizedUser.getTenantId(), number, request.getBillCode());

        BaseBill billEntity = new BaseBill();
        //数据权限过滤
        if (ValidatorUtil.isNotEmpty(request.getBillCode())) {
            billEntity = baseBillService.selectBaseBillByCode(authorizedUser.getTenantId(), request.getBillCode());
            if (billEntity != null && Arrays.asList(BillDataStatus._3.getCode(), BillDataStatus._1.getCode()).contains(billEntity.getBillDataStatus())) {
                throw new ValidationException("单据已锁定/已完成状态不可删除影像");
            }
            if (!dataResourceFacade.existBillDetailResourceAuth(authorizedUser.getTenantId(), request.getPageCode(), request.getScene(), billEntity, ResourceEnum.BILL_DETAILS_DELETE.getCode())) {
                throw new ValidationException("没有操作权限");
            }
        }
        if (ValidatorUtil.isEmpty(request.getIds())) {
            return CommonResponse.failed("未查询到发票数据");
        }
        final String message = filter(request);
        if (ValidatorUtil.isEmpty(request.getIds())) {
            return CommonResponse.failed(message);
        }
        final List<Image> images = imageService.selectImages(authorizedUser.getTenantId(), request.getIds());
        final List<String> billCodes = images.stream().map(Image::getBillCode).collect(Collectors.toList());
        final boolean result = billImageTicketService.deleteImageAndTicketByImageIds(authorizedUser.getTenantId(), request.getIds());
        if (result) {
            //JXCPX-2504【碧桂园物业】影像待办上传发票后，报错：单据异常提交失败。
            final String tenanyCode = authorizedUser.getTenantCode();
            CompletableFuture.runAsync(() -> {
                try {
                    contextService.set(ContextKeys.StringKeys.TENANTCODE_KEY, tenanyCode);
                    billCodes.stream().filter(billCode -> imageService.noFinishRecImageCount(authorizedUser.getTenantId(), billCode) > 0)
                             .forEach(billCode -> billImageTicketService.resetLevel(authorizedUser.getTenantId(), billCode));
                } finally {
                    contextService.clear();
                }
            }, imageThreadPool);
            //影像删成功,修改单据状态为待处理
            if (BillDataStatus.fromCode(billEntity.getBillDataStatus()) == BillDataStatus._6) {
                final JSONObject update = new JSONObject();
                update.put(EntityMeta.BaseBill.BILL_DATA_STATUS.code(), BillDataStatus._0.getCode());
                baseBillService.updateByBillIdSelective(billEntity.getId(), update);
            }

            return CommonResponse.ok(message, result);
        }
        return CommonResponse.failed(message);
    }

    protected String filter(final BillSelectAllRequest request) {
        final IAuthorizedUser authorizedUser = UserInfoHolder.get();
        final List<Image> images = imageService.selectImages(authorizedUser.getTenantId(), request.getIds());
        final List<String> billCodes = images.stream().map(Image::getBillCode).collect(Collectors.toList());
        if (ValidatorUtil.isNotEmpty(billCodes)) {
            final List<BaseBill> bills = baseBillService.selectBillDataByBillCodes(authorizedUser.getTenantId(), billCodes);
            if (bills.stream().anyMatch(bill -> Arrays.asList(BillDataStatus._3.getCode(), BillDataStatus._1.getCode()).contains(bill.getBillDataStatus()))) {
                request.getIds().clear();
                return "影像所属单据已锁定或已完成，不可删除";
            }
            // 已退回单据删除需要先下发
            billImageTicketService.billReturnDeleteSend(authorizedUser.getTenantId(), bills);
            return "删除成功";
        }
        return "删除成功";
    }

}