package com.xforceplus.elephant.image.controller.compare.billdata.process;

import static com.xforceplus.elephant.image.core.business.consts.Constants.TENANT_CODE;

import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Maps;
import com.xforceplus.elephant.basecommon.process.AbstractProcess;
import com.xforceplus.elephant.basecommon.process.response.CommonResponse;
import com.xforceplus.elephant.basecommon.vaildate.ValidatorUtil;
import com.xforceplus.elephant.image.client.model.BatchAuditBillRequest;
import com.xforceplus.elephant.image.core.business.application.config.common.enums.NoticeTypeEnum;
import com.xforceplus.elephant.image.core.business.config.queue.MQUtils;
import com.xforceplus.elephant.image.core.business.enums.MQEnum;
import com.xforceplus.elephant.image.core.domain.bill.BaseBillService;
import com.xforceplus.elephant.image.core.domain.bill.BillNoticeContext;
import com.xforceplus.elephant.image.core.domain.compare.bill.CompareBaseBillService;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.dict.YesNo;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.entity.BaseBill;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.meta.EntityMeta;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import javax.validation.ValidationException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

/**
 * 单据稽核结果审核process
 *
 * @author liming
 * @date 2021/5/8 18:22
 */
@Service
public class CompareAuditBillProcess extends AbstractProcess<BatchAuditBillRequest, Boolean> {

    @Autowired
    private CompareBaseBillService compareBaseBillService;
    @Autowired
    private BaseBillService baseBillService;
    @Autowired
    private MQUtils rabbitmqUtils;
    @Autowired
    private BillNoticeContext billNoticeContext;

    @Override
    protected void check(BatchAuditBillRequest request) throws ValidationException {
        checkEmpty(request.getBillCodes(), "billCodes不能为空");
        checkEmpty(request.getAuditStatus(), "稽核状态不能为空");
    }

    @Override
    protected CommonResponse<Boolean> process(BatchAuditBillRequest request) throws RuntimeException {
        final IAuthorizedUser authorizedUser = UserInfoHolder.get();// 获取登录用户上下文
        if (null == authorizedUser) {
            throw new ValidationException("获取用户信息为空，请重新登陆");
        }
        final List<BaseBill> billEntityList = compareBaseBillService.selectBillDataByBillCodes(authorizedUser.getTenantId(), request.getBillCodes());
        if (ValidatorUtil.isEmpty(billEntityList)) {
            return CommonResponse.failed("稽核单据不存在");
        }

        //更新稽核单据
        final List<BaseBill> waitAuditBillList = billEntityList.stream().filter(baseBill -> YesNo._0.getCode().equals(baseBill.getAuditStatus())).collect(Collectors.toList());
        for (BaseBill baseBill : waitAuditBillList) {
            final JSONObject jsonObject = new JSONObject();

            jsonObject.put("audit_status", request.getAuditStatus());
            jsonObject.put("audit_remark", request.getAuditRemark());
            jsonObject.put(EntityMeta.BaseBill.AUDIT_TIME.code(), System.currentTimeMillis());
            jsonObject.put(EntityMeta.BaseBill.AUDIT_USER_ID.code(), authorizedUser.getId());
            jsonObject.put(EntityMeta.BaseBill.AUDIT_USER_NAME.code(), authorizedUser.getUsername());
            compareBaseBillService.updateByBillIdSelective(baseBill.getId(), jsonObject);
        }
        updateAndCommitMainBill(request, authorizedUser, waitAuditBillList);
        billNoticeContext.choose(NoticeTypeEnum.COMPARE).batchSend(authorizedUser.getTenantId(), request.getBillCodes(), null);
        return CommonResponse.ok("成功", true);
    }

    /**
     * 同步更新并下发主数据单据信息 稽核审核时同步更新主数据，这样主数据下发稽核状态稽核备注时，就不用再额外查询稽核单据信息了
     *
     * @param request           请求
     * @param authorizedUser    用户
     * @param waitAuditBillList waitAuditBillList
     * @return void
     * @author liming
     * @date 2021/6/7 15:41
     */
    private void updateAndCommitMainBill(BatchAuditBillRequest request, IAuthorizedUser authorizedUser, List<BaseBill> waitAuditBillList) {
        if (ValidatorUtil.isEmpty(waitAuditBillList)) {
            return;
        }
        final List<String> billCodes = waitAuditBillList.stream().map(BaseBill::getBillCode).collect(Collectors.toList());
        if (ValidatorUtil.isEmpty(billCodes)) {
            return;
        }
        final List<BaseBill> billEntities = baseBillService.selectBillDataByBillCodes(authorizedUser.getTenantId(), billCodes);
        billEntities.forEach(baseBill -> {
            //更新主数据稽核状态及稽核备注
            final JSONObject jsonObject = new JSONObject();
            jsonObject.put("audit_status", request.getAuditStatus());
            jsonObject.put("audit_remark", request.getAuditRemark());
            jsonObject.put(EntityMeta.BaseBill.AUDIT_TIME.code(), System.currentTimeMillis());
            jsonObject.put(EntityMeta.BaseBill.AUDIT_USER_ID.code(), authorizedUser.getId());
            jsonObject.put(EntityMeta.BaseBill.AUDIT_USER_NAME.code(), authorizedUser.getUsername());
            baseBillService.updateByBillIdSelective(baseBill.getId(), jsonObject);

            //推送单据
            final Map<String, Object> pushMap = Maps.newHashMap();
            pushMap.put("source", "单据提交");
            pushMap.put("tenantId", authorizedUser.getTenantId());
            pushMap.put("billCode", baseBill.getBillCode());
            pushMap.put("billTypeCode", baseBill.getBillTypeCode());
            final Map<String, Object> headers = new HashMap<>();
            headers.put(TENANT_CODE, authorizedUser.getTenantCode());
            rabbitmqUtils.sendByDirectExchange(MQEnum.SYNC_BILL_TICKET_QUEUE, pushMap, headers);
        });
    }

}
