package com.xforceplus.elephant.image.controller.noauth.process;

import com.xforceplus.elephant.basecommon.process.AbstractProcess;
import com.xforceplus.elephant.basecommon.process.response.CommonResponse;
import com.xforceplus.elephant.basecommon.vaildate.ValidatorUtil;
import com.xforceplus.elephant.image.client.model.GetImageListRequest;
import com.xforceplus.elephant.image.client.model.ImageDTO;
import com.xforceplus.elephant.image.core.business.application.config.common.service.ConfigService;
import com.xforceplus.elephant.image.core.business.enums.DictEnum;
import com.xforceplus.elephant.image.core.business.util.PlainEntityServiceWrapper;
import com.xforceplus.elephant.image.core.domain.image.bean.ImageListResult;
import com.xforceplus.elephant.image.core.domain.image.bean.ImageSearchBean;
import com.xforceplus.elephant.image.core.domain.image.bean.ImageSearchResultBean;
import com.xforceplus.elephant.image.core.facade.application.collect.image.ImageFacade;
import com.xforceplus.elephant.image.core.facade.application.collect.ticket.TicketFacade;
import com.xforceplus.elephant.image.core.facade.dto.statistics.TicketStatisticsDto;
import com.xforceplus.elephant.image.core.util.RequestBuilder;
import com.xforceplus.elephant.image.mapper.ImageSearchMapper;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.dict.ImageType;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.dict.YesNo;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.entity.Image;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.meta.EntityMeta;
import com.xforceplus.ultraman.metadata.domain.vo.DataCollection;
import com.xforceplus.ultraman.metadata.domain.vo.dto.ConditionOp;
import com.xforceplus.ultraman.metadata.entity.IEntityClass;
import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;
import javax.validation.ValidationException;
import lombok.RequiredArgsConstructor;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.springframework.stereotype.Service;

@RequiredArgsConstructor
@Service
public class NoAuthGetImageListProcess extends AbstractProcess<GetImageListRequest, ImageSearchResultBean> {

    private final ImageFacade imageFacade;
    private final TicketFacade ticketFacade;
    private final ImageSearchMapper mapper;
    private final PlainEntityServiceWrapper entityService;
    private final ConfigService configService;

    @Override
    protected void check(GetImageListRequest request) throws ValidationException {
        super.check(request);
        checkEmpty(request.getBillCode(), "单据CODE不能为空");
    }

    @Override
    protected CommonResponse<ImageSearchResultBean> process(GetImageListRequest request) throws RuntimeException {
        final IAuthorizedUser user = UserInfoHolder.get();
        if (user == null) {
            throw new ValidationException("获取用户信息为空，请重新登陆");
        }
        //查询
        final ImageSearchBean searchBean = mapper.map(request);
        searchBean.setTenantId(user.getTenantId());
        final DataCollection<Image> imageList = imageFacade.selectBillImagesByCustomTicketParams(searchBean, EntityMeta.Image.code());
        if (ValidatorUtil.isEmpty(imageList.getRows())) {
            final ImageSearchResultBean resultBean = new ImageSearchResultBean();
            resultBean.setTotal(imageList.getRowNum().longValue());
            return CommonResponse.ok("成功", resultBean);
        }
        final List<Long> imageIds = imageList.getRows().stream().map(Image::getId).collect(Collectors.toList());
        final List<Map<String, Object>> ticketList = ticketFacade.selectTicketByImageIds(user.getTenantId(), imageIds, EntityMeta.Ticket.code());
        final Map<Long, List<Map<String, Object>>> ticketMap = ticketList.stream().collect(Collectors.groupingBy(map -> MapUtils.getLongValue(map, EntityMeta.Ticket.IMAGE_ID.code())));
        final List<String> tenantCodes = configService.selectDict(0L, DictEnum.NEED_WRAPPER_URL_TENANT_CONFIG.getCode(), String.class);
        final ImageListResult<ImageDTO> result = new ImageListResult<>(imageList.getRowNum().longValue(),
            imageList.getRows().stream().map(entity -> {
                final ImageDTO dto = mapper.map(entity);
                final StringBuilder sb = new StringBuilder();
                sb.append("/view/bill/image/").append(entity.getId())
                    .append(entity.getFileTypeHandle())
                    .append("?rid=").append(System.currentTimeMillis())
                    .append("&keyCode=").append(request.getRid());
                String fileUrl = sb.toString();
                if (CollectionUtils.isNotEmpty(tenantCodes) && !CollectionUtils.containsAny(tenantCodes, user.getTenantCode()) && !ImageType._3.getCode().equals(entity.getImageType())) {
                    fileUrl = ValidatorUtil.isNotEmpty(entity.getFileUrlLocal()) ? entity.getFileUrlLocal() : entity.getFileUrlHandle();
                }
                dto.setFileUrl(fileUrl);
                dto.setFileUrlHandle(null);
                dto.setFileUrlOrigin(null);
                dto.setIsAdd(YesNo._0.getCode());
                dto.setIsChange(YesNo._0.getCode());
                dto.setIsReplace(YesNo._0.getCode());
                dto.setAuditBackStatus(YesNo._0.getCode());
                dto.setAmountWithTax("0");
                dto.setTaxAmount("0");

                final List<Map<String, Object>> tickets = ticketMap.get(entity.getId());
                if (ValidatorUtil.isNotEmpty(tickets)) {
                    final AtomicReference<BigDecimal> totalAmountWithTax = new AtomicReference<>(BigDecimal.ZERO);
                    final AtomicReference<BigDecimal> totalTaxAmount = new AtomicReference<>(BigDecimal.ZERO);
                    tickets.forEach(ticket -> ticketFacade.calcAmountByTicket(totalAmountWithTax, totalTaxAmount, ticket));

                    dto.setAmountWithTax(totalAmountWithTax.get().toString());
                    dto.setTaxAmount(totalTaxAmount.get().toString());
                    dto.setIsAdd(tickets.stream().anyMatch(ticket -> YesNo._1.getCode().equals(MapUtils.getString(ticket, EntityMeta.Ticket.IS_ADD.code()))) ? YesNo._1.getCode()
                        : YesNo._0.getCode());
                    dto.setIsChange(tickets.stream().anyMatch(ticket -> YesNo._1.getCode().equals(MapUtils.getString(ticket, EntityMeta.Ticket.IS_CHANGE.code()))) ? YesNo._1.getCode()
                        : YesNo._0.getCode());
                    dto.setIsReplace(
                        tickets.stream().anyMatch(ticket -> YesNo._1.getCode().equals(MapUtils.getString(ticket, EntityMeta.Ticket.IS_REPLACE.code()))) ? YesNo._1.getCode()
                            : YesNo._0.getCode());
                    dto.setAuditBackStatus(
                        tickets.stream().anyMatch(ticket -> YesNo._1.getCode().equals(MapUtils.getString(ticket, EntityMeta.Ticket.AUDIT_BACK_STATUS.code()))) ? YesNo._1.getCode()
                            : YesNo._0.getCode());
                }
                if (ValidatorUtil.isEmpty(dto.getTicketCode())) {
                    return dto;
                }
                final IEntityClass ec = entityService.loadByCode(dto.getTicketCode());
                if (ec == null) {
                    return dto;
                }
                dto.setTicketName(ec.name());
                return dto;
            }).collect(Collectors.toList()));
        //统计
        final RequestBuilder ticketBuilder = new RequestBuilder()
            .field(EntityMeta.Ticket.TENANT_ID.code(), ConditionOp.eq, searchBean.getTenantId())
            .field(EntityMeta.Ticket.BILL_CODE.code(), ConditionOp.eq, searchBean.getBillCode());
        searchBean.getTicketParams().entrySet().stream()
            .filter(entry -> ValidatorUtil.isNotEmpty(entry.getValue()))
            .forEach(entry -> {
                final String[] values = entry.getValue().split(",");
                if (values.length > 1) {
                    ticketBuilder.field(entry.getKey(), ConditionOp.in, values);
                    return;
                }
                ticketBuilder.field(entry.getKey(), ConditionOp.eq, entry.getValue());
            });
        final TicketStatisticsDto ticketStatisticsDto = ticketFacade.statisticsTicketByParams(user.getTenantId(), ticketBuilder, EntityMeta.Ticket.code());
        final ImageSearchResultBean resultBean = mapper.map(ticketStatisticsDto);
        searchBean.setCountCoverFlag(true);
        searchBean.setPageSize(1);
        //单据封面个数
        final long billCoverCount = imageFacade.selectBillImagesByCustomTicketParams(searchBean, EntityMeta.Image.code()).getRowNum().longValue();
        resultBean.setBillCoverTotal(billCoverCount);
        resultBean.setTotal(result.getTotal());
        resultBean.setList(result.getList());
        return CommonResponse.ok("成功", resultBean);
    }

}