package com.xforceplus.elephant.image.controller.common;

import static com.xforceplus.elephant.image.core.business.consts.Constants.IMAGE_PREFIX;

import com.alibaba.fastjson.JSON;
import com.xforceplus.elephant.basecommon.process.response.CommonResponse;
import com.xforceplus.elephant.image.controller.BaseController;
import java.util.Set;
import lombok.RequiredArgsConstructor;
import org.apache.commons.collections4.CollectionUtils;
import org.redisson.api.RedissonClient;
import org.redisson.client.codec.StringCodec;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.data.redis.connection.DataType;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

@RestController
@RequestMapping("/cache/clean")
@RequiredArgsConstructor
public class CacheCleanController extends BaseController {

    private final RedisTemplate redisTemplate;
    private final RedissonClient redissonClient;

    @CacheEvict(cacheNames = {IMAGE_PREFIX + "hitOrgCache", IMAGE_PREFIX + "missOrgCache"}, allEntries = true)
    @GetMapping(value = "/org", produces = {"application/json"})
    public CommonResponse cleanOrgCache() {
        return CommonResponse.ok("清理组织缓存成功");
    }

    @CacheEvict(cacheNames = {IMAGE_PREFIX + "userInfoHitCache", IMAGE_PREFIX + "userInfoMissCache"}, allEntries = true)
    @GetMapping(value = "/userInfo", produces = {"application/json"})
    public CommonResponse cleanUserInfoCache() {
        return CommonResponse.ok("清理用户信息缓存成功");
    }

    @CacheEvict(cacheNames = {IMAGE_PREFIX + "userIdsHitCache", IMAGE_PREFIX + "userIdsMissCache"}, allEntries = true)
    @GetMapping(value = "/userIds", produces = {"application/json"})
    public CommonResponse cleanUserIdsCache() {
        return CommonResponse.ok("清理用户id缓存成功");
    }

    @CacheEvict(cacheNames = IMAGE_PREFIX + "blackCache", allEntries = true)
    @GetMapping(value = "/black", produces = {"application/json"})
    public CommonResponse cleanBlackCache() {
        return CommonResponse.ok("清理黑名单缓存成功");
    }

    @CacheEvict(cacheNames = IMAGE_PREFIX + "supplierDataCache", allEntries = true)
    @GetMapping(value = "/supplierData", produces = {"application/json"})
    public CommonResponse cleanSupplierDataCache() {
        return CommonResponse.ok("清理供应商缓存成功");
    }

    @GetMapping(value = "/key/{key}", produces = {"application/json"})
    public CommonResponse cleanKey(@PathVariable("key") final String key) {
        final Set<String> keys = redisTemplate.keys(key);
        if (!keys.isEmpty()) {
            redisTemplate.delete(keys);
            return CommonResponse.ok(String.format("清理%s条缓存成功，缓存key：%s", keys.size(), keys));
        }
        return CommonResponse.failed(String.format("%s缓存不存在", key));
    }

    @GetMapping(value = "/get/{key}", produces = {"application/json"})
    public CommonResponse getKey(@PathVariable("key") final String key) {
        final Set<String> keys = redisTemplate.keys(key);
        final StringBuilder sb = new StringBuilder("<br/>");
        keys.forEach(k -> {
            sb.append(k).append("：");
            final DataType dataType = redisTemplate.type(k);
            switch (dataType) {
                case STRING:
                    sb.append(redissonClient.getBucket(k, new StringCodec()).get());
                    break;
                case LIST:
                    sb.append(JSON.toJSONString(redissonClient.getList(k).readAll()));
                    break;
                case SET:
                case ZSET:
                    sb.append(JSON.toJSONString(redissonClient.getSet(k).readAll()));
                    break;
                case HASH:
                    sb.append(JSON.toJSONString(redissonClient.getMap(k)));
                    break;
                case STREAM:
                default:
                    break;
            }
            sb.append("<br/>");
        });
        if (CollectionUtils.isNotEmpty(keys)) {
            return CommonResponse.ok("查询成功", sb.toString());
        }
        return CommonResponse.failed(String.format("%s缓存不存在", key));
    }

}
