package com.xforceplus.elephant.image.controller.compare.billdata.process;

import com.xforceplus.elephant.basecommon.process.AbstractProcess;
import com.xforceplus.elephant.basecommon.process.response.CommonResponse;
import com.xforceplus.elephant.basecommon.vaildate.ValidatorUtil;
import com.xforceplus.elephant.image.client.model.BatchMarkRiskRequest;
import com.xforceplus.elephant.image.core.business.infrastructure.domain.bizorder.BizOrderMarkRiskRequest;
import com.xforceplus.elephant.image.core.business.infrastructure.port.BizOrderService;
import com.xforceplus.elephant.image.core.expand.compare.CompareBillImageTicketService;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import com.xforceplus.tenant.security.token.domain.UserType;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import javax.annotation.Resource;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.validation.ValidationException;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.web.util.WebUtils;

@Slf4j
@Service
@RequiredArgsConstructor
public class BatchMarkRiskProcess extends AbstractProcess<BatchMarkRiskRequest, Object> {

    private final BizOrderService bizOrderService;
    private final CompareBillImageTicketService compareBillImageTicketService;

    @Resource
    private HttpServletRequest httpServletRequest;

    @Override
    protected void check(BatchMarkRiskRequest request) throws ValidationException {
        super.check(request);
        checkEmpty(request.getBillCodes(), "单号不能为空");
    }

    @Override
    protected CommonResponse process(BatchMarkRiskRequest request) throws RuntimeException {
        final IAuthorizedUser user = UserInfoHolder.get();
        checkEmpty(user, "获取用户信息为空，请重新登陆");
        final List<BizOrderMarkRiskRequest> requestList = request.getBillCodes().stream()
                .map(billCode -> new BizOrderMarkRiskRequest()
                    .setBizOrderNo(billCode)
                    .setExceptionType(request.getExceptionType())
                    .setExceptionRemark(request.getExceptionRemark())
                    .setExceptionSubmitUserName(StringUtils.firstNonBlank(user.getUserName(), user.getUsername(), user.getEmail(), user.getMobile()))
                    .setExceptionTime(Long.toString(System.currentTimeMillis()))
                ).collect(Collectors.toList());
        final Map<String, String> markRiskResponse = bizOrderService.batchMarkRisk(user.getTenantCode(), getToken(), requestList);

        final StringBuilder message = new StringBuilder();
        markRiskResponse.forEach((billCode, msg) -> message.append("单据【").append(billCode).append("】标记风险失败：原因：").append(msg).append(";"));
        final List<String> billCodes = request.getBillCodes().stream().filter(billCode -> !markRiskResponse.containsKey(billCode)).collect(Collectors.toList());
        if (request.isDeleted() && ValidatorUtil.isNotEmpty(billCodes)) {
            compareBillImageTicketService.deleteBillData(user.getTenantId(), billCodes, false);
        }
        return message.length() > 0 ? CommonResponse.failed(message.toString()) : CommonResponse.ok("成功");
    }

    private String getToken() {
        final Cookie tokenCookie = WebUtils.getCookie(httpServletRequest, UserType.USER.tokenKey());
        if (tokenCookie != null) {
            return tokenCookie.getValue();
        }
        final String token = httpServletRequest.getHeader(UserType.USER.tokenKey());
        if (StringUtils.isNotBlank(token)) {
            return token;
        }
        return WebUtils.findParameterValue(httpServletRequest, UserType.USER.tokenKey());
    }

}
