package com.xforceplus.elephant.image.config;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.xforceplus.elephant.basecommon.logger.entity.RequestLogEntity;
import com.xforceplus.elephant.basecommon.logger.entity.ResponseLogEntity;
import com.xforceplus.general.utils.lang.SystemClock;
import javax.servlet.http.HttpServletRequest;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.annotation.AfterReturning;
import org.aspectj.lang.annotation.AfterThrowing;
import org.aspectj.lang.annotation.Before;
import org.aspectj.lang.annotation.Pointcut;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

/**
 * @author admin
 */
public class WebLogAspect {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    ThreadLocal<Long> startTime = new ThreadLocal<>();
    ThreadLocal<RequestLogEntity> req = new ThreadLocal<>();

    @Pointcut("execution(public * com.xforceplus.elephant.image.controller..*Controller.*(..))")
    public void webLog() {
    }

    @Pointcut("@annotation(com.xforceplus.elephant.basecommon.annotation.NotLog)")
    public void notLog() {
    }

    @Before("webLog() && !notLog()")
    public void doBefore(final JoinPoint joinPoint) throws Throwable {
        startTime.set(SystemClock.now());
        try {
            //接收到请求，记录请求内容
            final ServletRequestAttributes attributes = (ServletRequestAttributes) RequestContextHolder.getRequestAttributes();
            if (attributes == null) {
                req.set(null);
                return;
            }
            final HttpServletRequest request = attributes.getRequest();
            writeAppRequest(joinPoint, request);
        } catch (final Exception ex) {
            logger.error("拦截参数记录下请求内容出现异常：" + ex);
        }
    }

    @AfterReturning(returning = "ret", pointcut = "webLog() && !notLog()")
    public void doAfterReturning(final Object ret) {
        try {
            if (req.get() == null) {
                return;
            }
            // 处理完请求，返回内容
            final Long time = (SystemClock.now() - startTime.get());
            writeAppResponse(req.get(), ret, time);
        } catch (final Exception e) {
            logger.error("拦截返回记录下返回内容出现异常：" + e);
        } finally {
            req.remove();
            startTime.remove();
        }
    }

    @AfterThrowing(throwing = "throwable", pointcut = "webLog() && !notLog()")
    public void exception(final Throwable throwable) {
        logger.error("请求异常！", throwable);
        startTime.remove();
        req.remove();
    }

    private void writeAppRequest(final JoinPoint joinPoint, final HttpServletRequest request) {

        final StringBuffer requestParm = new StringBuffer("");
        if (joinPoint.getArgs() != null && joinPoint.getArgs().length > 0) {
            for (int i = 0; i < joinPoint.getArgs().length; i++) {
                requestParm.append(JSON.toJSONString(joinPoint.getArgs()[i], SerializerFeature.DisableCircularReferenceDetect));
            }
        }
        //final String traceId = IdUtil.fastSimpleUUID();
        //MDC.put("traceId", traceId);
        final RequestLogEntity logEntity = new RequestLogEntity();
        logEntity.setTitle("请求报文");
        //logEntity.setRid(traceId);
        logEntity.setName(request.getRequestURI().substring(request.getRequestURI().lastIndexOf("/") + 1));
        logEntity.setUrl(request.getRequestURL().toString());
        logEntity.setIp(request.getRemoteAddr());
        logEntity.setRequestStr(requestParm.toString());
        req.set(logEntity);
        logger.debug(JSON.toJSONString(logEntity));
    }

    private void writeAppResponse(final RequestLogEntity reqEntity, final Object ret, final Long time) {
        final ResponseLogEntity logEntity = new ResponseLogEntity();
        logEntity.setTitle("返回报文");
        logEntity.setRid(reqEntity.getRid());
        logEntity.setName(reqEntity.getName());
        logEntity.setUrl(reqEntity.getUrl());
        logEntity.setIp(reqEntity.getIp());
        logEntity.setResponseStr(JSON.toJSONString(ret));
        logEntity.setTime(time);
        logger.debug(JSON.toJSONString(logEntity));
        //MDC.remove("traceId");
    }

}
