package com.xforceplus.general.alarm.message;

import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Supplier;
import lombok.AccessLevel;
import lombok.NoArgsConstructor;

/**
 * @author zhuxingsheng@gmail.com
 * @description: Singleton object container
 * @date 2024/3/6 13:09
 */
@NoArgsConstructor(access = AccessLevel.PRIVATE)
public class Singleton {

    private static final ConcurrentHashMap<String, Object> SINGLE_OBJECT_POOL = new ConcurrentHashMap<>();

    /**
     * Get a singleton object by key.
     *
     * @param key
     * @param <T>
     * @return
     */
    public static <T> T get(final String key) {
        final Object result = SINGLE_OBJECT_POOL.get(key);
        return result == null ? null : (T) result;
    }

    /**
     * Get a singleton object by key.
     *
     * <p> When empty, build a singleton object through supplier and put it into the container.
     *
     * @param key
     * @param supplier
     * @param <T>
     * @return
     */
    public static <T> T get(final String key, final Supplier<T> supplier) {
        Object result = SINGLE_OBJECT_POOL.get(key);
        if (result == null) {
            result = supplier.get();
            if (result != null) {
                SINGLE_OBJECT_POOL.put(key, result);
            }
        }
        return result != null ? (T) result : null;
    }

    /**
     * Object into container.
     *
     * @param value
     */
    public static void put(final Object value) {
        put(value.getClass().getName(), value);
    }

    /**
     * Object into container.
     *
     * @param key
     * @param value
     */
    public static void put(final String key, final Object value) {
        SINGLE_OBJECT_POOL.put(key, value);
    }

}
