package com.xforceplus.general.alarm.message.controller;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.xforceplus.general.alarm.configuration.NotifyProperties;
import com.xforceplus.general.alarm.message.handler.ding.ExceptionAlarmContent;
import io.vavr.Tuple;
import io.vavr.Tuple2;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.commons.collections4.ListUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;

/**
 * @author zhuxingsheng@gmail.com
 * @description: TODO
 * @date 2023/10/22 22:39
 */
@Component
public class AlarmControlHandler {

    private final Map<String, ReentrantLock> alarmControllerLock = new HashMap<>();

    private final Map<String, Cache<String, Integer>> alarmControlCacheMap = new ConcurrentHashMap<>();

    /**
     * Control message push alarm frequency.
     */
    public Tuple2<Boolean, Integer> isSendAlarm(final String notifyKey, final NotifyProperties notifyProperties) {
        if (notifyProperties.isEnabled()) {
            return isSendAlarm(notifyProperties.getNotifyId(), notifyKey, notifyProperties);
        }
        return Tuple.of(false, 0);
    }

    public Tuple2<Boolean, Integer> isSendAlarm(final ExceptionAlarmContent exceptionAlarmContent, final NotifyProperties notifyProperties) {
        if (notifyProperties.isEnabled() && isAlarm(exceptionAlarmContent.getException(), notifyProperties)) {
            return isSendAlarm(notifyProperties.getNotifyId(), exceptionAlarmContent.getUid(), notifyProperties);
        }
        return Tuple.of(false, 0);
    }

    private boolean isAlarm(final Throwable exception, final NotifyProperties notifyProperties) {
        if (ListUtils.emptyIfNull(notifyProperties.getException().getExcludeExceptions()).stream().anyMatch((p) -> {
            return StringUtils.equalsIgnoreCase(exception.getClass().getName(), p);
        })) {
            return false;
        } else {
            return !ListUtils.emptyIfNull(notifyProperties.getException().getExcludeExceptionMessages()).stream().anyMatch((p) -> {
                return StringUtils.equalsIgnoreCase(exception.getMessage(), p);
            });
        }
    }

    public Tuple2<Boolean, Integer> isSendAlarm(final String notifyId, final String notifyKey, final NotifyProperties notifyProperties) {
        final Cache<String, Integer> cache = alarmControlCacheMap.get(notifyId);
        if (cache == null) {
            return Tuple.of(false, 0);
        }
        Integer count = cache.getIfPresent(notifyKey);
        if (count == null) {
            final ReentrantLock lock = alarmControllerLock.get(notifyId);
            lock.lock();
            try {
                count = cache.getIfPresent(notifyKey);
                if (count == null) {
                    count = 1;
                    cache.put(notifyKey, count);
                    return Tuple.of(true, count);
                } else {
                    return isSendAlarm(notifyKey, notifyProperties, cache);
                }
            } finally {
                lock.unlock();
            }
        } else {
            final ReentrantLock lock = alarmControllerLock.get(notifyId);
            lock.lock();
            try {
                return isSendAlarm(notifyKey, notifyProperties, cache);
            } finally {
                lock.unlock();
            }
        }
    }

    private static Tuple2<Boolean, Integer> isSendAlarm(final String notifyKey, final NotifyProperties notifyProperties, final Cache<String, Integer> cache) {
        Integer count = cache.getIfPresent(notifyKey);
        ++count;
        cache.put(notifyKey, count);
        if (notifyProperties.getSample() != 0 & count % notifyProperties.getSample() == 0) {
            return Tuple.of(true, count);
        }
        return Tuple.of(false, count);
    }

    /**
     * Init cache and lock.
     *
     * @param notifyId        notifyId
     * @param intervalSeconds intervalSeconds
     */
    public void initCacheAndLock(final String notifyId, final Integer intervalSeconds) {
        final String cacheKey = notifyId;
        final Cache<String, Integer> cache = Caffeine.newBuilder()
            .expireAfterWrite(intervalSeconds, TimeUnit.SECONDS)
            .build();
        alarmControlCacheMap.put(cacheKey, cache);
        // Set the lock to prevent false sending of alarm information.
        final ReentrantLock reentrantLock = new ReentrantLock();
        alarmControllerLock.put(cacheKey, reentrantLock);
    }

}
