package com.xforceplus.general.alarm.message.handler.ding;

import static java.util.stream.Collectors.toList;

import com.google.common.base.Joiner;
import com.xforceplus.general.alarm.configuration.NotifyProperties;
import io.vavr.control.Try;
import java.net.InetAddress;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import lombok.Data;
import org.apache.commons.lang3.ArrayUtils;
import org.springframework.util.DigestUtils;
import org.springframework.util.StringUtils;

/**
 * @author zhuxingsheng@gmail.com
 * @description: TODO
 * @date 2024/2/20 11:20
 */
@Data
public class ExceptionAlarmContent {

    /**
     * 分布式跟踪id
     */
    protected String traceId;
    /**
     * 工程名
     */
    protected String project;
    protected NotifyProperties notifyProperties;
    /**
     * 异常的标识码
     */
    protected String uid;
    /**
     * 方法名
     */
    protected String methodName;
    /**
     * 方法参数信息
     */
    protected Map<String, String> parames;
    /**
     * 类路径
     */
    protected String classPath;
    /**
     * 异常信息
     */
    protected String exceptionMessage;
    protected Throwable exception;
    /**
     * 异常追踪信息
     */
    protected List<String> traceInfo = new ArrayList<>();
    private int times;

    public ExceptionAlarmContent(final Throwable ex, final String filterTrace) {
        this.exception = ex;
        this.exceptionMessage = gainExceptionMessage(ex);
        final List<StackTraceElement> list = new ArrayList<>();
        for (final StackTraceElement x : ex.getStackTrace()) {
            if (x != null & x.getClassName() != null) {
                if (StringUtils.isEmpty(filterTrace)) {
                    list.add(x);
                } else if (x.getClassName().startsWith(filterTrace)) {
                    list.add(x);
                }
            }
        }
        if (list.size() > 0) {
            this.traceInfo = list.stream().map(x -> x.toString()).collect(toList());
            this.methodName = list.get(0).getMethodName();
            this.classPath = list.get(0).getClassName();
        }
        this.uid = calUid();
    }

    public ExceptionAlarmContent(final NotifyProperties notifyProperties, final String traceId, final Throwable ex, final String filterTrace, final Object[] args) {
        this(ex, filterTrace, args);
        this.traceId = traceId;
        this.notifyProperties = notifyProperties;
    }

    public ExceptionAlarmContent(final Throwable ex, final String filterTrace, final Object[] args) {
        this(ex, filterTrace);
        //钉钉告警时，有报文大小限制，暂放在这儿截取了
        int range = ArrayUtils.getLength(args) / 2;
        this.parames = IntStream.range(0, range).boxed().collect(Collectors.toMap(i -> String.valueOf(args[2 * i]), i -> String.valueOf(args[2 * i + 1])));
    }

    private String gainExceptionMessage(final Throwable exception) {
        String em = exception.toString();
        if (exception.getCause() != null) {
            em = String.format("%s\r\n\tcaused by : %s", em, gainExceptionMessage(exception.getCause()));
        }
        return em;
    }

    protected String calUid() {
        String traceInfoFirst = "";
        if (this.traceInfo != null && traceInfo.size() > 0) {
            traceInfoFirst = traceInfo.get(0);
        }
        final String md5 = DigestUtils.md5DigestAsHex(String.format("%s-%s", exceptionMessage, traceInfoFirst).getBytes());
        return md5;
    }

    public String createText() {
        final StringBuilder stringBuilder = new StringBuilder();
        stringBuilder.append("[" + notifyProperties.getKeyword() + "]\r\n");
        stringBuilder.append("traceId:").append(traceId).append("\r\n");
        stringBuilder.append("appName：").append(notifyProperties.getProjectName()).append("\r\n");
        stringBuilder.append("env：").append(System.getProperty("env")).append("\r\n");
        stringBuilder.append("host：").append(Try.of(() -> InetAddress.getLocalHost())
                                                .map(addr -> StringUtils.arrayToDelimitedString(new String[]{addr.getHostName(), addr.getHostAddress()}, ":")).getOrNull()).append("\r\n");
        stringBuilder.append("className：").append(classPath).append("\r\n");
        stringBuilder.append("methodName：").append(methodName).append("\r\n");
        stringBuilder.append("tag：").append(notifyProperties.getTag()).append("\r\n");
        if (parames != null && parames.size() > 0) {

            stringBuilder.append("parames：")
                         .append(Joiner.on(",").withKeyValueSeparator(":").join(parames)).append("\r\n");
        }
        stringBuilder.append("exceptionMessage：").append(exceptionMessage).append("\r\n");
        stringBuilder.append("stackTrace：").append("\r\n").append(String.join("\r\n", traceInfo)).append("\r\n");
        stringBuilder.append("times：").append(getTimes()).append("\r\n");
        return stringBuilder.toString();
    }

    /**
     * @return the project
     */
    public String getProject() {
        return project;
    }

    /**
     * @param project the project to set
     */
    public void setProject(final String project) {
        this.project = project;
    }

    /**
     * @return the methodName
     */
    public String getMethodName() {
        return methodName;
    }

    /**
     * @param methodName the methodName to set
     */
    public void setMethodName(final String methodName) {
        this.methodName = methodName;
    }

    /**
     * @return the classPath
     */
    public String getClassPath() {
        return classPath;
    }

    /**
     * @param classPath the classPath to set
     */
    public void setClassPath(final String classPath) {
        this.classPath = classPath;
    }

    /**
     * @return the exceptionMessage
     */
    public String getExceptionMessage() {
        return exceptionMessage;
    }

    /**
     * @param exceptionMessage the exceptionMessage to set
     */
    public void setExceptionMessage(final String exceptionMessage) {
        this.exceptionMessage = exceptionMessage;
    }

    /**
     * @return the traceInfo
     */
    public List<String> getTraceInfo() {
        return traceInfo;
    }

    /**
     * @param traceInfo the traceInfo to set
     */
    public void setTraceInfo(final List<String> traceInfo) {
        this.traceInfo = traceInfo;
    }

    /**
     * @return the uid
     */
    public String getUid() {
        return uid;
    }

    /**
     * @param uid the uid to set
     */
    public void setUid(final String uid) {
        this.uid = uid;
    }


    public String getTraceId() {
        return traceId;
    }

    public void setTraceId(final String traceId) {
        this.traceId = traceId;
    }

    public Throwable getException() {
        return exception;
    }

    public void setException(final Throwable exception) {
        this.exception = exception;
    }


}
