package com.xforceplus.general.starter.errorcode;

import com.google.common.base.Verify;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import java.lang.reflect.Field;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Optional;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.reflect.FieldUtils;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.core.MethodParameter;
import org.springframework.http.HttpMethod;
import org.springframework.http.MediaType;
import org.springframework.http.converter.HttpMessageConverter;
import org.springframework.http.server.ServerHttpRequest;
import org.springframework.http.server.ServerHttpResponse;
import org.springframework.util.ReflectionUtils;
import org.springframework.web.bind.annotation.RestControllerAdvice;
import org.springframework.web.servlet.mvc.method.annotation.ResponseBodyAdvice;

/**
 * @author zhuxingsheng@gmail.com
 * @description: TODO
 * @date 2022/5/4 17:54
 */
@ConditionalOnProperty(prefix = "xforce.gen-tool.errorcode", name = "enabled", havingValue = "true")
@RestControllerAdvice
@Slf4j
public class ErrorCodeInterceptor implements ResponseBodyAdvice<Object>, ApplicationContextAware {

    private ApplicationContext applicationContext;

    @Autowired
    private ErrorCodeConfiguration errorCodeConfiguration;

    @Override
    public boolean supports(MethodParameter returnType, Class<? extends HttpMessageConverter<?>> converterType) {
        return errorCodeConfiguration.isEnabled();
    }

    @Override
    public Object beforeBodyWrite(
        Object body, MethodParameter returnType, MediaType selectedContentType, Class<? extends HttpMessageConverter<?>> selectedConverterType, ServerHttpRequest request,
        ServerHttpResponse response) {

        modifyMessage(request.getMethod(), request.getURI().getPath(), body);
        return body;
    }

    private void modifyMessage(HttpMethod httpMethod, String requestPath, Object body) {

        Field declaredField = FieldUtils.getDeclaredField(body.getClass(), ErrorCodeContext.getInstance().getCodeFieldName(), true);
        Optional.ofNullable(declaredField)
            .map(field -> ReflectionUtils.getField(field, body))
            .map(Objects::toString).ifPresent(code -> {
                getMessage(httpMethod, requestPath, code).ifPresent(message -> {
                    try {
                        FieldUtils.writeField(body, ErrorCodeContext.getInstance().getMessageFieldName(), message, true);
                    } catch (IllegalAccessException e) {
                        log.warn("error code modify message error", e);
                    }
                });
            });
    }

    private Optional<String> getMessage(HttpMethod httpMethod, String requestPath, String code) {
        return errorCodeConfigRepository()
            .getErrorCode(httpMethod, requestPath, UserInfoHolder.get().getTenantCode(), code).map(ErrorCode::getMessage);
    }

    private ErrorCodeConfigRepository errorCodeConfigRepository() {
        RepositoryType repositoryType = errorCodeConfiguration.getRepositoryType();
        ErrorCodeConfigRepository repository = null;

        switch (repositoryType) {
            case LOCAL:
                Map<String, ErrorCodeConfigRepository> repositoryMap = applicationContext.getBeansOfType(ErrorCodeConfigRepository.class);
                Verify.verify(repositoryMap.size() == 3, "ErrorCodeConfigRepository instance size not match 3");
                repository = repositoryMap.entrySet().stream()
                    .filter(entry -> !StringUtils.equalsAny(entry.getKey(), ApolloErrorCodeConfigRepository.class.getName(), RemoteErrorCodeConfigRepository.class.getName()))
                    .findFirst().map(Entry::getValue).orElse(null);
                break;
            case APOLLO:
                repository = applicationContext.getBean(ApolloErrorCodeConfigRepository.class);
                break;
            case REMOTE:
                repository = applicationContext.getBean(RemoteErrorCodeConfigRepository.class);
                break;
        }
        Verify.verifyNotNull(repository, "not found ErrorCodeConfigRepository,check errorcode.repositoryType config");
        return repository;
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }

}
