package com.xforceplus.general.sqs;

import com.alibaba.fastjson.JSON;
import com.google.common.collect.Maps;
import com.xforceplus.general.starter.logger.annotation.Trace;
import com.xforceplus.xplat.aws.SqsData;
import com.xforceplus.xplat.aws.sqs.listener.AbsSQSListener;
import io.micrometer.core.instrument.Metrics;
import io.micrometer.core.instrument.Timer;
import java.util.Map;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

/**
 * @author zhuxingsheng@gmail.com
 * @description: 可观察的sqs基类，所有sqs listener都继承此类
 * <p>
 * 增加统一的接收日志记录, 无需每个子类重新编写一遍
 * <p>
 * 以防系统没有初始化完成，标识为@Lazy
 * @date 2023/4/8 21:48
 */
@Slf4j
public abstract class ObservableSqsListener extends AbsSQSListener {

    @Override
    @Trace
    public boolean cusListener(final SqsData sqsData) {
        long interval = 0;
        boolean result = false;
        Map header = null;
        final String strProperties = sqsData.getProperties();
        final Timer.Sample sample = Timer.start(Metrics.globalRegistry);
        String exceptionClass = "none";
        final String queueName = sqsData.getQueueName();
        try {
            header = StringUtils.isNotBlank(strProperties) ? JSON.parseObject(strProperties, Map.class) : Maps.newHashMap();

            log.info("SqsListener,onMessage start queueName:{}  onMessage=messageId:{} header:{} body:{}",
                queueName, sqsData.getMessageId(), header, sqsData.getMsg());

            // 上下文
            final String strCtx = sqsData.getContext();
            final long start = System.currentTimeMillis();

            result = onMessage(sqsData.getMessageId(), header, sqsData.getMsg());

            final long end = System.currentTimeMillis();
            interval = end - start;
        } catch (final Exception e) {

            exceptionClass = e.getClass().getSimpleName();

            log.error("SqsListener,onMessage failed queueName:{} result:{} onMessage=messageId:{}", queueName, result, sqsData.getMessageId());
            log.error("SqsListener,onMessage failed", e);
            throw e;
        } finally {
            log.info("SqsListener,onMessage end queueName:{} result:{} interval: {} ms ", queueName, result, interval);
            record(sample, queueName, exceptionClass);

        }
        return result;
    }

    private void record(final Timer.Sample sample, final String queueName, final String exceptionClass) {
        recordTime(sample, "receive_data_from_sqs", exceptionClass,
            "action",
            "receiveSQS",
            "sqs",
            queueName);
    }

    public void recordTime(final Timer.Sample sample, final String metricName, final String exceptionClassName, final String... tags) {

        try {
            sample.stop(Timer.builder(metricName)
                .tags(
                    tags
                )
                .tag("exception", exceptionClassName)
                .publishPercentileHistogram(false)
                .publishPercentiles(0.5, 0.9, 0.99)
                .register(Metrics.globalRegistry));
        } catch (final Exception e) {
            log.error("recordTime error,metricName:{}", metricName, e);
        }
    }

    /**
     * 处理消息
     *
     * @param messageId   消息id
     * @param headers     消息属性
     * @param messageBody 消息body
     * @return
     */
    public abstract boolean onMessage(String messageId, Map headers, String messageBody);

}
