package com.xforceplus.general.ultraman.advice;

import com.google.common.base.Preconditions;
import com.google.common.base.Stopwatch;
import com.xforceplus.general.alarm.service.AlarmService;
import com.xforceplus.general.starter.logger.TraceContext;
import com.xforceplus.general.ultraman.acl.Converter;
import com.xforceplus.general.ultraman.configuration.UltramanStoreProperties;
import com.xforceplus.general.ultraman.exception.UltramanException;
import com.xforceplus.general.utils.json.JsonUtil;
import com.xforceplus.tech.base.core.context.ContextKeys;
import com.xforceplus.tech.base.core.context.ContextService;
import com.xforceplus.ultraman.metadata.entity.IEntityClass;
import com.xforceplus.xlog.core.model.LogContext;
import com.xforceplus.xlog.core.model.impl.UltramanLogEvent;
import com.xforceplus.xlog.core.model.setting.XlogSettings;
import com.xforceplus.xlog.core.model.setting.XlogUltramanSetting;
import com.xforceplus.xlog.logsender.model.LogSender;
import com.xforceplus.xlog.springboot.autoconfiguration.model.XlogProperties;
import io.prometheus.client.Summary;
import java.lang.reflect.Method;
import java.nio.charset.StandardCharsets;
import java.util.Optional;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.Signature;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;

/**
 * purchaser-invoice-service
 *
 * @author MrYe
 * @date 2022-04-24 22:33
 */
@Slf4j
@Aspect
@Component
@RequiredArgsConstructor
public class DataStorageAdvice {

    private final AlarmService alarmService;

    private final UltramanStoreProperties ultramanStoreProperties;
    private final Optional<XlogProperties> xlogProperties;
    private final Optional<XlogSettings> xlogSettings;
    private final Optional<LogSender> logSender;
    private final ContextService contextService;
    private final Converter converter;

    @Qualifier("ultramanSummary")
    private final Summary ultramanSummary;

    @Pointcut("@annotation(com.xforceplus.general.ultraman.log.UltramanLog)")
    public void logCut() {

    }

    public void warning(final String methodName, final Exception error, final Object... args) {
        //钉钉通知
        alarmService.alarm(TraceContext.getPlusTraceId(), error, args);
    }

    private String getEntityClassCode(final Method method, final Object entity) {

        Preconditions.checkArgument(entity instanceof IEntityClass, String.format("%s second param must be of IEntityClass type", method.getName()));
        final IEntityClass entityClass = (IEntityClass) entity;
        return Optional.ofNullable(entityClass).map(IEntityClass::code).orElse(StringUtils.EMPTY);
    }

    @Around("logCut()")
    public Object around(final ProceedingJoinPoint joinPoint) throws Throwable {
        final Signature signature = joinPoint.getSignature();

        final MethodSignature methodSignature = (MethodSignature) signature;
        final Method method = joinPoint.getTarget().getClass().getMethod(methodSignature.getName(),
            methodSignature.getParameterTypes());

        final String methodName = method.getName();
        final Object[] params = joinPoint.getArgs();

        final Stopwatch stopwatch = Stopwatch.createStarted();

        final com.xforceplus.general.ultraman.domain.UltramanLog ultramanLog = createUltramanLog(method, params);

        Object result = null;
        try {
            result = joinPoint.proceed();

            ultramanLog.setResponseText(JsonUtil.toJsonString(result));
            ultramanLog.setResponseSize(ultramanLog.getResponseText().getBytes(StandardCharsets.UTF_8).length);
            ultramanLog.setSuccessful(true);

        } catch (final Exception e) {

            warning(methodName, e);
            log.error("ultraman error,method:" + methodName, e);

            ultramanLog.setThrowable(e);
            ultramanLog.setSuccessful(false);

            throw new UltramanException(methodName + "异常", e);
        } finally {
            record(stopwatch, ultramanLog);
        }
        return result;
    }

    @NotNull
    private UltramanLogEvent createUltramanLogEvent(final com.xforceplus.general.ultraman.domain.UltramanLog ultramanLog) {
        final UltramanLogEvent ultramanLogEvent = converter.convert(ultramanLog);
        if (xlogSettings.map(XlogSettings::getUltraman).map(XlogUltramanSetting::isIgnoreRequestText).orElse(false)) {
            ultramanLogEvent.setRequestText(null);
            ultramanLogEvent.setRequestSize(0);
        }

        if (xlogSettings.map(XlogSettings::getUltraman).map(XlogUltramanSetting::isIgnoreResponseText).orElse(false)) {
            ultramanLogEvent.setResponseText(null);
            ultramanLogEvent.setResponseSize(0);
        }
        ultramanLogEvent.setStoreName(xlogProperties.map(XlogProperties::getStoreName).orElse(StringUtils.EMPTY));
        ultramanLogEvent.setType("Ultraman");
        ultramanLogEvent.setTraceId(LogContext.getTraceId());
        ultramanLogEvent.setExtValue("contextTenantCode", ultramanLog.getTenantCodeInContext());

        //是不是多租户
        ultramanLogEvent.setExtValue("isMultiTenant", !StringUtils.equals(ultramanLog.getTenantCodeInContext(), ultramanLog.getTenantInfo().getTenantCode()));

        return ultramanLogEvent;
    }

    private com.xforceplus.general.ultraman.domain.UltramanLog createUltramanLog(final Method method, final Object[] params) {
        final String tenantCode = (String) params[0];
        final String entityCode = getEntityClassCode(method, params[1]);

        final com.xforceplus.general.ultraman.domain.UltramanLog ultramanLog = new com.xforceplus.general.ultraman.domain.UltramanLog();

        ultramanLog.setParams(params);
        //剔除掉第一个tenantCode和第二个IEntityClass参数
        final Object[] logParams = ArrayUtils.subarray(params, 2, params.length);
        ultramanLog.setRequestText(JsonUtil.toJsonString(logParams));
        ultramanLog.setRequestSize(ultramanLog.getRequestText().getBytes(StandardCharsets.UTF_8).length);

        //使用注解标识的method，没有标识则使用拦截的方法名
        final String methodName = StringUtils.defaultIfBlank(method.getAnnotation(com.xforceplus.general.ultraman.log.UltramanLog.class).method(), method.getName());
        ultramanLog.setMethod(methodName);
        ultramanLog.setEntityCode(entityCode);

        //从上下文里面取tenantId，消息类入口，会取不到值；tenantCode从入参上取值
        final String tenantId = contextService.get(ContextKeys.StringKeys.TENANTID_KEY);
        final String tenantCodeFromContext = contextService.get(ContextKeys.StringKeys.TENANTCODE_KEY);
        final String userName = contextService.get(ContextKeys.StringKeys.USER_DISPLAYNAME);

        final com.xforceplus.general.ultraman.domain.vo.TenantInfo tenantInfo = new com.xforceplus.general.ultraman.domain.vo.TenantInfo();
        tenantInfo.setTenantId(tenantId);
        tenantInfo.setTenantCode(tenantCode);
        tenantInfo.setUsername(userName);
        ultramanLog.setTenantInfo(tenantInfo);

        ultramanLog.setTenantCodeInContext(tenantCodeFromContext);

        return ultramanLog;
    }

    private void record(final Stopwatch stopwatch,
        final com.xforceplus.general.ultraman.domain.UltramanLog ultramanLog) {

        printLog(ultramanLog, stopwatch);

        if (xlogProperties.map(p -> p.getUltraman().isEnabled()).orElse(false)) {
            logSender.ifPresent(s -> s.send(createUltramanLogEvent(ultramanLog)));
        }

        try {
            final long cost = stopwatch.elapsed().toMillis();
            ultramanSummary.labels(ultramanLog.getMethod()).observe(cost);
        } catch (final Exception e) {
            log.error("uploadOqsInfo error entityClass[{}][{}] {}", ultramanLog.getEntityCode(), ultramanLog.getMethod(), e.getMessage(), e);
        }
    }

    private void printLog(final com.xforceplus.general.ultraman.domain.UltramanLog ultramanLog, final Stopwatch stopwatch) {
        if (!ultramanStoreProperties.isLogEnabled()) {
            return;
        }
        final Long userId = contextService.get(ContextKeys.LongKeys.ID);
        String parameters = "nonlog";
        if (ultramanStoreProperties.isLogRequestEnabled()) {
            parameters = ultramanLog.getRequestText();
        }
        String result = "nonlog";
        if (ultramanStoreProperties.isLogResponseEnabled()) {
            result = ultramanLog.getResponseText();
        }
        log.info("method:{},entity:{},userName:{},userId:{},tenantId:{},tenantCode:{},duration:{},params:【{}】,paramSize:{},result:【{}】,resultSize:{},success:{}",
            ultramanLog.getMethod(), ultramanLog.getEntityCode(), ultramanLog.getTenantInfo().getUsername(),
            userId, ultramanLog.getTenantInfo().getTenantId(), ultramanLog.getTenantInfo().getTenantCode(), stopwatch.toString(),
            parameters, ultramanLog.getRequestSize(), result, ultramanLog.getResponseSize(), ultramanLog.isSuccessful());

    }

}
