package com.xforceplus.general.ultraman.store;

import cn.hutool.core.util.ReflectUtil;
import com.google.common.collect.Lists;
import com.xforceplus.general.ultraman.configuration.UltramanStoreProperties;
import com.xforceplus.general.ultraman.log.UltramanLog;
import com.xforceplus.ultraman.extensions.business.EntityId;
import com.xforceplus.ultraman.extensions.business.EntityInstance;
import com.xforceplus.ultraman.extensions.business.EntityKey;
import com.xforceplus.ultraman.extensions.business.service.QueryConfig;
import com.xforceplus.ultraman.extensions.business.service.TenantAwareBusinessFacade;
import com.xforceplus.ultraman.metadata.domain.vo.DataCollection;
import com.xforceplus.ultraman.metadata.domain.vo.Page;
import com.xforceplus.ultraman.metadata.domain.vo.Summary;
import com.xforceplus.ultraman.metadata.domain.vo.dto.ConditionOp;
import com.xforceplus.ultraman.metadata.domain.vo.dto.ConditionQueryRequest;
import com.xforceplus.ultraman.metadata.entity.IEntityClass;
import com.xforceplus.ultraman.metadata.helper.RequestBuilder;
import com.xforceplus.ultraman.sdk.core.rel.legacy.ExpFactory;
import com.xforceplus.ultraman.sdk.core.rel.legacy.ExpQuery;
import com.xforceplus.ultraman.sdk.core.rel.legacy.ExpRange;
import com.xforceplus.ultraman.sdk.core.rel.legacy.ExpRel;
import io.vavr.Tuple2;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import lombok.RequiredArgsConstructor;
import lombok.var;
import org.apache.commons.collections4.MapUtils;
import org.apache.poi.ss.formula.functions.T;
import org.springframework.stereotype.Component;

/**
 * 奥特曼操作数据包装类
 */
@Component
@RequiredArgsConstructor
public class UltramanDataStoreFacadeWrapper {

    private final TenantAwareBusinessFacade tenantAwareBusinessFacade;

    private final UltramanStoreProperties ultramanStoreProperties;

    @UltramanLog(method = "countAlL")
    public Long countAll(final String tenantCode, final IEntityClass entityClass) {
        return tenantAwareBusinessFacade.countAll(tenantCode, entityClass);
    }

    @UltramanLog
    public Long count(final String tenantCode, final IEntityClass entityClass, final ConditionQueryRequest request) {
        return tenantAwareBusinessFacade.count(tenantCode, entityClass, ExpFactory.createFrom(request));
    }

    @UltramanLog(method = "countExplicitly")
    public Long countExplicitly(final String tenantCode, final IEntityClass entityClass, final ConditionQueryRequest request, final QueryConfig queryConfig) {
        return tenantAwareBusinessFacade.countExplicitly(tenantCode, entityClass, ExpFactory.createFrom(request), queryConfig);
    }

    @UltramanLog
    public Long count(final String tenantCode, final IEntityClass entityClass, final ExpQuery query) {
        return tenantAwareBusinessFacade.count(tenantCode, entityClass, query);
    }

    @UltramanLog(method = "countExplicitly")
    public Long countExplicitly(final String tenantCode, final IEntityClass entityClass, final ExpQuery query, final QueryConfig queryConfig) {
        return tenantAwareBusinessFacade.countExplicitly(tenantCode, entityClass, query, queryConfig);
    }

    @UltramanLog(method = "create")
    public Long insert(final String tenantCode, final IEntityClass entityClass, final Map<String, Object> value) {
        return tenantAwareBusinessFacade.create(tenantCode, entityClass, value);
    }

    @UltramanLog(method = "create")
    public <T> Long insert(final String tenantCode, final IEntityClass entityClass, final T model) {
        return insert(tenantCode, entityClass, model, null);
    }

    @UltramanLog(method = "create")
    public <T> Long insert(final String tenantCode, final IEntityClass entityClass, final T model, final Map<String, Object> extValue) {
        final Map<String, Object> values = toOqsMap(model);
        if (MapUtils.isNotEmpty(extValue)) {
            values.putAll(extValue);
        }
        return insert(tenantCode, entityClass, values);
    }

    @UltramanLog(method = "createMulti")
    public <T> Integer insertBatch(final String tenantCode, final IEntityClass entityClass, final List<T> models) {
        final List<Map<String, Object>> values = models.stream().map(this::toOqsMap).collect(Collectors.toList());
        return insertBatchMap(tenantCode, entityClass, values);
    }

    @UltramanLog(method = "createMulti")
    public <T> Integer insertBatchMap(final String tenantCode, final IEntityClass entityClass, final List<Map<String, Object>> values) {
        return tenantAwareBusinessFacade.createMulti(tenantCode, entityClass, values);
    }

    @UltramanLog(method = "deleteOne")
    public Integer deleteById(final String tenantCode, final IEntityClass entityClass, final Long id) {
        return tenantAwareBusinessFacade.deleteOne(tenantCode, EntityId.of(entityClass, id));
    }

    @UltramanLog(method = "deleteMulti")
    public Integer deleteBatch(final String tenantCode, final IEntityClass entityClass, final List<Long> ids) {
        return tenantAwareBusinessFacade.deleteMulti(tenantCode, ids.stream().map(id -> EntityId.of(entityClass, id)).collect(Collectors.toList()));
    }

    @UltramanLog(method = "updateById")
    public <T> Integer updateById(final String tenantCode, final IEntityClass entityClass, final T model) {
        final Long identity = ReflectUtil.invoke(model, "getId");
        return Optional.ofNullable(identity).map(id -> tenantAwareBusinessFacade.updateById(tenantCode, EntityId.of(entityClass, id), toOqsMap(model), false)).orElse(null);
    }

    @UltramanLog(method = "updateById")
    public Integer updateById(final String tenantCode, final IEntityClass entityClass, final Map<String, Object> updateData, final Long id) {
        return tenantAwareBusinessFacade.updateById(tenantCode, EntityId.of(entityClass, id), updateData, false);
    }

    /**
     * 基于ID批量更新
     *
     * @param models 对象实体
     * @return 条数
     */
    @UltramanLog(method = "updateMulti")
    public <T> Integer updateBatch(final String tenantCode, final IEntityClass entityClass, final List<T> models) {
        final var dataList = models.stream().map(this::toOqsMap).collect(Collectors.toList());
        return tenantAwareBusinessFacade.updateMulti(tenantCode, entityClass, dataList);
    }

    /**
     * 基于ID批量更新
     *
     * @param values 集合数据
     * @return 行数
     */
    @UltramanLog(method = "updateMulti")
    public Integer updateBatchMap(final String tenantCode, final IEntityClass entityClass, final List<Map<String, Object>> values) {
        final var dataList = values.stream().filter(v -> MapUtils.getLong(v, "id") != null).collect(Collectors.toList());
        return tenantAwareBusinessFacade.updateMulti(tenantCode, entityClass, dataList);
    }

    /**
     * 根据keyValues更新 key和value条件是equals
     *
     * @param updateData         数据
     * @param conditionKeyValues 条件
     * @return 条数
     */
    @UltramanLog(method = "updateByKey")
    public Integer updateByKeys(final String tenantCode, final IEntityClass entityClass, final Map<String, Object> updateData, final Tuple2<String, Object>... conditionKeyValues) {
        return tenantAwareBusinessFacade.updateByKey(tenantCode, EntityKey.of(entityClass, conditionKeyValues), updateData);
    }

    @UltramanLog(method = "findAll")
    public <T> List<T> findAllWithoutPageAndSort(final String tenantCode, final IEntityClass entityClass, final Class<T> entityClazz, final ConditionQueryRequest request) {
        final var instanceIterator = tenantAwareBusinessFacade.findAll(tenantCode, entityClass, ExpFactory.createFrom(request));
        final List<T> dataList = Lists.newArrayList();
        while (instanceIterator.hasNext()) {
            final var instance = instanceIterator.next();
            dataList.add(fromOqsMap(entityClazz, instance.getRecord().toMap(null)));
        }
        return dataList;
    }

    @UltramanLog(method = "findAll")
    public <T> List<T> findAllWithoutPageAndSort(final String tenantCode, final IEntityClass entityClass, final Class<T> entityClazz, final ExpQuery expQuery) {
        final var instanceIterator = tenantAwareBusinessFacade.findAll(tenantCode, entityClass, expQuery);
        final List<T> dataList = Lists.newArrayList();
        while (instanceIterator.hasNext()) {
            final var instance = instanceIterator.next();
            dataList.add(fromOqsMap(entityClazz, instance.getRecord().toMap(null)));
        }
        return dataList;
    }

    @UltramanLog(method = "findAll")
    public List<Map<String, Object>> findAllMapWithoutPageAndSort(final String tenantCode, final IEntityClass entityClass, final ConditionQueryRequest request) {
        final var instanceIterator = tenantAwareBusinessFacade.findAll(tenantCode, entityClass, ExpFactory.createFrom(request));
        final List<Map<String, Object>> dataList = Lists.newArrayList();
        while (instanceIterator.hasNext()) {
            final var instance = instanceIterator.next();
            dataList.add(instance.getRecord().toMap(null));
        }
        return dataList;
    }

    @UltramanLog(method = "findAll")
    public List<Map<String, Object>> findAllMapWithoutPageAndSort(final String tenantCode, final IEntityClass entityClass, final ExpQuery expQuery) {
        final var instanceIterator = tenantAwareBusinessFacade.findAll(tenantCode, entityClass, expQuery);
        final List<Map<String, Object>> dataList = Lists.newArrayList();
        while (instanceIterator.hasNext()) {
            final var instance = instanceIterator.next();
            dataList.add(instance.getRecord().toMap(null));
        }
        return dataList;
    }

    @UltramanLog(method = "findByCondition")
    public Page<T> findPageByCondition(final String tenantCode, final IEntityClass entityClass, final Class<T> entityClazz, final ConditionQueryRequest request) {
        final var instanceIterator = tenantAwareBusinessFacade.findByCondition(tenantCode, entityClass, ExpFactory.createFrom(defaultPageIfNull(request)));
        final Summary summary = new Summary();
        summary.setTotal(instanceIterator.getRowNum());
        return new Page<>(instanceIterator.getRows().stream().map(a -> fromOqsMap(entityClazz, a.getRecord().toMap(null)))
            .collect(Collectors.toList()), summary, request.getPageSize(), request.getPageNo());
    }

    @UltramanLog(method = "findByCondition")
    public <T> Page<T> findPageByCondition(final String tenantCode, final IEntityClass entityClass, final Class<T> entityClazz, final ExpQuery query) {
        final var instanceIterator = tenantAwareBusinessFacade.findByCondition(tenantCode, entityClass, defaultPageIfNull(query));
        final ExpRange range = query.getRange();
        final Summary summary = new Summary();
        summary.setTotal(instanceIterator.getRowNum());
        return new Page<>(instanceIterator.getRows().stream().map(a -> fromOqsMap(entityClazz, a.getRecord().toMap(null)))
            .collect(Collectors.toList()), summary, range.getIndex(), range.getSize());
    }

    @UltramanLog(method = "findByConditionExplicitly")
    public <T> Page<T> findPageByConditionExplicitly(final String tenantCode, final IEntityClass entityClass, final Class<T> entityClazz, final ExpQuery query, final QueryConfig queryConfig) {
        final var instanceIterator = tenantAwareBusinessFacade.findByConditionExplicitly(tenantCode, entityClass, defaultPageIfNull(query), queryConfig);
        final ExpRange range = query.getRange();
        final Summary summary = new Summary();
        summary.setTotal(instanceIterator.getRowNum());
        return new Page<>(instanceIterator.getRows().stream().map(a -> fromOqsMap(entityClazz, a.getRecord().toMap(null)))
            .collect(Collectors.toList()), summary, range.getIndex(), range.getSize());
    }

    @UltramanLog(method = "findByCondition")
    public Page<Map<String, Object>> findPageMapByCondition(final String tenantCode, final IEntityClass entityClass, final ConditionQueryRequest request) {
        final var instanceIterator = tenantAwareBusinessFacade.findByCondition(tenantCode, entityClass, ExpFactory.createFrom((defaultPageIfNull(request))));
        final Summary summary = new Summary();
        summary.setTotal(instanceIterator.getRowNum());
        return new Page<>(instanceIterator.getRows().stream().map(a -> a.getRecord().toMap(null))
            .collect(Collectors.toList()), summary, request.getPageSize(), request.getPageNo());
    }

    @UltramanLog(method = "findByCondition")
    public Page<Map<String, Object>> findPageMapByCondition(final String tenantCode, final IEntityClass entityClass, final ExpQuery query) {
        final var instanceIterator = tenantAwareBusinessFacade.findByCondition(tenantCode, entityClass, defaultPageIfNull(query));
        final ExpRange range = query.getRange();
        final Summary summary = new Summary();
        summary.setTotal(instanceIterator.getRowNum());
        return new Page<>(instanceIterator.getRows().stream().map(a -> a.getRecord().toMap(null))
            .collect(Collectors.toList()), summary, range.getIndex(), range.getSize());
    }

    @UltramanLog(method = "findByCondition")
    public <T> List<T> findByCondition(final String tenantCode, final IEntityClass entityClass, final Class<T> entityClazz, final ConditionQueryRequest request) {
        final var instanceIterator = tenantAwareBusinessFacade.findByCondition(tenantCode, entityClass, ExpFactory.createFrom(defaultPageIfNull(request)));
        return instanceIterator.getRows().stream().map(a -> fromOqsMap(entityClazz, a.getRecord().toMap(null))).collect(Collectors.toList());
    }

    @UltramanLog(method = "findByConditionExplicitly")
    public <T> List<T> findByConditionExplicitly(final String tenantCode, final IEntityClass entityClass, final Class<T> entityClazz, final ConditionQueryRequest request,
        final QueryConfig queryConfig) {
        final var instanceIterator = tenantAwareBusinessFacade.findByConditionExplicitly(tenantCode, entityClass, ExpFactory.createFrom(defaultPageIfNull(request)), queryConfig);
        return instanceIterator.getRows().stream().map(a -> fromOqsMap(entityClazz, a.getRecord().toMap(null))).collect(Collectors.toList());
    }

    @UltramanLog(method = "findByCondition")
    public <T> List<T> findByCondition(final String tenantCode, final IEntityClass entityClass, final Class<T> entityClazz, final ExpQuery expQuery) {
        final var instanceIterator = tenantAwareBusinessFacade.findByCondition(tenantCode, entityClass, defaultPageIfNull(expQuery));
        return instanceIterator.getRows().stream().map(a -> fromOqsMap(entityClazz, a.getRecord().toMap(null))).collect(Collectors.toList());
    }

    @UltramanLog(method = "findByConditionExplicitly")
    public <T> List<T> findByConditionExplicitly(final String tenantCode, final IEntityClass entityClass, final Class<T> entityClazz, final ExpQuery expQuery, final QueryConfig queryConfig) {
        final var instanceIterator = tenantAwareBusinessFacade.findByConditionExplicitly(tenantCode, entityClass, defaultPageIfNull(expQuery), queryConfig);
        return instanceIterator.getRows().stream().map(a -> fromOqsMap(entityClazz, a.getRecord().toMap(null))).collect(Collectors.toList());
    }

    @UltramanLog(method = "findByCondition")
    public List<Map<String, Object>> findByConditionMap(final String tenantCode, final IEntityClass entityClass, final ConditionQueryRequest request) {
        final var instanceIterator = tenantAwareBusinessFacade.findByCondition(tenantCode, entityClass, ExpFactory.createFrom(defaultPageIfNull(request)));
        return instanceIterator.getRows().stream().map(a -> a.getRecord().toMap(null)).collect(Collectors.toList());
    }

    @UltramanLog(method = "findByConditionExplicitly")
    public List<Map<String, Object>> findByConditionMapExplicitly(final String tenantCode, final IEntityClass entityClass, final ConditionQueryRequest request, final QueryConfig queryConfig) {
        final var instanceIterator = tenantAwareBusinessFacade.findByConditionExplicitly(tenantCode, entityClass, ExpFactory.createFrom(defaultPageIfNull(request)), queryConfig);
        return instanceIterator.getRows().stream().map(a -> a.getRecord().toMap(null)).collect(Collectors.toList());
    }

    @UltramanLog(method = "findByCondition")
    public List<Map<String, Object>> findByConditionMap(final String tenantCode, final IEntityClass entityClass, final ExpQuery expQuery) {
        final var instanceIterator = tenantAwareBusinessFacade.findByCondition(tenantCode, entityClass, defaultPageIfNull(expQuery));
        return instanceIterator.getRows().stream().map(a -> a.getRecord().toMap(null)).collect(Collectors.toList());
    }

    @UltramanLog(method = "findByConditionExplicitly")
    public List<Map<String, Object>> findByConditionMapExplicitly(final String tenantCode, final IEntityClass entityClass, final ExpQuery expQuery, final QueryConfig queryConfig) {
        final var instanceIterator = tenantAwareBusinessFacade.findByConditionExplicitly(tenantCode, entityClass, defaultPageIfNull(expQuery), queryConfig);
        return instanceIterator.getRows().stream().map(a -> a.getRecord().toMap(null)).collect(Collectors.toList());
    }

    @UltramanLog(method = "findByCondition")
    public DataCollection<Map<String, Object>> findCollectionMapByCondition(final String tenantCode, final IEntityClass entityClass, final ConditionQueryRequest request) {
        final DataCollection<EntityInstance> dataCollection = tenantAwareBusinessFacade.findByCondition(tenantCode, entityClass, ExpFactory.createFrom(defaultPageIfNull(request)));
        return new DataCollection(dataCollection.getRowNum(),
            dataCollection.getRows().stream().map(a -> a.getRecord().toMap(null)).collect(Collectors.toList()));
    }

    @UltramanLog(method = "findByCondition")
    public DataCollection<Map<String, Object>> findCollectionMapByCondition(final String tenantCode, final IEntityClass entityClass, final ExpRel expRel) {
        final DataCollection<EntityInstance> dataCollection = tenantAwareBusinessFacade.findByCondition(tenantCode, entityClass, defaultPageIfNull((ExpQuery) expRel));
        return new DataCollection(dataCollection.getRowNum(),
            dataCollection.getRows().stream().map(a -> a.getRecord().toMap(null)).collect(Collectors.toList()));
    }

    @UltramanLog(method = "findByCondition")
    public DataCollection<Map<String, Object>> findCollectionMapByIds(final String tenantCode, final IEntityClass entityClass, final List<Long> ids) {
        final RequestBuilder builder = new RequestBuilder().field("id", ConditionOp.in, ids);
        builder.pageSize(ids.size()).pageNo(1);
        final DataCollection<EntityInstance> dataCollection = tenantAwareBusinessFacade.findByCondition(tenantCode, entityClass, ExpFactory.createFrom(builder.build()));
        return new DataCollection(dataCollection.getRowNum(),
            dataCollection.getRows().stream().map(a -> a.getRecord().toMap(null)).collect(Collectors.toList()));
    }

    @UltramanLog(method = "findOne")
    public <T> T findOneById(final String tenantCode, final IEntityClass entityClass, final Class<T> entityClazz, final Long id) {
        return fromOqsMap(entityClazz, findOneMapById(tenantCode, entityClass, id));
    }

    @UltramanLog(method = "findOne")
    public Map<String, Object> findOneMapById(final String tenantCode, final IEntityClass entityClass, final Long id) {
        final var instanceOptional = tenantAwareBusinessFacade.findOne(tenantCode, EntityId.of(entityClass, id));
        return instanceOptional.map(entityInstance -> entityInstance.getRecord().toMap(null)).orElse(null);
    }

    @UltramanLog(method = "findOneByKey")
    public <T> T findOneByKeys(final String tenantCode, final IEntityClass entityClass, final Class<T> entityClazz, final Tuple2<String, Object>... keyValues) {
        final var instanceOptional = tenantAwareBusinessFacade.findOneByKey(tenantCode, EntityKey.of(entityClass, keyValues));
        return instanceOptional.map(entityInstance -> fromOqsMap(entityClazz, entityInstance.getRecord().toMap(null))).orElse(null);
    }

    @UltramanLog(method = "findOneByKey")
    public Map<String, Object> findOneMapByKeys(final String tenantCode, final IEntityClass entityClass, final Tuple2<String, Object>... keyValues) {
        final var instanceOptional = tenantAwareBusinessFacade.findOneByKey(tenantCode, EntityKey.of(entityClass, keyValues));
        return instanceOptional.map(entityInstance -> entityInstance.getRecord().toMap(null)).orElse(null);
    }

    private <T> T fromOqsMap(final Class<T> entityClazz, final Map<String, Object> map) {
        return MapUtils.isEmpty(map) ? null : ReflectUtil.invoke(ReflectUtil.newInstance(entityClazz), "fromOQSMap", map);
    }

    private <T> Map<String, Object> toOqsMap(final T model) {
        return null == model ? Collections.emptyMap() : ReflectUtil.invoke(model, "toOQSMap");
    }

    public IEntityClass load(final String entityCode, final String tenantCode) {
        return tenantAwareBusinessFacade.load(entityCode, tenantCode);
    }

    private ConditionQueryRequest defaultPageIfNull(final ConditionQueryRequest request) {
        if (request.getPageNo() == null) {
            request.setPageNo(1);
        }
        if (request.getPageSize() == null) {
            request.setPageSize(ultramanStoreProperties.getDefaultPageSize());
        }
        return request;
    }

    private ExpQuery defaultPageIfNull(final ExpQuery query) {
        if (null == query.getRange()) {
            query.setRange(new ExpRange(1, ultramanStoreProperties.getDefaultPageSize()));
        }
        return query;
    }

}
