package com.xforceplus.janus.framework.event;

import com.xforceplus.apollo.client.netty.MCFactory;
import com.xforceplus.apollo.client.utils.SealedMessageBuilder;
import com.xforceplus.apollo.msg.SealedMessage;
import com.xforceplus.apollo.utils.Constants;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.config.BeanFactoryPostProcessor;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;
import org.springframework.context.event.EventListener;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import lombok.extern.slf4j.Slf4j;

@Slf4j
@Component
public class SealedMessageEventBus implements BeanFactoryPostProcessor {

    @Autowired
    DefaultSealedMessageEventListener defaultSealedMessageEventListener;

    ConcurrentHashMap<String, ISealedMessageEventListener> EVENT_HANDLERS = new ConcurrentHashMap<>();

    public void register(String eventType, ISealedMessageEventListener sealedMessageEventListener) {
        if (StringUtils.isBlank(eventType)) {
            log.warn("SealedMessageEvent事件处理器名称不能为空！！！");
        }
        if (EVENT_HANDLERS.containsKey(eventType)) {
            log.warn("{}的SealedMessageEvent事件处理器已存在，将替换最新的处理器！！！");
        }
        EVENT_HANDLERS.put(eventType, sealedMessageEventListener);
        log.info("{}的SealedMessageEvent事件处理器注册成功！");
    }

    public void clear() {
        EVENT_HANDLERS.clear();
        log.warn("所有的的SealedMessageEvent事件处理器已注销！！！");
    }

    @Async(value = SealedMessageEventAsyncPoolConfig.TASK_EXECUTOR_NAME)
//    @TransactionalEventListener(phase = TransactionPhase.AFTER_COMMIT, classes = SealedMessageEvent.class)
    @EventListener(classes = SealedMessageEvent.class)
    public void sealedMessageEvent(final SealedMessageEvent event) {

        ISealedMessageEventListener sealedMessageEventListener = EVENT_HANDLERS.get(event.getRequestName());
        if (null == sealedMessageEventListener) {
            sealedMessageEventListener = EVENT_HANDLERS.get("*");
            if (null == sealedMessageEventListener) {
                sealedMessageEventListener = defaultSealedMessageEventListener;
            }
        }
        AckTuple<Boolean, String> ackTuple = sealedMessageEventListener.onEvent(event);
        if (null == ackTuple) {
            log.error("缺失有效回执，请完善回执上报！！！");
            return;
        }
        SealedMessage sealedMessage = event.getSealedMessage();
        if (sealedMessage.getHeader().getRequestName().contains(Constants.REQUEST_RECEIPT) || sealedMessage.getHeader().getRequestName().contains(Constants.REQUEST_KEEP_ALIVED)) {
            return;
        }
        try {
            MCFactory.getInstance().sendMessage(SealedMessageBuilder.buildReceiptMessage(sealedMessage, ackTuple.getStatus(), ackTuple.getMessage()));
        } catch (Exception e) {
            e.printStackTrace();
            log.error("发送回执异常:{}", e);
        }
    }

    @Override
    public void postProcessBeanFactory(ConfigurableListableBeanFactory configurableListableBeanFactory) throws BeansException {
        Map<String, Object> beansWithAnnotationMap = configurableListableBeanFactory.getBeansWithAnnotation(SealedMessageEventListener.class);
        if (null != beansWithAnnotationMap) {
            beansWithAnnotationMap.forEach((k, v) -> {
                BeanDefinition beanDefinition = configurableListableBeanFactory.getBeanDefinition(k);
                String requestName = v.getClass().getAnnotation(SealedMessageEventListener.class).requestName();
                register(requestName, (ISealedMessageEventListener) v);
            });
        }
    }
}
