package com.xforceplus.janus.generator.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.xforceplus.janus.common.constants.Constant;
import com.xforceplus.janus.common.text.Convert;
import com.xforceplus.janus.common.utils.StringUtils;
import com.xforceplus.janus.commons.util.PageUtils;
import com.xforceplus.janus.generator.domain.GenTable;
import com.xforceplus.janus.generator.domain.GenTableColumn;
import com.xforceplus.janus.generator.mapper.GenTableColumnMapper;
import com.xforceplus.janus.generator.mapper.GenTableMapper;
import com.xforceplus.janus.generator.param.GenTableParam;
import com.xforceplus.janus.generator.service.IGenTableColumnService;
import com.xforceplus.janus.generator.service.IGenTableService;
import com.xforceplus.janus.generator.util.GenUtils;
import com.xforceplus.janus.generator.util.VelocityInitializer;
import com.xforceplus.janus.generator.util.VelocityUtils;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.io.IOUtils;
import org.apache.velocity.Template;
import org.apache.velocity.VelocityContext;
import org.apache.velocity.app.Velocity;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import lombok.extern.slf4j.Slf4j;

/**
 * 业务 服务层实现
 *
 * @author just
 */
@Slf4j
@Component
public class GenTableServiceImpl extends ServiceImpl<GenTableMapper, GenTable> implements IGenTableService {

    @Autowired
    private GenTableMapper genTableMapper;

    @Autowired
    private GenTableColumnMapper genTableColumnMapper;

    @Autowired
    private IGenTableColumnService columnService;

    /**
     * 查询业务信息
     *
     * @param id 业务ID
     * @return 业务信息
     */
    public GenTable selectGenTableById(Long id) {
        GenTable genTable = genTableMapper.selectById(id);
        return genTable;
    }

    /**
     * 查询业务列表
     *
     * @param genTable 业务信息
     * @return 业务集合
     */
    @Override
    public List<GenTable> selectGenTableList(GenTable genTable) {

        return this.list(new QueryWrapper<GenTable>()
                .eq(Constant.IS_VALID, Constant.IS_VALIDED)
                .eq(StringUtils.isNotBlank(genTable.getProjectId()), "project_id", genTable.getProjectId())
                .and(StringUtils.isNotBlank(genTable.getBusinessName()) && StringUtils.isNotBlank(genTable.getTableComment()), w -> w.like(StringUtils.isNotBlank(genTable.getBusinessName()), "business_name", genTable.getBusinessName())
                        .or().like(StringUtils.isNotBlank(genTable.getTableComment()), "table_comment", genTable.getTableComment()))
        );
    }

    /**
     * 查询据库列表
     *
     * @param genTable 业务信息
     * @return 数据库表集合
     */
    @Override
    public List<GenTable> selectDbTableList(GenTable genTable) {
        return genTableMapper.selectDbTableList(genTable);
    }

    /**
     * 查询据库列表
     *
     * @param tableNames 表名称组
     * @return 数据库表集合
     */
    @Override
    public List<GenTable> selectDbTableListByNames(String[] tableNames) {
        return genTableMapper.selectDbTableListByNames(tableNames);
    }

    /**
     * 修改业务
     *
     * @param genTable 业务信息
     * @return 结果
     */
    @Override
    @Transactional
    public void updateGenTable(GenTable genTable) {
        int row = genTableMapper.updateById(genTable);
        if (row > 0) {
            for (GenTableColumn cenTableColumn : genTable.getColumns()) {
                columnService.updateById(cenTableColumn);
            }
        }
    }

    /**
     * 删除业务对象
     *
     * @param ids 需要删除的数据ID
     * @return 结果
     */
    @Override
    @Transactional
    public void deleteGenTableByIds(String ids) {
        genTableMapper.deleteBatchIds(Arrays.asList(Convert.toLongArray(ids)));
        genTableColumnMapper.deleteGenTableColumnByIds(Convert.toLongArray(ids));
    }

    @Override
    public void deleteGenTableById(long id) {
        GenTable genTable = new GenTable();
        genTable.setTableId(id);
        genTable.setIsValid(Constant.IS_NOT_VALIDED);
        this.updateById(genTable);
        this.columnService.deleteByTableId(id);

    }

    /**
     * 导入表结构
     *
     * @param tableList 导入表列表
     * @param operName  操作人员
     */
    @Override
    @Transactional
    public void importGenTable(List<GenTable> tableList, String operName) {
        for (GenTable table : tableList) {
            try {
                String tableName = table.getTableName();
                GenUtils.initTable(table, operName);
                int row = genTableMapper.insert(table);
                if (row > 0) {
                    // 保存列信息
                    List<GenTableColumn> genTableColumns = genTableColumnMapper.selectDbTableColumnsByName(tableName);
                    for (GenTableColumn column : genTableColumns) {
                        GenUtils.initColumnField(column, table);
                    }

                    columnService.saveBatch(genTableColumns);
                }
            } catch (Exception e) {
                log.error("表名 " + table.getTableName() + " 导入失败：", e);
            }
        }
    }

    /**
     * 预览代码
     *
     * @param tableId 表编号
     * @return 预览数据列表
     */
    @Override
    public Map<String, String> previewCode(Long tableId) {
        Map<String, String> dataMap = new LinkedHashMap<>();
        // 查询表信息
        GenTable table = genTableMapper.selectById(tableId);
        //
        List<GenTableColumn> columns = columnService.selectGenTableColumnListByTableId(tableId);
        table.setColumns(columns);
        // 查询列信息
        setPkColumn(table, columns);
        VelocityInitializer.initVelocity();

        VelocityContext context = VelocityUtils.prepareContext(table);

        // 获取模板列表
        List<String> templates = VelocityUtils.getCrudTemplateList(table.getTplCategory());
        for (String template : templates) {
            // 渲染模板
            StringWriter sw = new StringWriter();
            Template tpl = Velocity.getTemplate(template, Constant.CharSet.UTF_8);
            tpl.merge(context, sw);
            dataMap.put(template, sw.toString());
        }
        return dataMap;
    }

    /**
     * 生成代码
     *
     * @param tableName 表名称
     * @return 数据
     */
    @Override
    public byte[] generatorCode(String tableName, String projectId, String projectCode) throws IOException {
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        ZipOutputStream zip = new ZipOutputStream(outputStream);
        generatorCode(tableName, projectId, projectCode, zip);
        IOUtils.closeQuietly(zip);
        return outputStream.toByteArray();
    }

    /**
     * 批量生成代码
     *
     * @param tableNames 表数组
     * @return 数据
     */
    @Override
    public byte[] generatorCode(String[] tableNames, String projectId, String projectCode) throws IOException {
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        ZipOutputStream zip = new ZipOutputStream(outputStream);
        for (String tableName : tableNames) {
            generatorCode(tableName, projectId, projectCode, zip);
        }
        IOUtils.closeQuietly(zip);
        return outputStream.toByteArray();
    }

    /**
     * 查询表信息并生成代码
     */
    @Override
    public void generatorCode(String tableName, String projectId, String projectCode, ZipOutputStream zip) throws IOException {
        // 查询表信息
        GenTable table = genTableMapper.selectGenTableByName(tableName, projectId);
        // 查询列信息
        List<GenTableColumn> columns = table.getColumns();
        setPkColumn(table, columns);

        // 设置主子表信息
//        setSubTable(table);

        VelocityInitializer.initVelocity();

        VelocityContext context = VelocityUtils.prepareContext(table);
        context.put("projectCode", projectCode);

        // 获取模板列表
        List<String> templates = VelocityUtils.getCrudTemplateList(table.getTplCategory());
        for (String template : templates) {
            // 渲染模板
            StringWriter sw = new StringWriter();
            Template tpl = Velocity.getTemplate(template, Constant.CharSet.UTF_8);
            tpl.merge(context, sw);
            try {
                // 添加到zip
                zip.putNextEntry(new ZipEntry(VelocityUtils.getFileName(template, table, projectCode)));
                IOUtils.write(sw.toString(), zip, Constant.CharSet.UTF_8);
                IOUtils.closeQuietly(sw);
                zip.flush();
                zip.closeEntry();
            } catch (IOException e) {
                log.error("渲染模板失败，表名：" + table.getTableName(), e);
                throw e;
            }
        }


//        if (table.isSub()) {
//            generatorCode(table.getSubTableName(), projectId, projectCode, zip);
//        }


    }
    //TODO 后续查看是否有影响
//    public void setSubTable(GenTable table) {
//        String subTableName = table.getSubTableName();
//        if (StringUtils.isNotEmpty(subTableName)) {
//            GenTable subTbl = genTableMapper.selectGenTableByName(subTableName, table.getProjectId());
//            table.setSubTable(subTbl);
//
//        }
//    }


    /**
     * 设置主键列信息
     *
     * @param table   业务表信息
     * @param columns 业务字段列表
     */
    public void setPkColumn(GenTable table, List<GenTableColumn> columns) {
        for (GenTableColumn column : columns) {
            if (column.isPk()) {
                table.setPkColumn(column);
                break;
            }
        }
        if (table.getPkColumn() == null) {
            table.setPkColumn(columns.get(0));
        }
    }


    @Transactional(rollbackFor = Exception.class)
    @Override
    public boolean saveOrUpdateGenTable(GenTable genTable) {
        if (StringUtils.isBlank(genTable.getModuleName())) {
            genTable.setModuleName(GenTable.MODULE_NAME);
        }

        this.saveTbl(genTable);

        if (CollectionUtils.isNotEmpty(genTable.getSubTables())) {
            for(GenTable subtable:genTable.getSubTables()){
                subtable.setProjectId(genTable.getProjectId());
                this.saveTbl(subtable);
            }
        }

        return true;
    }

    private void saveTbl(GenTable genTable) {

        GenTable existTb = findByTbNameProjectId(genTable.getTableName(), genTable.getProjectId());
        if (existTb != null) {
            genTable.setTableId(existTb.getTableId());
        }

        if(StringUtils.isBlank(genTable.getClassName())){
            genTable.setClassName(StringUtils.convertToCamelCase(genTable.getTableName()));
        }

        if (StringUtils.isNotBlank(genTable.getClassName()) && StringUtils.isBlank(genTable.getBusinessName())) {
            genTable.setBusinessName(StringUtils.capitalize(genTable.getClassName()));
        }

        this.saveOrUpdate(genTable);

        List<GenTableColumn> existColumns = columnService.selectGenTableColumnListByTableId(genTable.getTableId());

        List<GenTableColumn> saveOrUpdateList = new ArrayList<>();

        if (CollectionUtils.isNotEmpty(existColumns)) {
            Map<String, GenTableColumn> fromMap = genTable.getColumns().stream().collect(Collectors.toMap(GenTableColumn::getColumnName, tableColumn -> tableColumn));
            existColumns.forEach(column -> {
                if (!fromMap.containsKey(column.getColumnName())) {
                    column.setIsValid(Constant.IS_NOT_VALIDED);
                    saveOrUpdateList.add(column);
                } else {
                    fromMap.get(column.getColumnName()).setColumnId(column.getColumnId());
                }
            });
            saveOrUpdateList.addAll(fromMap.values());
        } else {
            saveOrUpdateList.addAll(genTable.getColumns());
        }

        if (CollectionUtils.isNotEmpty(saveOrUpdateList)) {
            saveOrUpdateList.forEach(column -> {
                column.setTableId(genTable.getTableId());
                if(StringUtils.isBlank(column.getJavaField())){
                    column.setJavaField(StringUtils.toCamelCase(column.getColumnName()));
                }
            });
            columnService.saveOrUpdateBatch(saveOrUpdateList);
        }
    }

    @Override
    public GenTable selectSimpleGenTableName(String tableName, String projectId) {
        GenTable table = this.getOne(new QueryWrapper<GenTable>()
                .eq("table_name", tableName)
                .eq(StringUtils.isNotBlank(projectId), "project_id", projectId)
                .eq(Constant.IS_VALID, Constant.IS_VALIDED));
        return table;
    }

    @Override
    public GenTable selectGenTableName(String tableName, String projectId) {

        GenTable table = findByTbNameProjectId(tableName, projectId);

        if (table == null) {
            return null;
        }
        List<GenTableColumn> columns = columnService.selectGenTableColumnListByTableId(table.getTableId());
        table.setColumns(columns);
//        setSubTable(table);
        return table;
    }

    @Override
    public GenTable findByTbNameProjectId(String tbName, String projectId) {
        GenTable table = this.getOne(new QueryWrapper<GenTable>()
                .eq("table_name", tbName)
                .eq(StringUtils.isNotBlank(projectId), "project_id", projectId)
                .eq(Constant.IS_VALID, Constant.IS_VALIDED));
        return table;
    }

    @Override
    public PageUtils queryPager(GenTableParam param) {
        IPage<GenTable> pageParam =new Page<GenTable>(param.getPage(),param.getSize());
        IPage<GenTable> page = this.page(pageParam,
                new QueryWrapper<GenTable>()
                        .eq(StringUtils.isNotBlank(param.getProjectId()), "project_id", param.getProjectId())
                        .like(StringUtils.isNotBlank(param.getTableName()), "table_name", param.getTableName())
                        .like(StringUtils.isNotBlank(param.getBusinessName()), "business_name", param.getBusinessName())
                        .like(StringUtils.isNotBlank(param.getTableComment()), "table_comment", param.getTableComment())
                        .eq(Constant.IS_VALID, Constant.IS_VALIDED)
        );
        return new PageUtils(page);
    }

    @Override
    public GenTable getRichById(String id) {
        GenTable table = this.getById(id);

        if (table == null) {
            return null;
        }
        List<GenTableColumn> columns = columnService.selectGenTableColumnListByTableId(table.getTableId());
        table.setColumns(columns);
//        setSubTable(table);
        return table;
    }

    @Override
    public GenTable getRichByBusinessName(String businessName) {
        GenTable table = this.getOne(new QueryWrapper<GenTable>()
                .eq("business_name", businessName)
                .eq(Constant.IS_VALID, Constant.IS_VALIDED));
        if (table == null) {
            return null;
        }
        List<GenTableColumn> columns = columnService.selectGenTableColumnListByTableId(table.getTableId());
        table.setColumns(columns);
        return table;
    }


}