package com.xforceplus.janus.generator.util;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ValueNode;
import com.xforceplus.apollo.utils.JacksonUtil;
import com.xforceplus.janus.generator.constant.GenConstants;
import com.xforceplus.janus.generator.dto.ModelPropertyDto;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * @Author: xuchuanhou
 * @Date:2021/10/21下午7:05
 */
public class ModelGenTool {

    public static final String ROOT_KEY = "root";

    public static List<ModelPropertyDto> jsonToModelProperty(JsonNode jsonObject, String parentKey) {

        //数组类型默认增加到root 节点下
        if (jsonObject.isArray()) {
            Map<String, Object> root = new HashMap<>();
            root.put(ROOT_KEY, jsonObject);
            String js = JacksonUtil.getInstance().toJson(root);
            jsonObject = JacksonUtil.getInstance().fromJson(js, JsonNode.class);
        }

        List<ModelPropertyDto> propertyList = new ArrayList<>();

        Iterator<Map.Entry<String, JsonNode>> it = jsonObject.fields();
        while (it.hasNext()) {
            Map.Entry<String, JsonNode> entry = it.next();
            String name = entry.getKey();
            JsonNode jv = entry.getValue();
            String key = StringUtils.isBlank(parentKey) ? name : parentKey + "/" + name;

            if (jv.isArray()) {
                ModelPropertyDto arry = new ModelPropertyDto(name, GenConstants.TYPE_ARRAY, null, key);
                if (jv.size() > 0 ){
                    if(jv.get(0).isObject()) {
                        List<ModelPropertyDto> children = jsonToModelProperty(jv.get(0), arry.getKey());
                        arry.setChildren(children);
                    }else if(jv.get(0) instanceof ValueNode){
                        arry.setDeValue(jv.get(0).asText());
                    }
                }
                propertyList.add(arry);

            } else if (jv.isObject()) {
                ModelPropertyDto obj = new ModelPropertyDto(name, GenConstants.TYPE_OBJ, null, key);
                List<ModelPropertyDto> children = jsonToModelProperty(jv, obj.getKey());
                obj.setChildren(children);
                propertyList.add(obj);
            } else {
                String type = GenConstants.TYPE_STRING;
                if (jv.isBoolean()) {
                    type = GenConstants.TYPE_BOOLEAN;
                } else if (jv.isNumber()) {
                    type = GenConstants.TYPE_NUMBER;
                }
                ModelPropertyDto
                        field = new ModelPropertyDto(name, type, jv.textValue(), key);
                propertyList.add(field);
            }
        }

        return propertyList;
    }


    /**
     * 向来源属性中补全字段说明
     *
     * @author xucuanhou
     * @date 2021/10/26
     */
    public static void mergeProperty(List<ModelPropertyDto> sources, List<ModelPropertyDto> cmpProperties) {
        Map<String, ModelPropertyDto> sourcesMap = new HashMap<>();
        Map<String, ModelPropertyDto> cmpMap = new HashMap<>();
        if (CollectionUtils.isNotEmpty(sources) && CollectionUtils.isNotEmpty(cmpProperties)) {
            sources.forEach(field -> {
                treeToMap(field, sourcesMap, null);
            });
            cmpProperties.forEach(field -> {
                treeToMap(field, cmpMap, null);
            });
            for (Map.Entry<String, ModelPropertyDto> entry : sourcesMap.entrySet()) {
                if (cmpMap.containsKey(entry.getKey())) {
                    ModelPropertyDto exist = cmpMap.get(entry.getKey());
                    entry.getValue().setType(exist.getType());
                    entry.getValue().setDescribe(exist.getDescribe());
                    entry.getValue().setRefClassName(exist.getRefClassName());
                    entry.getValue().setMustFill(exist.getMustFill());
                    entry.getValue().setLen(exist.getLen());
                }
            }
        }
    }

    protected static void treeToMap(ModelPropertyDto propertyDto, Map<String, ModelPropertyDto> maps, String treePath) {
        String key = StringUtils.isBlank(treePath) ? propertyDto.getName() : treePath + "/" + propertyDto.getName();
        maps.put(key, propertyDto);
        if (CollectionUtils.isNotEmpty(propertyDto.getChildren())) {
            propertyDto.getChildren().forEach(field -> {
                treeToMap(field, maps, key);
            });
        }
    }


    /**
     * fieldList to JsonStr
     *
     * @author xucuanhou
     * @date 2021/10/27
     */
    public static String converField2JsonStr(List<ModelPropertyDto> fields) {
        Map<String, Object> jsonMap = converField2Json(fields);
        if (jsonMap.containsKey(ROOT_KEY)) {
            return JacksonUtil.getInstance().toJson(jsonMap.get(ROOT_KEY));
        }
        return JacksonUtil.getInstance().toJson(jsonMap);
    }

    /**
     * 将字段定义转换到json mock 样例
     *
     * @author xucuanhou
     * @date 2021/10/27
     */
    private static Map<String, Object> converField2Json(List<ModelPropertyDto> fields) {
        Map<String, Object> jsonMap = new HashMap<>();
        for (ModelPropertyDto field : fields) {
            switch (field.getType()) {
                case GenConstants.TYPE_OBJ:
                    jsonMap.put(field.getName(), converField2Json(field.getChildren()));
                    break;
                case GenConstants.TYPE_ARRAY:
                    List<Map<String, Object>> jsonMapList = new ArrayList<>();
                    if(CollectionUtils.isNotEmpty(field.getChildren())){
                        jsonMapList.add(converField2Json(field.getChildren()));
                    }
                    jsonMap.put(field.getName(), jsonMapList);
                    break;
                case GenConstants.TYPE_NUMBER:
                    try {
                        jsonMap.put(field.getName(), StringUtils.isNotBlank(field.getDeValue()) ? new BigDecimal(field.getDeValue()) : 0);
                    } catch (Exception ex) {
                        jsonMap.put(field.getName(), false);
                    }
                    break;
                case GenConstants.TYPE_BOOLEAN:
                    try {
                        jsonMap.put(field.getName(), StringUtils.isNotBlank(field.getDeValue()) ? Boolean.valueOf(field.getDeValue()) : false);
                    } catch (Exception ex) {
                        jsonMap.put(field.getName(), 0);
                    }
                    break;
                case GenConstants.TYPE_STRING:
                    jsonMap.put(field.getName(), field.getDeValue());
                    break;
                default:
                    break;
            }
        }

        return jsonMap;
    }


}
