package com.xforceplus.janus.generator.util;

import com.xforceplus.janus.common.constants.Constant;
import com.xforceplus.janus.generator.domain.ApiModel;
import com.xforceplus.janus.generator.domain.ApisInfo;
import com.xforceplus.janus.generator.domain.MddlwareInsDto;
import com.xforceplus.janus.generator.service.IGenTableService;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.velocity.Template;
import org.apache.velocity.VelocityContext;
import org.apache.velocity.app.Velocity;

import java.io.IOException;
import java.io.StringWriter;
import java.util.List;
import java.util.Set;
import java.util.zip.ZipEntry;

import lombok.Data;
import lombok.extern.slf4j.Slf4j;

@Slf4j
@Data
public class VelocityBusiness {

    private ZipUtils zipUtils;
    /**netty 配置*/
    private String clientId;
    private String tcpUrl;
    private Integer tcpPort;

    private String gatewayUrl;
    /**获取拉取配置信息Action*/
    private String fetchJanusConfigAction;
    private String authentication;
    private String uiaSign;
    //上传属地配置信息接口授权
    private String uploadSynConfigAction;

    public String projectCode;

    private String version;

    private boolean hasDb;
    private boolean hasRMQ;


    private IGenTableService genTableService;

    public VelocityBusiness(){
    }

    public void initFileZip(String desFile, String srcFile,String projectCode){
        try {
            this.zipUtils = new ZipUtils(desFile,projectCode);
            this.zipUtils.doCompress(srcFile);
        } catch (IOException e) {
            log.error("初始化模板文件",e);
        }
    }

    /***
     * 生成代码入口
     */
    public String generatorCode(List<ApisInfo> apisInfos, List<MddlwareInsDto> mddlwareInsDtos, Set<String> generateTbs)
        throws IOException {
        VelocityInitializer.initVelocity();

        if(StringUtils.isNotBlank(this.tcpUrl)){
            genNetty(apisInfos);
        }

        // 配置中间件
        genMddlware(mddlwareInsDtos);

        if (CollectionUtils.isNotEmpty(generateTbs)) {
            for (String tableName : generateTbs) {
                genTableService.generatorCode(tableName, this.clientId, projectCode, zipUtils.getOut());
            }
        }

        zipUtils.getOut().flush();
        zipUtils.close();
        return zipUtils.getZipFile();
    }


    private void genSoap(List<ApisInfo> apisInfos) throws IOException {
        VelocityContext context = VelocityUtils.prepareContextSoap(apisInfos);
        if (null == context) {
            return;
        }
        write(context, "vm/java/soapconfig.java.vm", "bridgehead-" + projectCode
            + "/src/main/java/com/xforceplus/janus/bridgehead/" + projectCode + "/soap/SoapConfig.java");
        write(context, "vm/java/soapservice.java.vm", "bridgehead-" + projectCode
            + "/src/main/java/com/xforceplus/janus/bridgehead/" + projectCode + "/soap/SoapService.java");
        write(context, "vm/java/soapserviceImpl.java.vm", "bridgehead-" + projectCode
            + "/src/main/java/com/xforceplus/janus/bridgehead/" + projectCode + "/soap/SoapServiceImpl.java");

    }

    private void write(VelocityContext context, String template, String javaFile) throws IOException {
        StringWriter pw = new StringWriter();
        Template tpl = Velocity.getTemplate(template, Constant.CharSet.UTF_8);
        tpl.merge(context, pw);
        // 添加到zip
        zipUtils.getOut().putNextEntry(new ZipEntry(javaFile));
        IOUtils.write(pw.toString(), zipUtils.getOut(), Constant.CharSet.UTF_8);
        IOUtils.closeQuietly(pw);
    }


    private void genNetty(List<ApisInfo> apisInfos) throws IOException {

        VelocityContext nettyContext =
            VelocityUtils.prepareContextNetty(clientId, this.tcpUrl, this.tcpPort, apisInfos);

        if (null == nettyContext) {
            return;
        }
        write(nettyContext, "vm/resources/client.properties.vm",
            "bridgehead-" + projectCode + "/src/main/resources/client.properties");

        for (ApisInfo dto : apisInfos) {
            if ("TCP".equalsIgnoreCase(dto.getProtocol())) {

                VelocityContext velocityContext = new VelocityContext();

                String requestName = dto.getAction().replaceAll("\\.", "").replaceAll("-", "").replaceAll("_", "");
                String msgEventClassName = StringUtils.capitalize(requestName) + "Event";

                velocityContext.put("ClassName", msgEventClassName);
                velocityContext.put("requestName", dto.getAction());
                velocityContext.put("projectCode", this.projectCode);
                velocityContext.put("priority", 1);
                if (CollectionUtils.isEmpty(dto.getMsgDealTypes())) {
                    //默认处理器
                    write(velocityContext, "vm/java/janusMsgEvent.java.vm",
                        "bridgehead-" + projectCode + "/src/main/java/com/xforceplus/janus/bridgehead/"
                            + projectCode + "/event/" + msgEventClassName + ".java");
                    continue;
                }

                int i = 0;
                for (String dealType : dto.getMsgDealTypes()) {
                    String classNameTml = msgEventClassName;
                    if (i != 0) {
                        classNameTml = msgEventClassName + SUFFIX[i];
                        velocityContext.put("ClassName", classNameTml);
                    }

                    velocityContext.put("priority", i);

                    if (ApisInfo.MSG_DEAL_TYPE_API.equals(dealType)) {
                        velocityContext.put("httpUrlKey", classNameTml+".url");
                        write(velocityContext, "vm/java/janusMsgInvokeApiEvent.java.vm",
                            "bridgehead-" + projectCode + "/src/main/java/com/xforceplus/janus/bridgehead/"
                                + projectCode + "/event/" + classNameTml + ".java");
                    } else if (ApisInfo.MSG_DEAL_TYPE_DB.equals(dealType)) {

                        // 生成service domain 根据表已经生成
                        String serviceClassName = "I" + requestName + "Service";
                        String domainClass = dto.getDbDomain();

                        velocityContext.put("domainClass", domainClass);
                        velocityContext.put("serviceClassName", serviceClassName);

                        write(velocityContext, "vm/java/janusMsg2DBEvent.java.vm",
                            "bridgehead-" +this.projectCode+ "/src/main/java/com/xforceplus/janus/bridgehead/"
                                + projectCode + "/event/" + classNameTml + ".java");

                        velocityContext.put("requestName", dto.getAction());
                    } else if (ApisInfo.MSG_DEAL_TYPE_RMQ.equals(dealType)) {

                        write(velocityContext, "vm/java/janusMsg2RmqEvent.java.vm",
                            "bridgehead-" +  this.projectCode + "/src/main/java/com/xforceplus/janus/bridgehead/"
                                + projectCode + "/event/" + classNameTml + ".java");
                        write(velocityContext, "vm/java/RmqProducer.java.vm",
                            "bridgehead-" + this.projectCode + "/src/main/java/com/xforceplus/janus/bridgehead/"
                                + projectCode + "/rmq/RmqProducer.java");

                    } else {
                        write(velocityContext, "vm/java/janusMsgEvent.java.vm",
                            "bridgehead-" +this.projectCode + "/src/main/java/com/xforceplus/janus/bridgehead/"
                                + projectCode + "/event/" + classNameTml + ".java");
                    }
                    i++;
                }
            }
        }
    }

    private static final String[] SUFFIX = {"A", "B", "C", "D", "E", "F", "G", "H", "I", "J", "H", "I"};

    /**
     * @param mddlwareInsDtos
     * @throws IOException
     * @author xuchuanhou
     */
    // 配置中间件属性
    public void genMddlware(List<MddlwareInsDto> mddlwareInsDtos) throws IOException {

        boolean hasDb = false;
        boolean hasRMQ = false;
        if (CollectionUtils.isNotEmpty(mddlwareInsDtos)) {
            for (MddlwareInsDto mddlware : mddlwareInsDtos) {
                if (MddlwareInsDto.MDDLWARETYPE_DB.equals(mddlware.getMddlwareType())) {
                    VelocityContext velocityContext = new VelocityContext();
                    velocityContext.put("mddlware", mddlware);

                    velocityContext.put("gatewayUrl", this.gatewayUrl);
                    velocityContext.put("authentication", this.authentication);
                    velocityContext.put("uiaSign", this.uiaSign);
                    velocityContext.put("fetchJanusConfigAction", this.fetchJanusConfigAction);
                    velocityContext.put("uploadSynConfigAction", this.uploadSynConfigAction);


                    hasDb = true;
                    write(velocityContext, "vm/resources/application-dev.yml.vm",
                        "bridgehead-" + this.projectCode + "/src/main/resources/application-dev.yml");
                } else if (MddlwareInsDto.MDDLWARETYPE_RMQ.equals(mddlware.getMddlwareType())) {
                    VelocityContext velocityContext = new VelocityContext();
                    velocityContext.put("mddlware", mddlware);
                    hasRMQ = true;
                    write(velocityContext, "vm/resources/application-amq.yml.vm",
                        "bridgehead-" +this.projectCode + "/src/main/resources/application-amq.yml");
                }
            }
        }

        VelocityContext velocityContext = new VelocityContext();
        velocityContext.put("hasDb", hasDb);
        velocityContext.put("hasRMQ", hasRMQ);
        this.hasDb=hasDb;
        this.hasRMQ=hasRMQ;

        velocityContext.put("bridgeHeadVersion", this.version);

        velocityContext.put("projectCode", this.projectCode);

        write(velocityContext, "vm/resources/pom.xml.vm", "bridgehead-" + this.projectCode+ "/pom.xml");
        write(velocityContext, "vm/java/application.java.vm", "bridgehead-" + this.projectCode
            + "/src/main/java/com/xforceplus/janus/bridgehead/" + projectCode + "/Application.java");

    }

    private boolean genModel(String name, List<ApiModel> models) throws IOException {
        VelocityContext context = VelocityUtils.prepareContextModel(name, models);
        context.put("projectCode", this.projectCode);
        write(context, "vm/java/janusmodel.java.vm", "bridgehead-" + this.projectCode
            + "/src/main/java/com/xforceplus/janus/bridgehead/" + projectCode + "/model/" + name + ".java");
        return true;
    }

    private void genService(List<ApisInfo> apisInfos) throws IOException {
        VelocityContext context = VelocityUtils.prepareContextService("Janus", apisInfos);
        if (null == context)
            return;
        context.put("janusUrl", this.gatewayUrl);
        context.put("clientId", this.clientId);
        context.put("authentication", this.authentication);
//        context.put("flowAction", this.flowAction);
        context.put("projectCode", this.projectCode);
        write(context, "vm/java/januscontroller.java.vm", "bridgehead-" + this.projectCode
            + "/src/main/java/com/xforceplus/janus/bridgehead/" + projectCode + "/controller/JanusController.java");
        write(context, "vm/java/janusservice.java.vm", "bridgehead-" + this.projectCode
            + "/src/main/java/com/xforceplus/janus/bridgehead/" + projectCode + "/service/IJanusService.java");
        write(context, "vm/java/janusserviceimpl.java.vm", "bridgehead-" +this.projectCode
            + "/src/main/java/com/xforceplus/janus/bridgehead/" + projectCode + "/service/impl/JanusServiceImpl.java");

        context.put("annnoChar", "#");
        write(context, "vm/resources/urls.properties.vm",
            "bridgehead-" + this.projectCode + "/src/main/resources/urls.properties");
    }

    /**
     * 创建支持前端的controller与aop 等
     */
    @Deprecated
    private void genFrontSupport() throws IOException {

        VelocityContext context = new VelocityContext();
        context.put("projectCode", this.projectCode);

        write(context, "vm/java/frontsupport/aspect/ControllerAspect.java.vm", "bridgehead-" + this.projectCode
            + "/src/main/java/com/xforceplus/janus/bridgehead/" + projectCode + "/aspect/ControllerAspect.java");
        write(context, "vm/java/frontsupport/controller/sys/IndexController.java.vm", "bridgehead-" + this.projectCode
            + "/src/main/java/com/xforceplus/janus/bridgehead/" + projectCode + "/controller/sys/IndexController.java");
        write(context, "vm/java/frontsupport/controller/sys/MenuController.java.vm", "bridgehead-" +this.projectCode
            + "/src/main/java/com/xforceplus/janus/bridgehead/" + projectCode + "/controller/sys/MenuController.java");
        write(context, "vm/java/frontsupport/controller/sys/RoleController.java.vm", "bridgehead-" + this.projectCode
            + "/src/main/java/com/xforceplus/janus/bridgehead/" + projectCode + "/controller/sys/RoleController.java");
        write(context, "vm/java/frontsupport/controller/sys/UserController.java.vm", "bridgehead-" + this.projectCode
            + "/src/main/java/com/xforceplus/janus/bridgehead/" + projectCode + "/controller/sys/UserController.java");

        write(context, "vm/java/frontsupport/vo/MonsterPageData.java.vm", "bridgehead-" + this.projectCode
            + "/src/main/java/com/xforceplus/janus/bridgehead/" + projectCode + "/vo/MonsterPageData.java");
        write(context, "vm/java/frontsupport/vo/MonsterResult.java.vm", "bridgehead-" + this.projectCode
            + "/src/main/java/com/xforceplus/janus/bridgehead/" + projectCode + "/vo/MonsterResult.java");

    }


    public static VelocityBusiness.Builder builder(){
        return new VelocityBusiness.Builder();
    }

    public static class Builder{
        private String clientId;
        private String tcpUrl;
        private Integer tcpPort;

        private String gatewayUrl;
        /**获取拉取配置信息Action*/
        private String fetchJanusConfigAction;
        private String authentication;
        private String uiaSign;
        //上传属地配置信息接口授权
        private String uploadSynConfigAction;
        public String projectCode;

        private boolean hasDb;
        private boolean hasRMQ;
        private String version;

        public Builder(){}

        public VelocityBusiness.Builder clientId(String clientId) { this.clientId = clientId; return this; }
        public VelocityBusiness.Builder tcpUrl(String tcpUrl) { this.tcpUrl = tcpUrl; return this; }
        public VelocityBusiness.Builder tcpPort(Integer tcpPort) { this.tcpPort = tcpPort; return this; }

        public VelocityBusiness.Builder gatewayUrl(String gatewayUrl) { this.gatewayUrl = gatewayUrl; return this; }
        public VelocityBusiness.Builder fetchJanusConfigAction(String fetchJanusConfigAction) { this.fetchJanusConfigAction = fetchJanusConfigAction; return this; }
        public VelocityBusiness.Builder authentication(String authentication) { this.authentication = authentication; return this; }
        public VelocityBusiness.Builder uiaSign(String uiaSign) { this.uiaSign = uiaSign; return this; }
        public VelocityBusiness.Builder uploadSynConfigAction(String uploadSynConfigAction) { this.uploadSynConfigAction = uploadSynConfigAction; return this; }
        public VelocityBusiness.Builder projectCode(String projectCode) { this.projectCode = projectCode; return this; }
        public VelocityBusiness.Builder hasDb(boolean hasDb) { this.hasDb = hasDb; return this; }
        public VelocityBusiness.Builder hasRMQ(boolean hasRMQ) { this.hasRMQ = hasRMQ; return this; }
        public VelocityBusiness.Builder version(String version) { this.version = version; return this; }

        public VelocityBusiness builder(){
            VelocityBusiness velocity=new VelocityBusiness();
            velocity.setClientId(this.clientId);
            velocity.setTcpUrl(this.tcpUrl);
            velocity.setTcpPort(this.tcpPort);
            velocity.setGatewayUrl(this.gatewayUrl);
            velocity.setFetchJanusConfigAction(this.fetchJanusConfigAction);
            velocity.setUploadSynConfigAction(this.uploadSynConfigAction);
            velocity.setAuthentication(this.authentication);
            velocity.setUiaSign(this.uiaSign);
            velocity.setProjectCode(projectCode);
            velocity.setHasDb(this.hasDb);
            velocity.setHasRMQ(this.hasRMQ);
            velocity.setVersion(this.version);

            return velocity;
        }

    }

}
