package com.xforceplus.janus.message.common.dto;

import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.NoArgsConstructor;
import org.springframework.util.StringUtils;

import static com.xforceplus.janus.message.common.dto.Result.ResultCode.*;

@Data
@NoArgsConstructor
@AllArgsConstructor
@ApiModel
public class Result<T> {
    private static final String MSG_PREFIX = "MSG";

    @ApiModelProperty(value = "状态码", example = "MSGCZZ0000")
    private String code;

    @ApiModelProperty(value = "消息")
    private String message;

    @ApiModelProperty(value = "返回数据")
    private T result;

    public Result(ResultCode responseCode) {
        this.code = responseCode.getCode();
        this.message = responseCode.getMessage();
    }

    public Result(T result, ResultCode responseCode) {
        this.result = result;
        this.code = responseCode.getCode();
        this.message = responseCode.getMessage();
    }

    public static <T> Result<T> ok(T body) {
        Result<T> response = new Result<>();
        response.setResult(body);
        response.setCode(OK.getCode());
        response.setMessage(OK.getMessage());
        return response;
    }

    public static <T> Result<T> ok() {
        Result<T> response = new Result<>();
        response.setCode(OK.getCode());
        response.setMessage(OK.getMessage());
        return response;
    }

    public static <T> Result<T> invalidParam(String message) {
        Result<T> response = new Result<>();
        response.setCode(INVALID_PARAM.getCode());
        response.setMessage(StringUtils.isEmpty(message) ? INVALID_PARAM.getMessage() : message);
        return response;
    }

    public static <T> Result<T> noData(String message) {
        Result<T> response = new Result<>();
        response.setCode(NO_DATA.getCode());
        response.setMessage(StringUtils.isEmpty(message) ? NO_DATA.getMessage() : message);
        return response;
    }

    public static <T> Result<T> serverError(String message) {
        Result<T> response = new Result<>();
        response.setCode(INTERNAL_SERVER_ERROR.getCode());
        response.setMessage(StringUtils.isEmpty(message) ? INTERNAL_SERVER_ERROR.getMessage() : message);
        return response;
    }

    public static <T> Result<T> invocationTimeout(String message) {
        Result<T> response = new Result<>();
        response.setCode(INVOCATION_TIMEOUT.getCode());
        response.setMessage(StringUtils.isEmpty(message) ? INVOCATION_TIMEOUT.getMessage() : message);
        return response;
    }

    public static <T> Result<T> invocationError(String message) {
        Result<T> response = new Result<>();
        response.setCode(INVOCATION_ERROR.getCode());
        response.setMessage(StringUtils.isEmpty(message) ? INVOCATION_ERROR.getMessage() : message);
        return response;
    }

    public enum ResultCode {
        // ---------------------------
        // 0000       处理成功
        // 1000～1999 区间表示参数错误
        // 2000～2999 区间表示用户错误
        // 3000～3999 区间表示系统业务异常
        // ---------------------------

        OK(MSG_PREFIX + "CZZ0000", "处理成功"),
        INVALID_PARAM(MSG_PREFIX + "CZZ1001", "非法参数"),
        EMPTY_PARAM(MSG_PREFIX + "CZZ1002", "参数为空"),
        MISSING_PARAM(MSG_PREFIX + "CZZ1003", "参数缺失"),
        NO_DATA(MSG_PREFIX + "CZZ1010", "查询不到该记录"),
        EXISTING_DATA(MSG_PREFIX + "CZZ1011", "记录已存在"),

        NOT_LOGIN(MSG_PREFIX + "CZZ2001", "用户未登录，访问路径需要验证"),
        LOGIN_FAILED(MSG_PREFIX + "CZZ2002", "账号不存在或密码错误"),
        DISABLED_ACCOUNT(MSG_PREFIX + "CZZ2003", "账号被禁用"),
        USER_NOT_EXISTS(MSG_PREFIX + "CZZ2004", "用户不存在"),
        MISSING_USER(MSG_PREFIX + "CZZ2005", "用户已经存在"),

        INTERNAL_SERVER_ERROR(MSG_PREFIX + "CZZ3001", "服务器内部错误"),
        CONCURRENCY_ERROR(MSG_PREFIX + "CZZ3002", "多线程并发执行错误"),
        INVOCATION_TIMEOUT(MSG_PREFIX + "CZZ3003", "第三方调用超时"),
        INVOCATION_ERROR(MSG_PREFIX + "CZZ3004", "第三方调用异常");

        private final String code;
        private final String message;

        ResultCode(String code, String message) {
            this.code = code;
            this.message = message;
        }

        public String getCode() {
            return code;
        }

        public String getMessage() {
            return message;
        }
    }
}
