package com.xforceplus.janus.message.common.utils;

import com.xforceplus.janus.message.common.dto.MessageAck;

import io.jsonwebtoken.Claims;
import io.jsonwebtoken.Jws;
import io.jsonwebtoken.Jwts;
import io.jsonwebtoken.SignatureAlgorithm;
import lombok.extern.slf4j.Slf4j;

import org.apache.commons.lang3.StringUtils;

import javax.servlet.http.HttpServletRequest;

import java.util.Date;
import java.util.concurrent.TimeUnit;

/**
 * JWT utility
 *
 * @author liuyao@xforceplus.com
 * @version 1.0
 * @date 2021/3/8 14:13
 */
@Slf4j
public class JwtUtils {
    public static final String HEADER_MB_TOKEN_NAME = "MB_TOKEN";
    /**
     * Expired time(After 50 years)
     */
    public static final long EXPIRED_TIME = TimeUnit.DAYS.toMillis(365 * 50);
    public static final String SIGN_KEY = "ukc8BDbRigUDaY6pZFfWus2jZWLPHO17L8s";

    /**
     * Generate JWT token
     *
     * @param appKey appKey
     * @return JWT token
     */
    public static String generateJwtToken(String appKey, String appSecret) {
        return Jwts.builder()
                .setHeaderParam("typ", "JWT")
                .setHeaderParam("alg", "HS256")
                .setSubject("message-bus")
                .setIssuedAt(new Date())
                .setExpiration(new Date(System.currentTimeMillis() + EXPIRED_TIME))
                .claim("appKey", appKey)
                .claim("appSecret", appSecret)
                .signWith(SignatureAlgorithm.HS256, SIGN_KEY)
                .compact();
    }

    /**
     * Validate the JWT token whether it is valid
     *
     * @param request http servlet request
     * @return true: valid, false: invalid
     */
    public static boolean validateToken(HttpServletRequest request) {
        return validateToken(request.getHeader(HEADER_MB_TOKEN_NAME));
    }

    /**
     * Validate the JWT token whether it is valid
     *
     * @param jwtToken JWT token
     * @return true: valid, false: invalid
     */
    private static boolean validateToken(String jwtToken) {
        if (StringUtils.isEmpty(jwtToken)) {
            return false;
        }
        try {
            Jwts.parser().setSigningKey(SIGN_KEY).parseClaimsJws(jwtToken);
        } catch (Exception e) {
            log.error("Failed to validate JWT token!", e);
            return false;
        }
        return true;
    }

    /**
     * Get app key from JWT token
     *
     * @param jwtToken JWT token
     * @return app key value
     */
    public static String getAppKeyFromJwtToken(String jwtToken) {
        if (StringUtils.isEmpty(jwtToken)) {
            return StringUtils.EMPTY;
        }
        try {
            Jws<Claims> claimsJws = Jwts.parser().setSigningKey(SIGN_KEY).parseClaimsJws(jwtToken);
            Claims claims = claimsJws.getBody();
            return (String) claims.get("appKey");
        } catch (Exception e) {
            log.error("Failed to get app key from JWT token!", e);
            return StringUtils.EMPTY;
        }
    }


    /**
     * Generate JWT receipt String
     *
     * @param topicName
     * @param groupId
     * @param receiptHandle
     * @return MB receiptHandle
     */
    public static String generateMsgReceipt(String topicName, String groupId, String receiptHandle) {
        return Jwts.builder()
                .setHeaderParam("typ", "JWT")
                .setHeaderParam("alg", "HS256")
                .setSubject("message-bus")
                .setIssuedAt(new Date())
                .setExpiration(new Date(System.currentTimeMillis() + EXPIRED_TIME))
                .claim("topicName", topicName)
                .claim("groupId", groupId)
                .claim("receiptHandle", receiptHandle)
                .signWith(SignatureAlgorithm.HS256, SIGN_KEY)
                .compact();
    }

    /**
     * @param receiptHandle JWT receiptHandle
     * @return MessageAck
     */
    public static MessageAck getMsgAck(String receiptHandle) throws IllegalArgumentException{
        if (StringUtils.isEmpty(receiptHandle)) {
            return null;
        }
        MessageAck messageAck = new MessageAck();
        Jws<Claims> claimsJws = Jwts.parser().setSigningKey(SIGN_KEY).parseClaimsJws(receiptHandle);
        Claims claims = claimsJws.getBody();
        messageAck.setTopicName((String) claims.get("topicName"));
        messageAck.setGroupId((String) claims.get("groupId"));
        messageAck.setReceiptHandle((String) claims.get("receiptHandle"));
        return messageAck;
    }

}