package com.xforceplus.janus.message.common.utils.kryo;

import com.esotericsoftware.kryo.Kryo;
import com.esotericsoftware.kryo.io.Input;
import com.esotericsoftware.kryo.io.Output;
import com.esotericsoftware.kryo.serializers.JavaSerializer;
import de.javakaffee.kryoserializers.*;

import java.lang.reflect.InvocationHandler;
import java.net.URI;
import java.text.SimpleDateFormat;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.regex.Pattern;

public class KryoSerializer {

    /**
     * Create a new {@link KryoHolder} instance for each thread
     */
    private final ThreadLocal<KryoHolder> kryoThreadLocal = ThreadLocal.withInitial(() -> new KryoHolder(new Kryo()));

    private KryoSerializer() {
    }

    public static KryoSerializer getInstance() {
        return KryoThreadSafeSerializationHolder.INSTANCE;
    }

    /**
     * Serialize input object to byte array
     *
     * @param object input object
     * @return byte array
     */
    public <T> byte[] serialize(T object) {
        KryoHolder kryoHolder = kryoThreadLocal.get();
        try {
            kryoHolder.output.clear();
            kryoHolder.kryo.writeClassAndObject(kryoHolder.output, object);
            return kryoHolder.output.toBytes();
        } finally {
            kryoHolder.output.close();
        }
    }

    /**
     * Deserialize byte array to output object
     *
     * @param bytes byte array
     * @return output object
     */
    @SuppressWarnings("unchecked")
    public <T> T deserialize(byte[] bytes) {
        KryoHolder kryoHolder = kryoThreadLocal.get();
        try {
            // Call it, and then use input object, discard any array
            kryoHolder.input.setBuffer(bytes, 0, bytes.length);
            return (T) kryoHolder.kryo.readClassAndObject(kryoHolder.input);
        } finally {
            kryoHolder.input.close();
        }
    }

    private static class KryoHolder {
        private final Kryo   kryo;
        static final  int    BUFFER_SIZE = 1024;
        private final Output output      = new Output(BUFFER_SIZE, -1); //reuse
        private final Input  input       = new Input();

        KryoHolder(Kryo kryo) {
            this.kryo = kryo;

            kryo.setRegistrationRequired(false);
            registerSystemClasses(kryo);
        }
    }

    private static void registerSystemClasses(Kryo kryo) {
        kryo.addDefaultSerializer(Throwable.class, new JavaSerializer());

        // Register some known classes for performance optimization
        kryo.register(Collections.singletonList("").getClass(), new ArraysAsListSerializer());
        kryo.register(GregorianCalendar.class, new GregorianCalendarSerializer());
        kryo.register(InvocationHandler.class, new JdkProxySerializer());
        kryo.register(Pattern.class, new RegexSerializer());
        kryo.register(URI.class, new URISerializer());
        kryo.register(UUID.class, new UUIDSerializer());
        UnmodifiableCollectionsSerializer.registerSerializers(kryo);
        SynchronizedCollectionsSerializer.registerSerializers(kryo);

        kryo.register(HashMap.class);
        kryo.register(ArrayList.class);
        kryo.register(LinkedList.class);
        kryo.register(HashSet.class);
        kryo.register(TreeSet.class);
        kryo.register(Hashtable.class);
        kryo.register(Instant.class);
        kryo.register(LocalDate.class);
        kryo.register(LocalDateTime.class);
        kryo.register(ConcurrentHashMap.class);
        kryo.register(SimpleDateFormat.class);
        kryo.register(GregorianCalendar.class);
        kryo.register(Vector.class);
        kryo.register(BitSet.class);
        kryo.register(StringBuffer.class);
        kryo.register(StringBuilder.class);
        kryo.register(Object.class);
        kryo.register(Object[].class);
        kryo.register(String[].class);
        kryo.register(byte[].class);
        kryo.register(char[].class);
        kryo.register(int[].class);
        kryo.register(float[].class);
        kryo.register(double[].class);
    }

    private static class KryoThreadSafeSerializationHolder {
        private static final KryoSerializer INSTANCE = new KryoSerializer();
    }
}