package com.xforceplus.janus.message.sdk.utils;

import com.xforceplus.janus.message.sdk.core.HttpResp;

import org.apache.http.HttpEntity;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpRequest;
import org.apache.http.client.HttpRequestRetryHandler;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.config.ConnectionConfig;
import org.apache.http.config.Registry;
import org.apache.http.config.RegistryBuilder;
import org.apache.http.config.SocketConfig;
import org.apache.http.conn.socket.ConnectionSocketFactory;
import org.apache.http.conn.socket.LayeredConnectionSocketFactory;
import org.apache.http.conn.socket.PlainConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLContexts;
import org.apache.http.conn.ssl.TrustStrategy;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.message.BasicHeader;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.InterruptedIOException;
import java.net.UnknownHostException;
import java.nio.charset.Charset;
import java.security.KeyManagementException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.Map;
import java.util.Map.Entry;

import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLException;

/**
 * <p>类名: HttpUtils</p>
 * <p>描述: http请求工具类</p>
 * <p>修改时间: 2019年04月30日 上午10:12:35</p>
 *
 * @author lidongyang
 */
public class HttpUtils {

    public static String defaultEncoding = "utf-8";

    static Logger log = LoggerFactory.getLogger(HttpUtils.class);

    static CloseableHttpClient httpClient = null;


    /**
     * 发送http post请求，并返回响应实体
     *
     * @param url     访问的url
     * @param headers 请求需要添加的请求头
     * @param json    请求参数
     */
    public static HttpResp postRequest(String url, Map<String, String> headers,
                                       String json) {
        CloseableHttpClient httpClient = buildHttpClient();
        HttpPost httpPost = new HttpPost(url);

        buildHeaders(httpPost, headers);

        if (json != null) {
            httpPost.setEntity(new StringEntity(json, Charset.forName(defaultEncoding)));
        }

        return call(httpClient, httpPost);
    }


    /**
     * 发送http put请求，并返回响应实体
     *
     * @param url     访问的url
     * @param headers 请求需要添加的请求头
     * @param json    请求参数
     */
    public static HttpResp putRequest(String url, Map<String, String> headers,
                                      String json) {
        CloseableHttpClient httpClient = buildHttpClient();
        HttpPut httpPut = new HttpPut(url);

        buildHeaders(httpPut, headers);

        if (json != null) {
            httpPut.setEntity(new StringEntity(json, Charset.forName(defaultEncoding)));
        }

        return call(httpClient, httpPut);
    }


    /**
     * 发送http get请求
     *
     * @param url     请求的url
     * @param headers 请求头
     * @param params  请求的参数
     */
    public static HttpResp getRequest(String url, Map<String, String> headers, Map<String, String> params) {
        CloseableHttpClient httpClient = buildHttpClient();
        String apiUrl = url;
        if (null != params && params.size() > 0) {
            StringBuffer param = new StringBuffer();
            int i = 0;
            for (String key : params.keySet()) {
                if (i == 0) {
                    param.append("?");
                } else {
                    param.append("&");
                }
                param.append(key).append("=").append(params.get(key));
                i++;
            }
            apiUrl += param;
        }

        HttpGet httpGet = new HttpGet(apiUrl);
        buildHeaders(httpGet, headers);

        return call(httpClient, httpGet);
    }

    private static HttpResp call(CloseableHttpClient httpClient, HttpRequestBase requestBase) {
        HttpResp httpResp = new HttpResp();
        CloseableHttpResponse response = null;
        try {
            response = httpClient.execute(requestBase);
            httpResp.setHttpStatus(response.getStatusLine().getStatusCode());
        } catch (IOException e) {
            e.printStackTrace();
            try {
                response.close();
            } catch (IOException e1) {
                e1.printStackTrace();
            }
        }
        HttpEntity entity = response.getEntity();
        if (null != entity) {
            try {
                String result = EntityUtils.toString(entity, defaultEncoding);
                httpResp.setRespStr(result);
            } catch (IOException e) {
                e.printStackTrace();
            }

        }

        return httpResp;
    }

    private static void buildHeaders(HttpRequestBase requestBase, Map<String, String> headers) {
        if (null != headers && headers.size() > 0) {
            for (Entry<String, String> entry : headers.entrySet()) {
                String key = entry.getKey();
                String value = entry.getValue();
                requestBase.addHeader(new BasicHeader(key, value));
            }
        }
    }

    /**
     * 创建httpclient
     */
    public static CloseableHttpClient buildHttpClient() {
        if (httpClient == null) {
            try {
                RegistryBuilder<ConnectionSocketFactory> builder = RegistryBuilder
                        .create();
                ConnectionSocketFactory factory = new PlainConnectionSocketFactory();
                builder.register("http", factory);
                KeyStore trustStore = KeyStore.getInstance(KeyStore
                        .getDefaultType());
                SSLContext context = SSLContexts.custom().useTLS()
                        .loadTrustMaterial(trustStore, new TrustStrategy() {
                            @Override
                            public boolean isTrusted(X509Certificate[] chain,
                                                     String authType) throws CertificateException {
                                return true;
                            }
                        }).build();
                LayeredConnectionSocketFactory sslFactory = new SSLConnectionSocketFactory(
                        context,
                        SSLConnectionSocketFactory.ALLOW_ALL_HOSTNAME_VERIFIER);
                builder.register("https", sslFactory);
                Registry<ConnectionSocketFactory> registry = builder.build();
                PoolingHttpClientConnectionManager manager = new PoolingHttpClientConnectionManager(
                        registry);
                ConnectionConfig connConfig = ConnectionConfig.custom()
                        .setCharset(Charset.forName(defaultEncoding)).build();
                SocketConfig socketConfig = SocketConfig.custom()
                        .setSoTimeout(5000).build();
                manager.setDefaultConnectionConfig(connConfig);
                manager.setDefaultSocketConfig(socketConfig);
                //连接池大小
                manager.setMaxTotal(100);
                httpClient = HttpClientBuilder.create().setConnectionManager(manager).setRetryHandler(retryHandler())
                        .build();
            } catch (KeyStoreException e) {
                e.printStackTrace();
            } catch (KeyManagementException e) {
                e.printStackTrace();
            } catch (NoSuchAlgorithmException e) {
                e.printStackTrace();
            }
        }

        return httpClient;
    }

    private static HttpRequestRetryHandler retryHandler() {
        return (exception, executionCount, context) -> {
            log.info("try request: [{}]", executionCount);

            if (executionCount >= 5) {
                // Do not retry if over max retry count
                return false;
            }
            if (exception instanceof InterruptedIOException) {
                // Timeout
                return false;
            }
            if (exception instanceof UnknownHostException) {
                // Unknown host
                return false;
            }
            if (exception instanceof SSLException) {
                // SSL handshake exception
                return false;
            }
            HttpClientContext clientContext = HttpClientContext.adapt(context);
            HttpRequest request = clientContext.getRequest();
            boolean idempotent = !(request instanceof HttpEntityEnclosingRequest);
            if (idempotent) {
                // Retry if the request is considered idempotent
                return true;
            }
            return false;
        };
    }


}