package com.xforceplus.janus.message.sdk;

import com.xforceplus.apollo.utils.JacksonUtil;
import com.xforceplus.janus.message.sdk.core.AbsMbRequest;
import com.xforceplus.janus.message.sdk.core.HttpResp;
import com.xforceplus.janus.message.sdk.request.AckRequest;
import com.xforceplus.janus.message.sdk.request.PubRequest;
import com.xforceplus.janus.message.sdk.request.SubRequest;
import com.xforceplus.janus.message.sdk.response.AckResponse;
import com.xforceplus.janus.message.sdk.response.PubResponse;
import com.xforceplus.janus.message.sdk.response.SubResponse;
import com.xforceplus.janus.message.sdk.utils.HttpUtils;

import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;

/**
 * @author zhujie@xforceplus.com
 * @date 2021/5/10 9:37 AM
 **/
public class MBClient {

    static Logger log = LoggerFactory.getLogger(MBClient.class);

    private final Integer MAX_THREAD_NUM = 2;

    final Semaphore semaphore = new Semaphore(MAX_THREAD_NUM);

    private String  url;
    private String  token;
    private Integer threadNum = 1;

    private static class SingletonClassInstance {
        private static final MBClient instance = new MBClient();
    }

    private MBClient() {
    }

    public static MBClient getInstance(String url, String token) {
        MBClient mbClient = SingletonClassInstance.instance;
        if (Objects.isNull(mbClient.getUrl()) || "".equals(mbClient.getUrl())) {
            mbClient.setUrl(url);
            mbClient.setToken(token);
        }

        return mbClient;
    }

    public PubResponse pub(PubRequest request) {
        PubResponse response = new PubResponse();
        try {
            response = PubResponse.transform(exec(request));
            return response;
        } catch (Exception e) {
            e.printStackTrace();
        }

        response.setSuccess(false);
        return response;
    }

    public SubResponse sub() {
        SubResponse response = new SubResponse();
        try {
            semaphore.acquire();
            SubRequest request = new SubRequest();
            response = SubResponse.transform(exec(request));
            if (!response.getSuccess() || (response.getSuccess() && CollectionUtils.isEmpty(response.getResponseMessages()))) {
                TimeUnit.SECONDS.sleep(3);
            }
            return response;
        } catch (InterruptedException e) {
            e.printStackTrace();
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            semaphore.release();
        }
        response.setSuccess(false);
        return response;
    }

    public AckResponse ack(AckRequest request) {
        AckResponse response = new AckResponse();
        try {
            response = AckResponse.transform(exec(request));
            return response;
        } catch (Exception e) {
            e.printStackTrace();
        }
        response.setSuccess(false);
        return response;
    }

    private HttpResp exec(AbsMbRequest request) {
        String url = getUrl() + request.getRequestPath();
        Map<String, String> headers = request.getHeaders();
        if (headers == null) {
            headers = new HashMap<>();
        }
        headers.put(Constants.MB_HEADER_TOKEN, getToken());
        headers.put(Constants.MB_HEADER_CONTENT_TYPE, "application/json");
        headers.put(Constants.MB_HEADER_CLIENT_VERSION, Constants.CLIENT_VERSION);
        String requestId = request.requestId();
        headers.put(Constants.MB_HEADER_CLIENT_REQUEST_ID, requestId);

        log.info("requestId[{}]总线请求内容[{}]", requestId, request.getRequestPath(), JacksonUtil.getInstance().toJson(request));

        HttpResp resp = null;
        if (Constants.HTTP_METHOD_GET.equals(request.httpMethod())) {
            resp = HttpUtils.getRequest(url, headers, request.getParam());
        } else if (Constants.HTTP_METHOD_POST.equals(request.httpMethod())) {
            resp = HttpUtils.postRequest(url, headers, request.getParamJson());
        } else if (Constants.HTTP_METHOD_PUT.equals(request.httpMethod())) {
            resp = HttpUtils.putRequest(url, headers, request.getParamJson());
        }
        log.info("requestId[{}]调用结束response[{}]", requestId, JacksonUtil.getInstance().toJson(resp));
        return resp;

    }

    public String getUrl() {
        return url;
    }

    public void setUrl(String url) {
        this.url = url;
    }

    public String getToken() {
        return token;
    }

    public void setToken(String token) {
        this.token = token;
    }

    public Integer getThreadNum() {
        return threadNum;
    }

    public void setThreadNum(Integer threadNum) {
        if (threadNum > MAX_THREAD_NUM) {
            log.warn("最大支持[{}]个线程");
            threadNum = MAX_THREAD_NUM;
        }
        this.threadNum = threadNum;
    }
}
