package com.xforceplus.janus.message.sdk.core;


import org.apache.commons.lang3.StringUtils;

public class Result<T> {
    private static final String MSG_PREFIX = "MSG";

    private String code;

    private String message;

    private T result;

    public Result() {
    }

    public Result(ResultCode responseCode) {
        this.code = responseCode.getCode();
        this.message = responseCode.getMessage();
    }

    public Result(T result, ResultCode responseCode) {
        this.result = result;
        this.code = responseCode.getCode();
        this.message = responseCode.getMessage();
    }

    public static <T> Result<T> ok(T body) {
        Result<T> response = new Result<>();
        response.setResult(body);
        response.setCode(ResultCode.OK.getCode());
        response.setMessage(ResultCode.OK.getMessage());
        return response;
    }

    public static <T> Result<T> ok() {
        Result<T> response = new Result<>();
        response.setCode(ResultCode.OK.getCode());
        response.setMessage(ResultCode.OK.getMessage());
        return response;
    }

    public static <T> Result<T> invalidParam(String message) {
        Result<T> response = new Result<>();
        response.setCode(ResultCode.INVALID_PARAM.getCode());
        response.setMessage(StringUtils.isEmpty(message) ? ResultCode.INVALID_PARAM.getMessage() : message);
        return response;
    }

    public static <T> Result<T> noData(String message) {
        Result<T> response = new Result<>();
        response.setCode(ResultCode.NO_DATA.getCode());
        response.setMessage(StringUtils.isEmpty(message) ? ResultCode.NO_DATA.getMessage() : message);
        return response;
    }

    public static <T> Result<T> serverError(String message) {
        Result<T> response = new Result<>();
        response.setCode(ResultCode.INTERNAL_SERVER_ERROR.getCode());
        response.setMessage(StringUtils.isEmpty(message) ? ResultCode.INTERNAL_SERVER_ERROR.getMessage() : message);
        return response;
    }

    public static <T> Result<T> invocationTimeout(String message) {
        Result<T> response = new Result<>();
        response.setCode(ResultCode.INVOCATION_TIMEOUT.getCode());
        response.setMessage(StringUtils.isEmpty(message) ? ResultCode.INVOCATION_TIMEOUT.getMessage() : message);
        return response;
    }

    public static <T> Result<T> invocationError(String message) {
        Result<T> response = new Result<>();
        response.setCode(ResultCode.INVOCATION_ERROR.getCode());
        response.setMessage(StringUtils.isEmpty(message) ? ResultCode.INVOCATION_ERROR.getMessage() : message);
        return response;
    }

    public static <T> Result<T> rateLimit(String message) {
        Result<T> response = new Result<>();
        response.setCode(ResultCode.RATE_LIMIT_ERROR.getCode());
        response.setMessage(StringUtils.isEmpty(message) ? ResultCode.RATE_LIMIT_ERROR.getMessage() : message);
        return response;
    }

    public static <T> Result<T> accessDeny(String message) {
        Result<T> response = new Result<>();
        response.setCode(ResultCode.ACCESS_DENIED.getCode());
        response.setMessage(StringUtils.isEmpty(message) ? ResultCode.ACCESS_DENIED.getMessage() : message);
        return response;
    }

    public enum ResultCode {
        // ---------------------------
        // 0000       处理成功
        // 1000～1999 区间表示参数错误
        // 2000～2999 区间表示用户错误
        // 3000～3999 区间表示系统业务异常
        // ---------------------------

        OK("1", "处理成功"),
        INVALID_PARAM(MSG_PREFIX + "CZZ1001", "非法参数"),
        EMPTY_PARAM(MSG_PREFIX + "CZZ1002", "参数为空"),
        MISSING_PARAM(MSG_PREFIX + "CZZ1003", "参数缺失"),
        NO_DATA(MSG_PREFIX + "CZZ1010", "查询不到该记录"),
        EXISTING_DATA(MSG_PREFIX + "CZZ1011", "记录已存在"),

        NOT_LOGIN(MSG_PREFIX + "CZZ2001", "用户未登录，访问路径需要验证"),
        LOGIN_FAILED(MSG_PREFIX + "CZZ2002", "账号不存在或密码错误"),
        DISABLED_ACCOUNT(MSG_PREFIX + "CZZ2003", "账号被禁用"),
        USER_NOT_EXISTS(MSG_PREFIX + "CZZ2004", "用户不存在"),
        EXISTS_USER(MSG_PREFIX + "CZZ2005", "用户已经存在"),
        ACCESS_DENIED(MSG_PREFIX + "CZZ2006", "访问拒绝"),

        INTERNAL_SERVER_ERROR(MSG_PREFIX + "CZZ3001", "服务器内部错误"),
        CONCURRENCY_ERROR(MSG_PREFIX + "CZZ3002", "多线程并发执行错误"),
        INVOCATION_TIMEOUT(MSG_PREFIX + "CZZ3003", "第三方调用超时"),
        INVOCATION_ERROR(MSG_PREFIX + "CZZ3004", "第三方调用异常"),
        RATE_LIMIT_ERROR(MSG_PREFIX + "CZZ3005", "服务限流");

        private final String code;
        private final String message;

        ResultCode(String code, String message) {
            this.code = code;
            this.message = message;
        }

        public String getCode() {
            return code;
        }

        public String getMessage() {
            return message;
        }
    }

    public String getCode() {
        return code;
    }

    public void setCode(String code) {
        this.code = code;
    }

    public String getMessage() {
        return message;
    }

    public void setMessage(String message) {
        this.message = message;
    }

    public T getResult() {
        return result;
    }

    public void setResult(T result) {
        this.result = result;
    }
}
