package com.xforceplus.local.ssdp.service.impl;

import com.google.common.collect.Maps;
import com.xforceplus.local.ssdp.domain.SsdpTable;
import com.xforceplus.local.ssdp.service.SsdpResult;
import com.xforceplus.local.ssdp.service.SsdpResultHandler;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.EmbeddedValueResolverAware;
import org.springframework.core.annotation.AnnotationAwareOrderComparator;
import org.springframework.core.annotation.AnnotationUtils;
import org.springframework.stereotype.Component;
import org.springframework.util.StringValueResolver;

import java.util.List;
import java.util.concurrent.ConcurrentMap;

@Slf4j
@Component
public class SsdpResultHandlerFactory implements InitializingBean, EmbeddedValueResolverAware {

    private StringValueResolver embeddedValueResolver;

    private ConcurrentMap<String, SsdpResultHandler> cacheMaps;

    @Autowired
    private List<SsdpResultHandler> ssdpResultHandlers;

    /**
     * Set the StringValueResolver to use for resolving embedded definition values.
     *
     * @param resolver
     */
    @Override
    public void setEmbeddedValueResolver(StringValueResolver resolver) {
        this.embeddedValueResolver = resolver;
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        this.cacheMaps = Maps.newConcurrentMap();
        log.debug("SsdpResultHandler list - {}", this.ssdpResultHandlers);
        AnnotationAwareOrderComparator.sort(this.ssdpResultHandlers);
        this.ssdpResultHandlers.forEach(this::toCacheMap);
        log.debug("SsdpResultHandler cache map - {}", cacheMaps);
    }

    private void toCacheMap(SsdpResultHandler ssdpResultHandler) {
        SsdpResult ssdpResult = AnnotationUtils.findAnnotation(ssdpResultHandler.getClass(), SsdpResult.class);
        final String apiId = this.embeddedValueResolver.resolveStringValue(ssdpResult.apiId());
        final String apiVersion = this.embeddedValueResolver.resolveStringValue(ssdpResult.apiVersion());
        this.cacheMaps.putIfAbsent(apiId + "[" + apiVersion + "]", ssdpResultHandler);
    }

    public SsdpResultHandler get(SsdpTable ssdpTable) {
        String apiId = ssdpTable.getApiId();
        String apiVersion = ssdpTable.getApiVersion();
        String matchKey = apiId + "[" + apiVersion + "]";
        log.trace("SsdpResultHandler matching - {}", matchKey);
        if (this.cacheMaps.containsKey(matchKey)) {
            return this.cacheMaps.get(matchKey);
        }
        matchKey = apiId + "[*]";
        log.trace("SsdpResultHandler matching - {}", matchKey);
        if (this.cacheMaps.containsKey(matchKey)) {
            return this.cacheMaps.get(matchKey);
        }
        matchKey = "*[*]";
        log.trace("SsdpResultHandler matching - {}", matchKey);
        return this.cacheMaps.get(matchKey);
    }

}
