package com.xforceplus.local.ssdp.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.crc.openapi.sdk.entity.Result;
import com.xforceplus.local.base.retry.XRetry;
import com.xforceplus.local.base.util.XResult;
import com.xforceplus.local.ssdp.config.SsdpRetryConfig;
import com.xforceplus.local.ssdp.config.SsdpTableConfig;
import com.xforceplus.local.ssdp.domain.SsdpTable;
import com.xforceplus.local.ssdp.service.SsdpRequestDataConverter;
import com.xforceplus.local.ssdp.service.SsdpService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

@Slf4j
@Service
public class SsdpServiceImpl implements SsdpService {

    @Value("${local.ssdp.client.disabled:false}")
    private boolean sysClientDisabled;

    @Autowired
    private SsdpService ssdpService;

    @Autowired
    private SsdpSysClient ssdpSysClient;

    @Autowired
    private SsdpTableConfig ssdpTableConfig;

    @Autowired
    private SsdpRequestDataConverter ssdpRequestDataConverter;

    @Autowired
    private SsdpResultHandlerFactory ssdpResultHandlerFactory;

    /**
     * 执行SSDP请求（名称）
     *
     * @param ssdpName
     * @param requestData
     * @return
     */
    @Override
    public XResult execute(String ssdpName, Object requestData) {
        SsdpTable[] ssdpTables = this.ssdpTableConfig.getSsdpTable(ssdpName);
        if (ssdpTables == null || ssdpTables.length == 0) {
            return XResult.fail("Ssdp config does not exist - " + ssdpName);
        }
        Map<String, XResult> xResults = new HashMap<>(ssdpTables.length);
        for (SsdpTable ssdpTable : ssdpTables) {
            xResults.put(ssdpTable.getApiId(), this.ssdpService.execute(ssdpTable, requestData));
        }
        if (xResults.size() == 1) {
            return xResults.values().iterator().next();
        }
        final Set<String> keys = xResults.values().stream().map(XResult::getCode).collect(Collectors.toSet());
        return (keys.size() == 1 && keys.contains(XResult.SUCCESS_CODE)
                ? XResult.ok() : XResult.fail()).setData(xResults);
    }

    /**
     * 执行SSDP请求（自动重试）
     *
     * @param ssdpTable
     * @param requestData
     * @return
     */
    @Override
    @XRetry(config = SsdpRetryConfig.class)
    public XResult execute(SsdpTable ssdpTable, Object requestData) {
        if (this.sysClientDisabled) {
            log.info("Ssdp SysClient is disabled.");
            return XResult.ok("SSDP客户端已禁用");
        }
        Result result = this.ssdpSysClient.doPost(ssdpTable, this.toRequestData(ssdpTable, requestData));
        return this.ssdpResultHandlerFactory.get(ssdpTable).convert(ssdpTable, result);
    }

    private String toRequestData(SsdpTable ssdpTable, Object requestData) {
        if (requestData instanceof String) {
            requestData = JSON.parseObject((String) requestData);
        }
        if (ssdpTable.isBusData()) {
            requestData = new JSONObject()
                    .fluentPut("BUS_PARA", null)
                    .fluentPut("BUS_DATA", requestData);
        }
        return this.ssdpRequestDataConverter.toJSONString(ssdpTable, requestData);
    }

}
