package com.xforceplus.local.ssdp.servlet;

import com.alibaba.fastjson.JSON;
import com.xforceplus.local.ssdp.SsdpContext;
import com.xforceplus.local.ssdp.SsdpUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.IOUtils;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.core.MethodParameter;
import org.springframework.core.annotation.AnnotationUtils;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpInputMessage;
import org.springframework.http.converter.HttpMessageConverter;
import org.springframework.web.bind.annotation.RestControllerAdvice;
import org.springframework.web.servlet.mvc.method.annotation.RequestBodyAdviceAdapter;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Type;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

@Slf4j
@RestControllerAdvice
public class SsdpRequestBodyAdvice extends RequestBodyAdviceAdapter {

    private Map<String, SsdpRequestConverter> mapConverter;

    public SsdpRequestBodyAdvice(ObjectProvider<List<SsdpRequestConverter>> converterProvider) {
        this.mapConverter = new HashMap<>();
        converterProvider.ifAvailable(converters ->
                converters.forEach(converter ->
                        this.mapConverter.put(Objects.requireNonNull(AnnotationUtils.findAnnotation(converter.getClass(), Ssdp.class)).name(), converter)
                )
        );
    }

    /**
     * Invoked first to determine if this interceptor applies.
     *
     * @param methodParameter the method parameter
     * @param targetType      the target type, not necessarily the same as the method
     *                        parameter type, e.g. for {@code HttpEntity<String>}.
     * @param converterType   the selected converter type
     * @return whether this interceptor should be invoked or not
     */
    @Override
    public boolean supports(MethodParameter methodParameter, Type targetType, Class<? extends HttpMessageConverter<?>> converterType) {
        return methodParameter.hasMethodAnnotation(Ssdp.class)
                && methodParameter.hasParameterAnnotation(SsdpParam.class);
    }

    /**
     * The default implementation returns the InputMessage that was passed in.
     *
     * @param inputMessage
     * @param parameter
     * @param targetType
     * @param converterType
     */
    @Override
    public HttpInputMessage beforeBodyRead(HttpInputMessage inputMessage, MethodParameter parameter, Type targetType, Class<? extends HttpMessageConverter<?>> converterType) throws IOException {
        return new HttpInputMessage() {
            @Override
            public InputStream getBody() throws IOException {
                return new ByteArrayInputStream(getRequestData(inputMessage, parameter).getBytes(StandardCharsets.UTF_8));
            }

            @Override
            public HttpHeaders getHeaders() {
                return inputMessage.getHeaders();
            }
        };
    }

    private String getRequestData(HttpInputMessage inputMessage, MethodParameter parameter) throws IOException {
        final String ssdpJson = IOUtils.toString(inputMessage.getBody(), StandardCharsets.UTF_8);
        log.info("Ssdp original request -> {}", ssdpJson);
        Object orgiJson = JSON.parse(ssdpJson);
        SsdpContext.current().setOriginal(orgiJson);
        Object requestData = SsdpUtils.extractRequestData(orgiJson);
        SsdpContext.current().setRequestData(requestData);
        final Ssdp ssdp = parameter.getMethodAnnotation(Ssdp.class);
        if (this.mapConverter.containsKey(ssdp.name())) {
            requestData = this.mapConverter.get(ssdp.name()).convert(requestData);
        }
        return JSON.toJSONString(requestData);
    }

}
