package com.xforceplus.local.ssdp.servlet;

import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.parser.ParserConfig;
import com.xforceplus.local.base.trace.TraceIdConst;
import com.xforceplus.local.base.trace.TraceIdUtils;
import com.xforceplus.local.base.util.SpringUtils;
import com.xforceplus.local.base.util.XBeanUtils;
import com.xforceplus.local.base.util.XResult;
import com.xforceplus.local.ssdp.SsdpContext;
import com.xforceplus.local.ssdp.SsdpUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.core.MethodParameter;
import org.springframework.core.annotation.AnnotationUtils;
import org.springframework.http.MediaType;
import org.springframework.http.converter.HttpMessageConverter;
import org.springframework.http.server.ServerHttpRequest;
import org.springframework.http.server.ServerHttpResponse;
import org.springframework.web.bind.annotation.RestControllerAdvice;
import org.springframework.web.servlet.mvc.method.annotation.ResponseBodyAdvice;

import java.util.*;

@Slf4j
@RestControllerAdvice
public class SsdpResponseBodyAdvice implements ResponseBodyAdvice<Object> {

    private Map<String, SsdpResponseConverter> mapConverter;

    public SsdpResponseBodyAdvice(ObjectProvider<List<SsdpResponseConverter>> converterProvider) {
        this.mapConverter = new HashMap<>();
        converterProvider.ifAvailable(converters ->
                converters.forEach(converter ->
                        this.mapConverter.put(Objects.requireNonNull(AnnotationUtils.findAnnotation(converter.getClass(), Ssdp.class)).name(), converter)
                )
        );
    }

    /**
     * Whether this component supports the given controller method return type
     * and the selected {@code HttpMessageConverter} type.
     *
     * @param returnType    the return type
     * @param converterType the selected converter type
     * @return {@code true} if {@link #beforeBodyWrite} should be invoked;
     * {@code false} otherwise
     */
    @Override
    public boolean supports(MethodParameter returnType, Class<? extends HttpMessageConverter<?>> converterType) {
        return returnType.hasMethodAnnotation(Ssdp.class);
    }

    /**
     * Invoked after an {@code HttpMessageConverter} is selected and just before
     * its write method is invoked.
     *
     * @param body                  the body to be written
     * @param returnType            the return type of the controller method
     * @param selectedContentType   the content type selected through content negotiation
     * @param selectedConverterType the converter type selected to write to the response
     * @param request               the current request
     * @param response              the current response
     * @return the body that was passed in or a modified (possibly new) instance
     */
    @Override
    public Object beforeBodyWrite(Object body, MethodParameter returnType, MediaType selectedContentType,
                                  Class<? extends HttpMessageConverter<?>> selectedConverterType,
                                  ServerHttpRequest request, ServerHttpResponse response) {
        final Ssdp ssdp = returnType.getMethodAnnotation(Ssdp.class);
        final SsdpContext.SsdpHolder ssdpHolder = SsdpContext.current();
        String returnCode = this.defaultIfBlank(ssdpHolder.getReturnCode(), ssdp.successCode());
        String returnDesc = this.defaultIfBlank(ssdpHolder.getReturnDesc(), ssdp.successDesc());
        Object returnData = body == null ? null : this.toReturnData(body);
        JSONObject ssdpResult = SsdpUtils.toReturnData(returnCode, returnDesc, returnData);
        if (this.mapConverter.containsKey(ssdp.name())) {
            ssdpResult = this.mapConverter.get(ssdp.name()).convert(ssdpResult);
        }
        log.info("Ssdp original response -> {}", ssdpResult);
        SsdpContext.clean();
        return ssdpResult;
    }

    /**
     * 处理msg和traceId
     *
     * @param returnData
     * @return
     */
    private Object toReturnData(Object returnData) {
        Class<?> clazz = returnData.getClass();
        if (clazz.isEnum()) {
            return ((Enum<?>) returnData).name();
        }
        if (clazz.isArray()) {
            return returnData;
        }
        if (ParserConfig.isPrimitive2(clazz)) {
            return returnData;
        }
        Map<String, Object> xResult;
        if (returnData instanceof Map) {
            xResult = (Map) returnData;
        } else {
            if (returnData instanceof Collection) {
                return returnData;
            } else {
                xResult = XBeanUtils.obj2Map(returnData);
            }
        }
        String msg = (String) xResult.remove("message");
        if (StringUtils.isBlank(msg)) {
            msg = MapUtils.getString(xResult, XResult.MSG);
        }
        xResult.put(XResult.MSG, this.handleMsgField(msg));
        xResult.put(TraceIdConst.TRACE_ID, TraceIdUtils.getTraceId());
        return xResult;
    }

    private String handleMsgField(String msg) {
        return msg.startsWith("JSON parse error:") ? "数据格式错误" : msg;
    }

    private String defaultIfBlank(String value, String defval) {
        if (StringUtils.isBlank(value)) {
            value = SpringUtils.resolveStringValue(defval);
        }
        return value;
    }

}
