package com.xforceplus.phoenix.tools.event;

import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.EventBus;
import com.xforceplus.phoenix.tools.util.SpringContextUtil;
import lombok.extern.slf4j.Slf4j;

import java.util.concurrent.Executor;

/**
 * @program: phoenix-purchaser-tools
 * @description 事件发布者
 * @author: wdye
 * @created: 2020-03-27 12:55
 */
@Slf4j
public class EventBusPublisher {

    private static class EventBusHolder {
        private static final EventBus instance = new EventBus();
        private static AsyncEventBus asyncInstance = null;
    }

    private EventBusPublisher() {
    }

    private static EventBus getInstance(Executor executor) {
        if (null == executor) {
            return EventBusHolder.instance;
        }
        if (EventBusHolder.asyncInstance == null) {
            // 应用启动自动注册时, 是串行无需进行同步
            EventBusHolder.asyncInstance = new AsyncEventBus(executor);
        }
        return EventBusHolder.asyncInstance;
    }

    /**
     * 发布事件
     *
     * @param event
     */
    public static void publish(BaseCustomEvent event) {
        if (event == null) {
            return;
        }
        AsyncCustomEvent asyncEvent = event.getClass().getAnnotation(AsyncCustomEvent.class);
        Executor executor = null;
        if (null != asyncEvent) {
            executor = SpringContextUtil.getBean(asyncEvent.beanName(), Executor.class);
        }
        getInstance(executor).post(event);
    }

    protected static void register(AbstractCustomEventHandler<? extends BaseCustomEvent> handler, Executor executor) {
        if (handler == null) {
            return;
        }
        getInstance(executor).register(handler);
        log.debug("EventBusPublisher register: {} success", handler);
    }

    protected static void unregister(AbstractCustomEventHandler<? extends BaseCustomEvent> handler, Executor executor) {
        if (handler == null) {
            return;
        }
        getInstance(executor).unregister(handler);
        log.debug("EventBusPublisher unregister: {} success", handler);
    }
}
