package com.xforceplus.phoenix.tools.redisson;

import com.xforceplus.phoenix.tools.constant.RedissonConstants;
import com.xforceplus.phoenix.tools.enable.AutoConfiguration;
import org.apache.commons.lang3.StringUtils;
import org.redisson.Redisson;
import org.redisson.api.RedissonClient;
import org.redisson.config.ClusterServersConfig;
import org.redisson.config.Config;
import org.redisson.config.SentinelServersConfig;
import org.redisson.config.SingleServerConfig;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;

import java.util.Objects;

/**
 * @program: phoenix-purchaser-tools
 * @description RedissonConfiguration
 * @author: wdye
 * @created: 2019-09-20 19:35
 */
public class RedissonConfiguration implements AutoConfiguration {

    @Value("${redis.type:standalone}")
    private String type;
    @Value("${redis.standaloneHost:127.0.0.1}")
    private String standaloneHost;
    @Value("${redis.standalonePort:6379}")
    private Integer standalonePort;
    @Value("${redis.maxActive:20}")
    private int maxActive;
    @Value("${redis.minIdle:5}")
    private Integer minIdle;
    @Value("${redis.maxTotal:}")
    private Integer maxTotal;
    @Value("${redis.maxWaitMillis:}")
    private Integer maxWaitMillis;
    @Value("${redis.minEvictableIdleTimeMillis:}")
    private Integer minEvictableIdleTimeMillis;
    @Value("${redis.numTestsPerEvictionRun:}")
    private Integer numTestsPerEvictionRun;
    @Value("${redis.testOnBorrow:true}")
    private boolean testOnBorrow;
    @Value("${redis.testWhileIdle:false}")
    private boolean testWhileIdle;
    @Value("${redis.sentinelNodes:}")
    private String sentinelNodes;
    @Value("${redis.sentinelMaster:}")
    private String sentinelMaster;
    @Value("${redis.password:}")
    private String password;
    @Value("${redis.connectTimeout:10000}")
    private int connectTimeout;
    @Value("${redis.timeout:3000}")
    private Integer timeout;
    @Value("${redis.clusterNodes:}")
    private String clusterNodes;

    @Bean
    public RedissonClient redissonClient() {
        Config config = new Config();

        final String prefix = "redis://";
        final String standaloneType = "standalone";
        if (Objects.equals(type, standaloneType)) {
            final String address = prefix + standaloneHost + ":" + standalonePort;
            SingleServerConfig singleServerConfig = config.useSingleServer()
                    .setAddress(address)
                    .setConnectionPoolSize(maxActive)
                    .setConnectionMinimumIdleSize(minIdle)
                    // 连接超时，单位：毫秒
                    .setConnectTimeout(connectTimeout)
                    // 同任何节点建立连接时的等待超时。时间单位是毫秒
                    .setTimeout(timeout);
            if (StringUtils.isNotBlank(password)) {
                singleServerConfig.setPassword(password);
            }
        }else if (type.equals("cluster")){
            StringBuilder clusterAddressBuilder = new StringBuilder();
            String[] clusterNodeArray = StringUtils.split(clusterNodes, RedissonConstants.COMMA);
            for (String node : clusterNodeArray) {
                clusterAddressBuilder.append(prefix).append(node).append(RedissonConstants.COMMA);
            }
            String clusterAddress = clusterAddressBuilder.toString();
            ClusterServersConfig clusterServersConfig = config.useClusterServers()
                    .addNodeAddress(clusterAddress.substring(0, clusterAddress.length() - 1))
                    .setMasterConnectionPoolSize(maxActive)
                    .setMasterConnectionMinimumIdleSize(minIdle)
                    .setSlaveConnectionPoolSize(maxActive)
                    .setSlaveConnectionMinimumIdleSize(minIdle)
                    .setConnectTimeout(connectTimeout)
                    .setTimeout(timeout);
            if (StringUtils.isNotBlank(password)) {
                clusterServersConfig.setPassword(password);
            }
        }else {
            StringBuilder sentinelAddressBuilder = new StringBuilder();
            String[] sentinelNodeArray = StringUtils.split(sentinelNodes, RedissonConstants.COMMA);
            for (String node : sentinelNodeArray) {
                sentinelAddressBuilder.append(prefix).append(node).append(RedissonConstants.COMMA);
            }
            String sentinelAddress = sentinelAddressBuilder.toString();
            SentinelServersConfig sentinelServersConfig = config.useSentinelServers().setMasterName(sentinelMaster)
                    .addSentinelAddress(sentinelAddress.substring(0, sentinelAddress.length() - 1))
                    .setMasterConnectionPoolSize(maxActive)
                    .setMasterConnectionMinimumIdleSize(minIdle)
                    .setSlaveConnectionPoolSize(maxActive)
                    .setSlaveConnectionMinimumIdleSize(minIdle)
                    .setConnectTimeout(connectTimeout)
                    .setTimeout(timeout);
            if (StringUtils.isNotBlank(password)) {
                sentinelServersConfig.setPassword(password);
            }
        }
        return Redisson.create(config);
    }
}
