package com.xforceplus.phoenix.tools.util;

import com.google.common.base.CaseFormat;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.openxml4j.exceptions.InvalidFormatException;
import org.apache.poi.ss.usermodel.*;
import org.apache.poi.ss.usermodel.DateUtil;
import org.springframework.stereotype.Component;

import java.io.*;
import java.lang.reflect.Field;
import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.Map;

import static org.apache.poi.ss.usermodel.Cell.*;

/**
 * @program: phoenix-purchaser-tools
 * @description: excel解析工具
 * @author: cuixiangyu
 * @created: 2020-07-08 15:01
 */
@Component
public class ExcelTools {

    /**
     * 准备解析Excel文件,返回指定类型的数据集合
     *
     * @param file
     * @param sheetName
     * @param metaMap
     * @param clazz
     * @param limitNums
     * @param <T>
     * @return
     */
    public <T> List<T> importExcel(File file, String sheetName, Map<String, String> metaMap, Class<T> clazz, Integer limitNums) {
        Sheet sheet = getSheet(file, sheetName);
        return readExcelData(sheet, metaMap, clazz, limitNums);
    }

    /**
     * 读取excel数据解析
     *
     * @param sheet
     * @param metaMap
     * @param clazz
     * @param limitNums
     * @param <T>
     * @return
     */
    private <T> List<T> readExcelData(Sheet sheet, Map<String, String> metaMap, Class<T> clazz, Integer limitNums) {
        // 创建一个存放map集合的泛型list集合,用于存储sheet页中读取的所有数据
        List result = Lists.newArrayList();
        // 逐个读取第一行单元格的所有单元格（firstRow.getLastCellNum()获取最大列索引）
        Row firstRow = sheet.getRow(0);
        List<String> sheetColumnNameList = Lists.newArrayList();
        for (Cell cell : firstRow) {
            // 把第一行数据的所有单元格内容按列索引顺序均存放于list集合中
            sheetColumnNameList.add(getCellValue(cell));
        }
        // 按照sheet第一行的单元格数据顺序，依次获取表列名，产生一个list->sortedTableColumnNameList
        String[] sortedTableColumnNameList = new String[sheetColumnNameList.size()];
        int count = 0;
        for (String sheetColumnName : sheetColumnNameList) {
            for (String key : metaMap.keySet()) {
                if (sheetColumnName.equals(metaMap.get(key))) {
                    sortedTableColumnNameList[count] = key;
                }
            }
            count++;
        }
        // 遍历sheet页中的所有行数据
        int rowNo = 1;
        int colNo = 1;
        for (Row row : sheet) {
            //第一行跳出循环，进行第二行循环不进行数据处理
            if (rowNo == 1) {
                rowNo++;
                continue;
            }
            //从第二行开始，如果第一，第二单元格都是空值就直接结束循环，不在处理此Excel
            if (StringUtils.isEmpty(getCellValue(row.getCell(0))) && StringUtils.isEmpty(getCellValue(row.getCell(1)))) {
                break;
            }
            // 遍历指定行数据的所有单元格数据( row.getLastCellNum()获取最大列索引)
            if (Map.class.isAssignableFrom(clazz)) {
                //如果最终转换对象类型是map类型的对象
                Map map = Maps.newHashMap();
                for (Cell cell : row) {
                    String cellValue = getCellValue(cell); // 获取该单元格数据
                    String cellName = sortedTableColumnNameList[cell.getColumnIndex()]; // 获取该位置的单元格的英文名称
                    if (!org.springframework.util.StringUtils.isEmpty(cellName)) {
                        map.put(CaseFormat.LOWER_UNDERSCORE.to(CaseFormat.LOWER_CAMEL, cellName), cellValue);
                    }
                }
                result.add(map);
            } else {
                //如果最终转换对象类型不是map类型的对象
                try {
                    Object obj = clazz.newInstance();
                    //循环所有列
                    for (Cell cell : row) {
                        // 最后列，无列头，但单元格有值时直接退出
                        if (sortedTableColumnNameList.length - 1 < cell.getColumnIndex()) {
                            continue;
                        }
                        // 获取该位置的单元格的英文名称
                        String cellName = sortedTableColumnNameList[cell.getColumnIndex()];
                        if (!org.springframework.util.StringUtils.isEmpty(cellName)) {
                            try {
                                Field field = clazz.getDeclaredField(cellName);
                                //用反射时访问私有变量
                                field.setAccessible(true);
                                field.set(obj, getFieldValue(field, cell));
                            } catch (NoSuchFieldException e) {
                                e.printStackTrace();
                                throw new RuntimeException("提示：对象类型不存在该字段" + cellName);
                            } catch (IllegalAccessException e) {
                                throw new RuntimeException("提示：获取传入对象类型的属性失败.");
                            }
                        }
                        colNo++;
                    }
                    result.add(obj);
                } catch (InstantiationException e) {
                    e.printStackTrace();
                    throw new RuntimeException("提示：传入的对象类型实例化失败.[" + e.getMessage() + "]");
                } catch (IllegalAccessException e) {
                    e.printStackTrace();
                    throw new RuntimeException("第" + rowNo + "行第" + colNo + "列,获取传入对象类型的属性失败.[" + e.getMessage() + "]");
                } catch (Exception e) {
                    e.printStackTrace();
                    throw new RuntimeException("第" + rowNo + "行第" + colNo + "列,解析数据失败.[" + e.getMessage() + "]");
                }
            }
            rowNo++;
            if (null != limitNums && rowNo >= (limitNums + 3)) {
                throw new RuntimeException("导入失败,单次上传上限为" + limitNums + "条,您已超限");
            }
        }
        return result;
    }

    /**
     * 根据文件和sheet名称获取sheet
     *
     * @param file
     * @param sheetName
     * @return
     */
    private Sheet getSheet(File file, String sheetName) {
        Workbook wb = null;
        InputStream inputStream = null;
        try {
            inputStream = new FileInputStream(file);
            wb = WorkbookFactory.create(inputStream); //根据读取的文件创建一个Excel文件
        } catch (FileNotFoundException e) {
            e.printStackTrace();
            throw new RuntimeException("提示：未获得传入的文件,请刷新页面,重新选择文件上传.");
        } catch (InvalidFormatException e) {
            e.printStackTrace();
            throw new RuntimeException("提示：不是有效的Excel文件,请刷新页面,重新选择文件上传.");
        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException("提示：未知错误，请查阅报错日志.");
        } finally {
            try {
                if (inputStream != null) {
                    inputStream.close();
                }
            } catch (IOException e) {
                e.printStackTrace();
            }
        }

        Sheet sheet = wb.getSheet(sheetName); // 创建一个Excel sheet页;读取Excel中sheet页名称为公司信息的sheet页

        // 当读取的名称为公司信息的sheet页为null时报错
        if (sheet == null) {
            throw new RuntimeException("提示:处理失败,文件中未找到[" + sheetName + "]sheet页.");
        }

        return sheet;
    }

    /**
     * 获取Excel中单元格中的值
     *
     * @param cell
     * @return
     */
    private String getCellValue(Cell cell) {
        String ret = "";
        try {
            if (null == cell) {
                return ret;
            }
            switch (cell.getCellType()) {
                case CELL_TYPE_BOOLEAN:
                    ret = String.valueOf(cell.getBooleanCellValue());
                    break;
                case CELL_TYPE_NUMERIC:
                    // 处理日期类型,Excel是用数字存贮日期
                    if (org.apache.poi.ss.usermodel.DateUtil.isCellDateFormatted(cell)) {
                        double d = cell.getNumericCellValue();
                        Date date = org.apache.poi.ss.usermodel.DateUtil.getJavaDate(d);
                        ret = new SimpleDateFormat("yyyy-MM-dd").format(date);
                    } else {
                        Long temp = Math.round(cell.getNumericCellValue());
                        ret = Double.parseDouble(temp + ".0") == cell.getNumericCellValue()
                                ? String.valueOf(temp) : getPlainString(String.valueOf(cell.getNumericCellValue()));
                    }
                    break;
                case CELL_TYPE_STRING:
                    ret = cell.getStringCellValue().trim();
                    break;
                case CELL_TYPE_FORMULA:
                    try {
                        ret = org.apache.poi.ss.usermodel.DateUtil.isCellDateFormatted(cell) ? new SimpleDateFormat("yyyy-MM-dd").format(cell.getDateCellValue())
                                : parseFormula(cell);
                    } catch (Exception e) {
                        ret = String.valueOf(cell.getRichStringCellValue());
                    }
                    break;
                default:
                    break;
            }
        } catch (Exception e) {
            throw new RuntimeException("文件中存在单元格格式不正确等问题,数据有误,请检查文件中公式等是否存在引用失效等问题");
        }
        return ret;
    }

    /**
     * 获取Excel中单元格中的值
     *
     * @param cell
     * @return
     */
    private String parseFormula(Cell cell) {
        FormulaEvaluator evaluator = cell.getSheet().getWorkbook().getCreationHelper().createFormulaEvaluator();

        CellValue cellValue = evaluator.evaluate(cell);

        String ret = StringUtils.EMPTY;
        switch (cellValue.getCellType()) {
            case Cell.CELL_TYPE_BOOLEAN:
                ret = String.valueOf(cell.getBooleanCellValue());
                break;
            case Cell.CELL_TYPE_NUMERIC:
                // 处理日期类型,Excel是用数字存贮日期
                if (org.apache.poi.ss.usermodel.DateUtil.isCellDateFormatted(cell)) {
                    double d = cell.getNumericCellValue();
                    Date date = DateUtil.getJavaDate(d);
                    ret = new SimpleDateFormat("yyyy-MM-dd").format(date);
                } else {
                    Long temp = Math.round(cell.getNumericCellValue());
                    ret = Double.parseDouble(temp + ".0") == cell.getNumericCellValue()
                            ? String.valueOf(temp) : getPlainString(String.valueOf(cell.getNumericCellValue()));
                }
                break;
            case Cell.CELL_TYPE_STRING:
                ret = cell.getStringCellValue().trim();
                break;
        }
        return ret;
    }

    /**
     * 获取Excel中单元格中的值
     *
     * @param field
     * @param cell
     * @return
     */
    private Object getFieldValue(Field field, Cell cell) {
        if (BigDecimal.class.isAssignableFrom(field.getType())) {
            if (cell.getCellType() == CELL_TYPE_STRING) {
                return BigDecimal.valueOf(Double.parseDouble(cell.getStringCellValue()));
            }
            return BigDecimal.valueOf(cell.getNumericCellValue());
        } else if (Boolean.class.isAssignableFrom(field.getType())) {
            return cell.getBooleanCellValue();
        } else if (Integer.class.isAssignableFrom(field.getType())) {
            Long temp = Math.round(cell.getNumericCellValue());
            return temp.intValue();
        } else if (Double.class.isAssignableFrom(field.getType())) {
            return cell.getNumericCellValue();
        } else {
            return getCellValue(cell);
        }
    }

    /**
     * 科学计数法重获为字符串
     *
     * @param value
     * @return
     */
    public static String getPlainString(String value) {
        BigDecimal bd = new BigDecimal(value);
        return bd.toPlainString();
    }
}
