package com.xforceplus.phoenix.tools.util;

import org.springframework.beans.BeansException;
import org.springframework.lang.Nullable;

import java.beans.PropertyDescriptor;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author liuda
 * @description: 对象拷贝
 */
public class BeanExtUtil extends org.springframework.beans.BeanUtils {
    /**
     * 对象拷贝
     *
     * @param source 数据来源
     * @param target 目标对象 （不能为Null）
     * @throws BeansException
     */
    public static void copyProperties(Object source, Object target) throws BeansException {
        baseCopyProperties(source, target, null);
    }

    /**
     * 对象拷贝
     *
     * @param source      数据来源
     * @param targetClass 目标类型对象
     * @throws BeansException
     */
    public static <T> T copyProperties(Object source, Class<T> targetClass) throws BeansException {
        T target = null;
        try {
            target = targetClass.newInstance();
        } catch (Exception e) {
            e.printStackTrace();
        }
        baseCopyProperties(source, target, null);

        return target;
    }

    /**
     * List拷贝
     *
     * @param sourceList  数据来源
     * @param targetClass 目标类型对象
     * @throws BeansException
     */
    public static <E, T> List<T> copyList(List<E> sourceList, Class<T> targetClass) throws BeansException {
        List<T> targetList = new ArrayList<>();

        if (null == sourceList || null == targetClass) {
            return targetList;
        }
        for (E sourceObj : sourceList) {
            T targetObj = copyProperties(sourceObj, targetClass);
            targetList.add(targetObj);
        }
        return targetList;

    }

    /**
     * 对象拷贝(支持list 拷贝)
     * 1、支持List 属性拷贝
     * 2、不支持Null 属性拷贝
     *
     * @param source
     * @param target
     * @param targetListFieldTypes target list 属性类型，如 target 属性名为 dataList<User> 传值为 key=dataList;value=User.class
     * @throws BeansException
     */
    public static void copyPropertiesSpList(Object source, Object target, @Nullable Map<String, Class> targetListFieldTypes) throws BeansException {
        baseCopyProperties(source, target, targetListFieldTypes);
    }

    /**
     * 对象拷贝(支持list 拷贝)
     * 1、支持List 属性拷贝
     * 2、不支持Null 属性拷贝
     *
     * @param source
     * @param target
     * @param targetListFieldTypes target list 属性类型，如 target 属性名为 dataList<User> 传值为 key=dataList;value=User.class
     * @throws BeansException
     */
    private static void baseCopyProperties(Object source, Object target, @Nullable Map<String, Class> targetListFieldTypes) throws BeansException {
        PropertyDescriptor[] sourcePds = getPropertyDescriptors(source.getClass());
        PropertyDescriptor[] targetPds = getPropertyDescriptors(target.getClass());

        HashMap<String, PropertyDescriptor> targetMaps = new HashMap<>();

        for (PropertyDescriptor targetPd : targetPds) {
            targetMaps.put(targetPd.getName(), targetPd);
        }
        for (PropertyDescriptor sourcePd : sourcePds) {

            Method readMethod = sourcePd.getReadMethod();
            if (null == readMethod) {
                continue;
            }
            if (!Modifier.isPublic(readMethod.getDeclaringClass().getModifiers())) {
                readMethod.setAccessible(true);
            }
            Object sourceValue = null;
            try {
                sourceValue = readMethod.invoke(source);
            } catch (Exception e) {
            }
            if (null == sourceValue) {
                continue;
            }
            PropertyDescriptor targetPd = targetMaps.get(sourcePd.getName());

            if (null == targetPd) {
                continue;
            }

            Method writeMethod = targetPd.getWriteMethod();
            if (null == writeMethod) {
                continue;
            }
            Class paramType = null;
            try {

                paramType = writeMethod.getParameters()[0].getType();
            } catch (Exception e) {
            }
            if (null == paramType) {
                continue;
            }
            if (paramType != sourceValue.getClass()) {
                Object customSourceValue = getCustomSourceValue(sourceValue, paramType, targetPd, targetListFieldTypes);
                if (null == customSourceValue) {
                    continue;
                }
                sourceValue = customSourceValue;

            }
            try {
                writeMethod.invoke(target, sourceValue);
            } catch (Exception e) {
            }

        }
    }

    /**
     * 自定义参数特殊处理
     *
     * @param paramType
     * @param targetPd
     * @param targetListFieldTypes
     * @return
     */
    private static Object getCustomSourceValue(Object sourceValue, Class paramType, PropertyDescriptor targetPd, Map<String, Class> targetListFieldTypes) {
        if (paramType.equals(BigDecimal.class)) {
            return new BigDecimal(sourceValue.toString());
        }

        if (paramType.equals(Long.class)) {
            return Long.valueOf(sourceValue.toString());
        }

        if (paramType.equals(Integer.class)) {
            return Integer.valueOf(sourceValue.toString());
        }

        if (paramType.equals(Integer.TYPE)) {
            return Integer.valueOf(sourceValue.toString());
        }

        // 时间类型转换为 时间戳(必须在String.class 之前)
        if (paramType.equals(String.class) && sourceValue instanceof Date) {
            return String.valueOf(((Date) sourceValue).getTime());
        }

        if (paramType.equals(String.class)) {
            return sourceValue.toString();
        }
        // List 属性特殊处理
        if (sourceValue instanceof List && paramType == List.class) {
            if (null == targetListFieldTypes) {
                return sourceValue;
            }

            Class aClass = targetListFieldTypes.get(targetPd.getName());
            if (null == aClass) {
                // 未指定对应的list类型 直接跳过赋值
                return sourceValue;
            }
            List sourceList = (List) sourceValue;
            List targetList = new ArrayList();
            for (Object sourceObj : sourceList) {
                try {
                    Object targetObj = aClass.newInstance();
                    // 循环拷贝 list 对象
                    copyPropertiesSpList(sourceObj, targetObj, null);
                    targetList.add(targetObj);
                } catch (Exception e) {
                }

            }
            return targetList;

        }
        try {
            // 对象类型不一致处理
            Object targetObj = paramType.newInstance();
            copyPropertiesSpList(sourceValue, targetObj, null);
            return targetObj;
        } catch (Exception e) {
        }

        return sourceValue;
    }

}