package com.xforceplus.phoenix.tools.util;

import org.apache.commons.lang3.StringUtils;
import org.apache.poi.hssf.usermodel.HSSFFont;
import org.apache.poi.hssf.util.HSSFColor;
import org.apache.poi.ss.usermodel.*;
import org.apache.poi.xssf.streaming.SXSSFWorkbook;
import org.apache.poi.xssf.usermodel.*;
import org.springframework.util.CollectionUtils;

import java.io.*;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @program: phoenix-purchaser-tools
 * @description excel工具类
 * @author: liuda
 * @created: 2020-04-24 10:03
 */
public class ExcelUtil {

    public static final Integer ALIGN_CENTER = 1;
    public static final Integer ALIGN_RIGHT = 2;

    /**
     * 导出Excel
     *
     * @param sheetName sheet名称
     * @param titles    标题
     * @param contents  内容
     * @param workbook  SXSSFWorkbook对象
     * @return
     */
    public static SXSSFWorkbook createSXSSFWorkbook(String sheetName, List<String> titles, List<Integer> alignList, List<String[]> contents, SXSSFWorkbook workbook) {
        /** **********创建工作簿************ */
        workbook = (workbook == null) ? new SXSSFWorkbook() : workbook;

        /** **********创建工作表************ */
        Sheet sheet = workbook.createSheet(sheetName);

        /** **********设置纵横打印（默认为纵打）、打印纸***************** */
        PrintSetup printSetup = sheet.getPrintSetup();
        printSetup.setLandscape(false);//false-默认纵向打印  true-横向打印

        /** ************设置单元格字体************** */
        //通常字体
        Font normalFont = workbook.createFont();
        normalFont.setFontName(HSSFFont.FONT_ARIAL);
        normalFont.setFontHeightInPoints((short) 12);// 设置字体大小
        normalFont.setBold(true);
        //粗体
        Font boldFont = workbook.createFont();
        boldFont.setFontName(HSSFFont.FONT_ARIAL);
        boldFont.setFontHeightInPoints((short) 10);// 设置字体大小
        boldFont.setBold(true);
        boldFont.setColor(HSSFColor.VIOLET.index);

        /** ************以下设置三种单元格样式************ */


        // 用于标题居中
        CellStyle hcs_center = workbook.createCellStyle();
        hcs_center.setAlignment(HorizontalAlignment.CENTER);// 文字水平对齐
        hcs_center.setVerticalAlignment(VerticalAlignment.CENTER);// 文字垂直对齐
        hcs_center.setWrapText(false);// 文字是否换行
        hcs_center.setFillForegroundColor(IndexedColors.SKY_BLUE.index);
        hcs_center.setFillPattern(FillPatternType.SOLID_FOREGROUND);
        hcs_center.setBorderBottom(BorderStyle.THIN);
        hcs_center.setBorderLeft(BorderStyle.THIN);
        hcs_center.setBorderRight(BorderStyle.THIN);
        hcs_center.setBorderTop(BorderStyle.THIN);

        // 用于正文居中
        CellStyle ctt_center = workbook.createCellStyle();
        ctt_center.setAlignment(HorizontalAlignment.CENTER);// 文字居中对齐
        ctt_center.setVerticalAlignment(VerticalAlignment.CENTER);// 文字垂直对齐
        ctt_center.setWrapText(false);// 文字是否换行
        ctt_center.setFillForegroundColor(IndexedColors.LIGHT_YELLOW.index);
        ctt_center.setFillPattern(FillPatternType.SOLID_FOREGROUND);
        ctt_center.setBorderBottom(BorderStyle.THIN);
        ctt_center.setBorderLeft(BorderStyle.THIN);
        ctt_center.setBorderRight(BorderStyle.THIN);
        ctt_center.setBorderTop(BorderStyle.THIN);


        // 用于正文居右
        CellStyle ctt_right = workbook.createCellStyle();
        ctt_right.setAlignment(HorizontalAlignment.RIGHT);// 文字居右对齐
        ctt_right.setVerticalAlignment(VerticalAlignment.CENTER);// 文字垂直对齐
        ctt_right.setWrapText(false);// 文字是否换行
        ctt_right.setFillForegroundColor(IndexedColors.LIGHT_YELLOW.index);
        ctt_right.setFillPattern(FillPatternType.SOLID_FOREGROUND);
        ctt_right.setBorderBottom(BorderStyle.THIN);
        ctt_right.setBorderLeft(BorderStyle.THIN);
        ctt_right.setBorderRight(BorderStyle.THIN);
        ctt_right.setBorderTop(BorderStyle.THIN);

        /** ***************以下是EXCEL第一行列标题********************* */
        Row row = sheet.createRow(0);
        Cell cell = null;
        int titleLen = titles.size();
        Map<Integer, CellStyle> alignMap = new HashMap<>(titleLen);
        for (int i = 0; i < titleLen; i++) {
            cell = row.createCell(i);
            cell.setCellValue(titles.get(i));
            cell.setCellStyle(hcs_center);
            cell.setCellType(Cell.CELL_TYPE_STRING);
            try {
                if ("发票序号".equals(titles.get(i))) {
                    sheet.setColumnWidth(i, titles.get(i).getBytes("utf-8").length * 256);
                } else if ("税额".equals(titles.get(i)) || "数量".equals(titles.get(i))) {
                    sheet.setColumnWidth(i, titles.get(i).getBytes("utf-8").length * 3 * 256);
                } else {
                    sheet.setColumnWidth(i, titles.get(i).getBytes("utf-8").length * 2 * 256);
                }
            } catch (UnsupportedEncodingException e) {
                e.printStackTrace();
            }

            //设置文本格式
            CellStyle cellStyle = workbook.createCellStyle();
            DataFormat dataFormat = workbook.createDataFormat();
            cellStyle.setDataFormat(dataFormat.getFormat("@"));
            cell.setCellStyle(cellStyle);

            //设置居中
            if (CollectionUtils.isEmpty(alignList) || alignList.size() <= i) {
                alignMap.put(i, ctt_center);
                continue;
            }
            if (ALIGN_RIGHT.equals(alignList.get(i))) {
                //居右对齐
                alignMap.put(i, ctt_right);
            } else {
                alignMap.put(i, ctt_center);
            }
        }


        /** ***************以下是EXCEL正文数据********************* */
        int contentLen = contents.size();

        String[] rowContent = null;
        String content = "";

        // 用于正文居右
        CellStyle cellStyle = workbook.createCellStyle();
        cellStyle.setAlignment(HorizontalAlignment.RIGHT);// 文字居右对齐
        cellStyle.setVerticalAlignment(VerticalAlignment.CENTER);// 文字垂直对齐
        cellStyle.setWrapText(false);// 文字是否换行
        cellStyle.setFillForegroundColor(IndexedColors.LIGHT_YELLOW.index);
        cellStyle.setFillPattern(FillPatternType.SOLID_FOREGROUND);
        cellStyle.setBorderBottom(BorderStyle.THIN);
        cellStyle.setBorderLeft(BorderStyle.THIN);
        cellStyle.setBorderRight(BorderStyle.THIN);
        cellStyle.setBorderTop(BorderStyle.THIN);
        DataFormat format = workbook.createDataFormat();
        cellStyle.setDataFormat(format.getFormat("0.00"));

        for (int i = 0; i < contentLen; i++) {
            //row
            row = sheet.createRow(i + 1);
            rowContent = contents.get(i);
            for (int j = 0; j < titleLen; j++) {
                if (j < rowContent.length) {
                    content = rowContent[j];
                }
                //将内容按顺序赋给对应的列对象
                cell = row.createCell(j);
                cell.setCellStyle(alignMap.get(j));
                if ("含税金额".equals(titles.get(j)) || "税额".equals(titles.get(j)) || "不含税金额".equals(titles.get(j)) || "数量".equals(titles.get(j))
                        || "不含税单价".equals(titles.get(j)) || "不含税折扣金额".equals(titles.get(j)) || "折扣税额".equals(titles.get(j))
                        || "含税折扣金额".equals(titles.get(j)) || "转出金额".equals(titles.get(j)) || "付款金额".equals(titles.get(j)) || "有效税额".equals(titles.get(j))) {
                    cell.setCellStyle(cellStyle);
                    if(StringUtils.isNotEmpty(content)){
                        cell.setCellValue(Double.parseDouble(content));
                    }
                } else {
                    cell.setCellValue(content);
                }
            }
        }


        return workbook;
    }

    /**
     * 创建xlsx Excel 并且支持列头批注插入
     *
     * @param sheetName       sheet 名称
     * @param titles          列头列表
     * @param contents        内容列表
     * @param headAnnotations 批注map 如列头是 发票代码 需要提交 必填的批注：发票代码：必填
     * @return
     */
    public static XSSFWorkbook createXSSFWorkbook(String sheetName, List<String> titles, List<Integer> alignList, List<String[]> contents, Map<String, String> headAnnotations) {
        /** **********创建工作簿************ */
        XSSFWorkbook workbook = new XSSFWorkbook();

        /** **********创建工作表************ */
        XSSFSheet sheet = workbook.createSheet(sheetName);

        /** **********设置纵横打印（默认为纵打）、打印纸***************** */
        PrintSetup printSetup = sheet.getPrintSetup();
        printSetup.setLandscape(false);//false-默认纵向打印  true-横向打印

        /** ************设置单元格字体************** */
        //通常字体
        Font normalFont = workbook.createFont();
        normalFont.setFontName(HSSFFont.FONT_ARIAL);
        normalFont.setFontHeightInPoints((short) 12);// 设置字体大小
        normalFont.setBold(true);
        //粗体
        Font boldFont = workbook.createFont();
        boldFont.setFontName("微软雅黑");
        boldFont.setFontHeightInPoints((short) 10);// 设置字体大小
        boldFont.setBold(true);
        boldFont.setColor(HSSFColor.BLACK.index);

        /** ************以下设置三种单元格样式************ */
        // 设置为文本格式
        DataFormat dataFormat = workbook.createDataFormat();
        short textFormat = dataFormat.getFormat("@");

        // 用于标题居中
        CellStyle hcs_center = workbook.createCellStyle();
        hcs_center.setAlignment(HorizontalAlignment.CENTER);// 文字水平对齐
        hcs_center.setVerticalAlignment(VerticalAlignment.CENTER);// 文字垂直对齐
        hcs_center.setWrapText(false);// 文字是否换行
        hcs_center.setFillForegroundColor(IndexedColors.SKY_BLUE.index);
        hcs_center.setFillPattern(FillPatternType.SOLID_FOREGROUND);
        hcs_center.setBorderBottom(BorderStyle.THIN);
        hcs_center.setBorderLeft(BorderStyle.THIN);
        hcs_center.setBorderRight(BorderStyle.THIN);
        hcs_center.setBorderTop(BorderStyle.THIN);
        hcs_center.setFont(boldFont);
        hcs_center.setFillForegroundColor(IndexedColors.GREY_25_PERCENT.index);
        hcs_center.setDataFormat(textFormat);
        // 用于正文居中
        CellStyle ctt_center = workbook.createCellStyle();
        ctt_center.setAlignment(HorizontalAlignment.CENTER);// 文字居中对齐
        ctt_center.setVerticalAlignment(VerticalAlignment.CENTER);// 文字垂直对齐
        ctt_center.setWrapText(false);// 文字是否换行
        ctt_center.setFillForegroundColor(IndexedColors.LIGHT_YELLOW.index);
        ctt_center.setFillPattern(FillPatternType.SOLID_FOREGROUND);
        ctt_center.setBorderBottom(BorderStyle.THIN);
        ctt_center.setBorderLeft(BorderStyle.THIN);
        ctt_center.setBorderRight(BorderStyle.THIN);
        ctt_center.setBorderTop(BorderStyle.THIN);
        ctt_center.setDataFormat(textFormat);
        // 用于正文居右
        CellStyle ctt_right = workbook.createCellStyle();
        ctt_right.setAlignment(HorizontalAlignment.RIGHT);// 文字居右对齐
        ctt_right.setVerticalAlignment(VerticalAlignment.CENTER);// 文字垂直对齐
        ctt_right.setWrapText(false);// 文字是否换行
        ctt_right.setFillForegroundColor(IndexedColors.LIGHT_YELLOW.index);
        ctt_right.setFillPattern(FillPatternType.SOLID_FOREGROUND);
        ctt_right.setBorderBottom(BorderStyle.THIN);
        ctt_right.setBorderLeft(BorderStyle.THIN);
        ctt_right.setBorderRight(BorderStyle.THIN);
        ctt_right.setBorderTop(BorderStyle.THIN);

        /** ***************以下是EXCEL第一行列标题********************* */
        XSSFDrawing drawingPatriarch = sheet.createDrawingPatriarch();
        Row row = sheet.createRow(0);
        Cell cell = null;
        int titleLen = titles.size();
        Map<Integer, CellStyle> alignMap = new HashMap<>(titleLen);
        for (int i = 0; i < titleLen; i++) {
            String titleText = titles.get(i);
            cell = row.createCell(i);
            cell.setCellValue(titleText);
            cell.setCellStyle(hcs_center);
            cell.setCellType(Cell.CELL_TYPE_STRING);
            // 判断是否需要插入批注
            if (null != headAnnotations && headAnnotations.containsKey(titleText)) {
                String annotationText = headAnnotations.get(titleText);
                // 前四个参数是坐标点,后四个参数是编辑和显示批注时的大小.
                XSSFComment comment = drawingPatriarch.createCellComment(new XSSFClientAnchor(0, 0, 0, 0, (short) 3, 3, (short) 4, 4));
                // 输入批注信息
                comment.setString(new XSSFRichTextString(annotationText));
                cell.setCellComment(comment);
            }
            try {
                sheet.setColumnWidth(i, titles.get(i).getBytes("utf-8").length * 2 * 256);
            } catch (UnsupportedEncodingException e) {
                e.printStackTrace();
            }

            //设置居中
            if (CollectionUtils.isEmpty(alignList) || alignList.size() <= i) {
                // 默认居中
                alignMap.put(i, ctt_center);
            } else if (ALIGN_RIGHT.equals(alignList.get(i))) {
                //居右对齐
                alignMap.put(i, ctt_right);
            } else {
                alignMap.put(i, ctt_center);
            }

        }

        /** ***************以下是EXCEL正文数据********************* */
        int contentLen = contents.size();
        String[] rowContent = null;
        String content = "";
        for (int i = 0; i < contentLen; i++) {
            row = sheet.createRow(i + 1);
            rowContent = contents.get(i);
            for (int j = 0; j < titleLen; j++) {
                if (j < rowContent.length) {
                    content = rowContent[j];
                }
                // 将内容按顺序赋给对应的列对象
                cell = row.createCell(j);
                cell.setCellStyle(alignMap.get(j));
                cell.setCellValue(content);
            }
        }
        return workbook;
    }


    /**
     * 创建xlsx Excel,分单据和发票两类
     *
     * @param sheetName            sheet名称
     * @param titles               标题
     * @param contents             内容
     * @param workbook             SXSSFWorkbook对象
     * @param billAndInvoiceBorder 单据和发票的分界线
     */
    public static SXSSFWorkbook createSXSSFWorkbookWithBillInvoice(String sheetName, List<String> titles, List<Integer> alignList, List<String[]> contents, SXSSFWorkbook workbook, Integer billAndInvoiceBorder) {
        /** **********创建工作簿************ */
        workbook = (workbook == null) ? new SXSSFWorkbook() : workbook;

        /** **********创建工作表************ */
        Sheet sheet = workbook.createSheet(sheetName);

        /** **********设置纵横打印（默认为纵打）、打印纸***************** */
        PrintSetup printSetup = sheet.getPrintSetup();
        printSetup.setLandscape(false);//false-默认纵向打印  true-横向打印

        /** ************设置单元格字体************** */
        //通常字体
        Font normalFont = workbook.createFont();
        normalFont.setFontName(HSSFFont.FONT_ARIAL);
        normalFont.setFontHeightInPoints((short) 12);// 设置字体大小
        normalFont.setBold(true);
        //粗体
        Font boldFont = workbook.createFont();
        boldFont.setFontName(HSSFFont.FONT_ARIAL);
        boldFont.setFontHeightInPoints((short) 10);// 设置字体大小
        boldFont.setBold(true);
        boldFont.setColor(HSSFColor.VIOLET.index);

        /** ************设置标题和正文格式************ */
        DataFormat format = workbook.createDataFormat();

        // 用于标题 单据侧
        CellStyle billTitle = workbook.createCellStyle();
        billTitle.setAlignment(HorizontalAlignment.CENTER);// 文字居右对齐
        billTitle.setVerticalAlignment(VerticalAlignment.CENTER);// 文字垂直对齐
        billTitle.setWrapText(false);// 文字是否换行
        billTitle.setFillForegroundColor(IndexedColors.LIGHT_YELLOW.index);
        billTitle.setFillPattern(FillPatternType.SOLID_FOREGROUND);
        billTitle.setBorderBottom(BorderStyle.THIN);
        billTitle.setBorderLeft(BorderStyle.THIN);
        billTitle.setBorderRight(BorderStyle.THIN);
        billTitle.setBorderTop(BorderStyle.THIN);
        billTitle.setDataFormat(format.getFormat("0.00"));

        // 用于标题 发票侧
        CellStyle invoiceTitle = workbook.createCellStyle();
        invoiceTitle.setAlignment(HorizontalAlignment.RIGHT);// 文字居右对齐
        invoiceTitle.setVerticalAlignment(VerticalAlignment.CENTER);// 文字垂直对齐
        invoiceTitle.setWrapText(false);// 文字是否换行
        invoiceTitle.setFillForegroundColor(IndexedColors.LIGHT_GREEN.index);
        invoiceTitle.setFillPattern(FillPatternType.SOLID_FOREGROUND);
        invoiceTitle.setBorderBottom(BorderStyle.THIN);
        invoiceTitle.setBorderLeft(BorderStyle.THIN);
        invoiceTitle.setBorderRight(BorderStyle.THIN);
        invoiceTitle.setBorderTop(BorderStyle.THIN);
        invoiceTitle.setDataFormat(format.getFormat("0.00"));

        // 用于正文居右 单据侧
        CellStyle billStyle = workbook.createCellStyle();
        billStyle.setAlignment(HorizontalAlignment.RIGHT);// 文字居右对齐
        billStyle.setVerticalAlignment(VerticalAlignment.CENTER);// 文字垂直对齐
        billStyle.setWrapText(false);// 文字是否换行
        billStyle.setFillForegroundColor(IndexedColors.LIGHT_YELLOW.index);
        billStyle.setFillPattern(FillPatternType.SOLID_FOREGROUND);
        billStyle.setBorderBottom(BorderStyle.THIN);
        billStyle.setBorderLeft(BorderStyle.THIN);
        billStyle.setBorderRight(BorderStyle.THIN);
        billStyle.setBorderTop(BorderStyle.THIN);
        billStyle.setDataFormat(format.getFormat("0.00"));

        // 用于正文居右 发票侧
        CellStyle invoiceStyle = workbook.createCellStyle();
        invoiceStyle.setAlignment(HorizontalAlignment.RIGHT);// 文字居右对齐
        invoiceStyle.setVerticalAlignment(VerticalAlignment.CENTER);// 文字垂直对齐
        invoiceStyle.setWrapText(false);// 文字是否换行
        invoiceStyle.setFillForegroundColor(IndexedColors.LIGHT_GREEN.index);
        invoiceStyle.setFillPattern(FillPatternType.SOLID_FOREGROUND);
        invoiceStyle.setBorderBottom(BorderStyle.THIN);
        invoiceStyle.setBorderLeft(BorderStyle.THIN);
        invoiceStyle.setBorderRight(BorderStyle.THIN);
        invoiceStyle.setBorderTop(BorderStyle.THIN);
        invoiceStyle.setDataFormat(format.getFormat("0.00"));

        /** ***************以下是EXCEL第一行列标题********************* */
        Row row = sheet.createRow(0);
        Cell cell = null;
        int titleLen = titles.size();
        Map<Integer, CellStyle> alignMap = new HashMap<>(titleLen);
        for (int i = 0; i < titleLen; i++) {
            cell = row.createCell(i);
            cell.setCellValue(titles.get(i));
            cell.setCellType(Cell.CELL_TYPE_STRING);
            try {
                if ("税额".equals(titles.get(i))) {
                    sheet.setColumnWidth(i, titles.get(i).getBytes("utf-8").length * 3 * 256);
                } else {
                    sheet.setColumnWidth(i, titles.get(i).getBytes("utf-8").length * 2 * 256);
                }
            } catch (UnsupportedEncodingException e) {
                e.printStackTrace();
            }

            if (i < billAndInvoiceBorder) {
                cell.setCellStyle(billTitle);
            } else {
                cell.setCellStyle(invoiceTitle);
            }
        }

        /** ***************以下是EXCEL正文数据********************* */
        int contentLen = contents.size();
        String[] rowContent = null;
        String content = "";

        for (int i = 0; i < contentLen; i++) {
            //row
            row = sheet.createRow(i + 1);
            rowContent = contents.get(i);
            for (int j = 0; j < titleLen; j++) {
                if (j < rowContent.length) {
                    content = rowContent[j];
                }
                //将内容按顺序赋给对应的列对象
                cell = row.createCell(j);
                if (j < billAndInvoiceBorder) {
                    cell.setCellStyle(billStyle);
                } else {
                    cell.setCellStyle(invoiceStyle);
                }
                if ("含税金额".equals(titles.get(j)) || "税额".equals(titles.get(j)) || "不含税金额".equals(titles.get(j)) || "已匹配金额".equals(titles.get(j))
                        || "应付金额".equals(titles.get(j)) || "本次占用金额".equals(titles.get(j)) || "剩余占用金额".equals(titles.get(j)) || "本次占用税额".equals(titles.get(j))) {
                    if (StringUtils.isNotEmpty(content)) {
                        cell.setCellValue(Double.parseDouble(content));
                    }
                } else {
                    cell.setCellValue(content);
                }
            }
        }
        return workbook;
    }

}
